"""DataTreePresenter class that will build a datatree model-view."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.tree_model import FolderItem


class DataTreePresenter:
    """DataTreePresenter class that will build a datatree model-view."""

    def __init__(self):
        """Initialize the DataTreePresenter class.
        """
        self.folder_item_list = [FolderItem]

    def build_tree(self, model_dict):
        """Build the tree from the model.

        Args:
            model_dict (dict of models): The model with tree data
        """
        data_tree = []
        # data_tree = FolderItem(name='Data Explorer', item_class='root', tool_tip='Data Explorer')
        # data_tree.set_standard_icon()
        # data_tree.is_read_only = True
        # data_tree.is_visible = False
        # current_item = data_tree

        single_model = len(model_dict) == 1
        for model in model_dict.values():
            model_root = FolderItem(name=model.model_name, class_name='Model', item_class='Model',
                                    parent=data_tree, icon=model.model_icon)

            projects_list = model.get_projects_list()
            for project in projects_list:
                project_root = FolderItem(name=project.name, class_name='Project', item_class='Project',
                                          parent=data_tree, icon=model.project_icon)
                project_root.uuid = project.uuid
                project_root.item_uuid = project.uuid
                project_root.is_checked = project.is_checked
                project_root.is_expanded = project.is_expanded
                project_root.is_editing = project.is_editing
                project_root.is_selected = project.is_selected
                project_root.is_drag_enabled = project.is_drag_enabled
                project_root.is_drop_enabled = project.is_drop_enabled
                project_root.is_read_only = project.is_read_only
                project_root.tool_tip = project.current_file
                if project.current_file is None or project.current_file != '':
                    project_root.tool_tip = 'This project has not been saved yet.'
                # model_root.is_read_only = True
                # data_tree.children.append(model_root)

                # Add current file as child
                if project.current_file is not None:
                    current_file = FolderItem(name=str(project.current_file), class_name='File', item_class='File',
                                              parent=project_root)
                    self._convert_project_data_to_tree_data_recursive(current_file, project_root)

                # Add project settings as child
                self._convert_project_data_to_tree_data_recursive(project.project_settings, project_root,
                                                                  name_is_read_only=True)
                project_root.children[-1].icon = FolderItem.settings_icon

                for child in project.children:
                    _ = self._convert_project_data_to_tree_data_recursive(child, project_root)

                if single_model:
                    # Next line will set model icon to the project root in single model mode
                    # project_root.icon = model.model_icon
                    data_tree.append(project_root)
                else:
                    model_root.children.append(project_root)
                    project_root.parent = model_root

            if not single_model:
                data_tree.append(model_root)
            # self._add_tree_items_recursive(model_tree_root, current_item)

        return data_tree

    # def _add_tree_items_recursive(self, model_item, current_item):
    #     """Recursively build the tree from the model.

    #     Args:
    #         model_item (TreeItem): The model item
    #         current_item (FolderItem): The current item in the tree
    #     """
    #     icon_str = model_item.icon
    #     if isinstance(icon_str, Path):
    #         icon_str = str(icon_str)
    #     item = FolderItem(name=model_item.name, item_class=model_item.item_class, parent=current_item,
    #                       icon=icon_str, is_checked=model_item.is_checked, is_expanded=model_item.is_expanded,
    #                       is_selected=model_item.is_selected, tool_tip=model_item.tool_tip,
    #                       item_uuid=model_item.item_uuid)

    #     current_item.children.append(item)
    #     item.is_drag_enabled = model_item.is_drag_enabled
    #     item.is_drop_enabled = model_item.is_drop_enabled
    #     item.is_read_only = model_item.is_read_only

    #     for child in model_item.children:
    #         self._add_tree_items_recursive(child, item)

    def _convert_project_data_to_tree_data_recursive(self, project_item, parent=None, name_is_read_only=False):
        """Converts project data to tree data recursively.

        Args:
            project_item (Varies): project item
            parent (TreeItem): tree item
            name_is_read_only (bool): True if the name is read only
        """
        tree_item = self.convert_calc_to_tree_item(project_item, parent, name_is_read_only=name_is_read_only)

        if hasattr(project_item, 'children'):
            for child in project_item.children:
                _ = self._convert_project_data_to_tree_data_recursive(child, tree_item)

        return tree_item

    def convert_calc_to_tree_item(self, calc, parent, name_is_read_only=False):
        """Converts a CalcData to a tree item.

        Args:
            calc (CalcData): CalcData to convert

        Returns:
            TreeItem: tree item
        """
        icon = None
        is_checked = False
        is_enabled = True
        item_uuid = calc.uuid
        tool_tip = None
        item_name = calc.name
        # if calc.class_name in self.calcdata_dict:
        #     icon = self.calcdata_dict[calc.class_name].icon
        #     tool_tip = self.calcdata_dict[calc.class_name].tool_tip
        if hasattr(calc, 'class_name'):
            if calc.class_name == 'Project':
                item_name = f'Project: {calc.name}'
        if hasattr(calc, 'icon') and calc.icon is not None:
            icon = calc.icon
        if hasattr(calc, 'tool_tip') and calc.tool_tip is not None:
            tool_tip = calc.tool_tip
        if tool_tip is None:
            tool_tip = calc.name
        tree_item = FolderItem(item_name, icon=icon, is_checked=is_checked, is_enabled=is_enabled, tool_tip=tool_tip,
                               item_uuid=item_uuid)

        if hasattr(calc, 'tree_data'):
            tree_item.is_checked = calc.tree_data.is_checked
            tree_item.is_expanded = calc.tree_data.is_expanded
            tree_item.is_editing = calc.tree_data.is_editing
            tree_item.is_selected = calc.tree_data.is_selected
            if name_is_read_only:
                tree_item.is_read_only = name_is_read_only
        else:
            tree_item.is_checked = calc.is_checked
            tree_item.is_partially_checked = calc.is_partially_checked
            tree_item.is_enabled = calc.is_enabled
            tree_item.is_expanded = calc.is_expanded
            tree_item.is_editing = calc.is_editing
            tree_item.is_selected = calc.is_selected
            tree_item.is_drag_enabled = calc.is_drag_enabled
            tree_item.is_drop_enabled = calc.is_drop_enabled
            tree_item.is_read_only = calc.is_read_only
            if name_is_read_only:
                tree_item.is_read_only = name_is_read_only

        if parent is not None:
            parent.children.append(tree_item)
            tree_item.parent = parent

        return tree_item
