"""Classes to handle the File IO side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import ast
import importlib
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class ReadCalculator:
    """Provides a class that will take calculators and save them to a given file."""

    def __init__(self):
        """Initialize the ExportCalculator Class.
        """
        # self.debug_count = 0

    def read_calculator_from_hdf5(self, calculator, hdf_group, unit_system, app_data, model_name=None,
                                  project_uuid=None, is_setting: bool = False):
        """Read a calculator from the file.

        Args:
            calculator (CalcData): CalcData to have data set to it
            hdf_group: group within an HDF file
            unit_system: the selected Unit System
            app_data: Application data
            model_name: Name of the model
            project_uuid: UUID of the project
            is_setting: Whether this is a setting

        Return:
            result (bool): True if the variable was read
            calculator (CalcData): CalcData to have data set to it
            unread_variables (list): list of unread variables
        """
        if calculator is None or isinstance(calculator, str) or isinstance(calculator, int) or isinstance(calculator,
                                                                                                          float):
            return False, calculator, []
        calculator.name = hdf_group.attrs.get('name', calculator.name)
        calc_uuid = hdf_group.attrs.get('uuid', None)
        if calc_uuid:
            if isinstance(calc_uuid, str):
                calc_uuid = uuid.UUID(calc_uuid)
            calculator.uuid = calc_uuid

        if hasattr(calculator, 'tree_data'):
            calculator.tree_data.is_checked = hdf_group.attrs.get('is_checked', calculator.tree_data.is_checked)
            calculator.tree_data.is_expanded = hdf_group.attrs.get('is_expanded', calculator.tree_data.is_expanded)
            # Do not save selections - We do not want selections to persist
            # calculator.tree_data.is_selected = hdf_group.attrs.get('is_selected', calculator.tree_data.is_selected)
            calculator.tree_data.is_drag_enabled = hdf_group.attrs.get('is_drag_enabled',
                                                                       calculator.tree_data.is_drag_enabled)
            calculator.tree_data.is_drop_enabled = hdf_group.attrs.get('is_drop_enabled',
                                                                       calculator.tree_data.is_drop_enabled)
            calculator.tree_data.is_read_only = hdf_group.attrs.get('is_read_only', calculator.tree_data.is_read_only)
            parent_uuid = hdf_group.attrs.get('parent_uuid', calculator.tree_data.parent_uuid)
            if isinstance(parent_uuid, str) and parent_uuid != 'None':
                parent_uuid = uuid.UUID(parent_uuid)
            calculator.tree_data.parent_uuid = parent_uuid

        unread_variables = []

        input_variables = calculator.input
        results_variables = calculator.results

        input_group = hdf_group['input']
        results_group = hdf_group['results']

        input_dict_group = None
        results_dict_group = None
        if 'dictionaries_group' in input_group:
            input_dict_group = input_group['dictionaries_group']

        if 'dictionaries_group' in results_group:
            results_dict_group = results_group['dictionaries_group']

        for cur_variable in input_variables:
            unread_var = []
            if 'dictionaries_group' == cur_variable or isinstance(input_variables[cur_variable], dict) \
                    and len(input_variables[cur_variable]) <= 0:
                continue
            if input_dict_group and cur_variable in input_dict_group and isinstance(input_variables[cur_variable],
                                                                                    dict):
                _, _, unread_var = self.read_variable_from_hdf5_to_proper_variable(
                    cur_variable, calculator, True, input_dict_group[cur_variable], unit_system, app_data,
                    model_name, project_uuid=project_uuid, is_setting=is_setting)
            elif cur_variable in input_group:
                _, _, unread_var = self.read_variable_from_hdf5_to_proper_variable(
                    cur_variable, calculator, True, input_group[cur_variable], unit_system, app_data,
                    model_name, project_uuid=project_uuid, is_setting=is_setting)
            unread_variables.extend(unread_var)

        for cur_variable in results_variables:
            unread_var = []
            if 'dictionaries_group' == cur_variable or isinstance(results_variables[cur_variable], dict) \
                    and len(results_variables[cur_variable]) <= 0:
                continue
            if results_dict_group and cur_variable in results_dict_group and isinstance(results_variables[cur_variable],
                                                                                        dict):
                _, _, unread_var = self.read_variable_from_hdf5_to_proper_variable(
                    cur_variable, calculator, False, results_dict_group[cur_variable], unit_system, app_data,
                    model_name, project_uuid=project_uuid, is_setting=is_setting)
            elif cur_variable in results_group:
                _, _, unread_var = self.read_variable_from_hdf5_to_proper_variable(
                    cur_variable, calculator, False, results_group[cur_variable], unit_system, app_data,
                    model_name, project_uuid=project_uuid, is_setting=is_setting)
            unread_variables.extend(unread_var)

        # Code to read a variable (so we can determine the selected profile)
        if calculator.type == 'var_group' and len(input_variables) <= 0:
            for key in input_group.keys():
                if key in input_group[key]:
                    value = input_group[key][key][()]
                    if isinstance(value, bytes):
                        value = value.decode('utf-8')
                    type = input_group[key][key].attrs.get('type')
                    if type is not None:
                        calculator.input[key] = Variable(key, type)
                        if type == 'list':
                            if isinstance(value, str):
                                value = ast.literal_eval(value)
                            list_length = len(value)
                            for i in range(list_length):
                                new_val = value[i]
                                if isinstance(new_val, bytes):
                                    new_val = new_val.decode('utf-8')
                                if new_val not in calculator.input[key].value_options:
                                    calculator.input[key].value_options.append(new_val)
                                    calculator.input[key].value += 1
                        var_uuid = input_group[key][key].attrs.get('uuid')
                        if var_uuid is not None:
                            calculator.input[key].uuid = var_uuid
                        index = input_group[key][key].attrs.get('index')
                        if index is not None:
                            calculator.input[key].set_val(index)

        return True, calculator, unread_variables

    def read_variable_from_hdf5_to_proper_variable(self, name, calculator, is_input, data_folder, unit_system,
                                                   app_data, model_name=None, project_uuid=None,
                                                   is_setting: bool = False):
        """Read a calculator from the file.

        Args:
            name (string): name of the variable
            calculator (CalcData): CalcData to have data set to it
            is_input: whether the variables are input or result tables
            data_folder: group within an HDF file
            unit_system: the selected Unit System
            app_data: Application data
            model_name: Name of the model
            project_uuid: UUID of the project
            is_setting: Whether this is a setting

        Return:
            result (bool): True if the variable was read
            calculator (CalcData): CalcData to have data set to it
            unread_variables (list): list of unread variables
        """
        unread_variables = []
        var_group = calculator.input
        if not is_input:
            var_group = calculator.results

        if name in var_group:
            if isinstance(var_group[name], dict):
                for key in var_group[name]:
                    if key in data_folder:
                        result, unread_var = self.read_variable_from_hdf5(
                            key, var_group[name], data_folder[key], unit_system, app_data, model_name, project_uuid,
                            is_setting)
                        unread_variables.extend(unread_var)
                        if not result:
                            unread_variables.append(key)
                return True, calculator, unread_variables
            result, var_unread = self.read_variable_from_hdf5(
                name, var_group, data_folder, unit_system, app_data, model_name, project_uuid, is_setting)
            unread_variables.extend(var_unread)
            if not result:
                unread_variables.append(name)

        return True, calculator, unread_variables

    def read_variable_from_hdf5(self, name, calculator_var_list, data_folder, unit_system, app_data,
                                model_name=None, project_uuid=None, is_setting: bool = False):
        """Write a variable to the file.

        Args:
            name (string): name of the variable
            calculator_var_list (dict): List of the variables for a calculator
            is_input: whether the variables are input or result tables
            data_folder: group within an HDF file
            unit_system: the selected Unit System
            app_data: Application data
            model_name: Name of the model
            project_uuid: UUID of the project
            is_setting: Whether this is a setting

        Return:
            result (bool): True if the variable was read
            unread_variables (list): list of unread variables
        """
        variable = Variable()
        if name in calculator_var_list:
            variable = calculator_var_list[name]
        if name not in data_folder:
            return False, []
        dataset = data_folder[name]
        type = dataset.attrs.get('type')
        unread_variables = []
        result = False

        if type in ['calc', 'class', 'UserArray', 'table', 'PersonalDetails']:
            # print(self.debug_count, name, type, variable.get_val())
            # self.debug_count += 1
            for calc in list(dataset.keys()):  # Open the calculator folder
                # TODO Fix gradation settings; it has a variable that is set to None so it can init without circular ref
                # But that causes problems here!
                if variable is not None:
                    result, _, unread_var = self.read_calculator_from_hdf5(
                        variable.get_val(), dataset[calc], unit_system, app_data, model_name, project_uuid=project_uuid,
                        is_setting=is_setting)
                    unread_variables.extend(unread_var)
                else:
                    unread_variables.append(name)
            calculator_var_list[name] = variable
            return result, unread_variables

        if type in ['calc list']:
            input_group = dataset['input']
            for var in variable.value.input:
                result, unread_var = self.read_variable_from_hdf5(
                    var, variable.value.input, input_group[var], unit_system, app_data, model_name,
                    project_uuid=project_uuid, is_setting=is_setting)
                unread_variables.extend(unread_var)

            num_items = variable.value.input['Number of items'].get_val()

            variable.value.item_list = []
            item_group = dataset['item_list']
            for calc_name in list(item_group.keys()):
                group = item_group[calc_name]
                variable.value.add_item_to_list()
                result, _, unread_var = self.read_calculator_from_hdf5(
                    variable.value.item_list[-1], group, unit_system, app_data, model_name, project_uuid=project_uuid,
                    is_setting=is_setting)
                unread_variables.extend(unread_var)

            variable.value.input['Number of items'].set_val(num_items)
            return result, unread_variables

        # print(self.debug_count, name, variable.type, variable.get_val())
        # self.debug_count += 1

        value = dataset[()]
        if isinstance(value, bytes):
            value = value.decode('utf-8')

        if type in ['float']:
            if isinstance(value, np.floating):
                variable.value = float(value.item())
            else:
                variable.value = float(value)
        elif type == 'int':
            if isinstance(value, int):
                variable.value = value
            else:
                variable.value = int(value.item())
        elif type in variable.string_types or type == 'file':
            if isinstance(value, str):
                variable.value = value
            elif hasattr(value, 'item'):
                variable.value = str(value.item())
            else:
                variable.value = str(value.decode())

        elif type == 'list':
            if isinstance(value, str):
                value = ast.literal_eval(value)
            list_length = len(value)
            for i in range(list_length):
                new_val = value[i]
                if isinstance(new_val, bytes):
                    new_val = new_val.decode('utf-8')
                if new_val not in variable.value_options:
                    variable.value_options.append(new_val)
                    variable.value += 1

            selected = dataset.attrs.get('index')
            variable.set_val(selected)

        elif type == 'bool':
            if isinstance(value, str):
                if value.lower() == 'true':
                    variable.value = True
                else:
                    variable.value = False
            else:
                variable.value = bool(value)

        elif type in ['float_list', 'int_list', 'string_list']:
            if isinstance(value, str):
                value = ast.literal_eval(value)
            list_length = len(value)
            new_list = []
            for i in range(list_length):
                new_val = value[i]
                if isinstance(new_val, bytes):
                    new_val = new_val.decode('utf-8')
                else:
                    new_val = new_val

                if type == 'float_list':
                    new_val = float(new_val)
                elif type == 'int_list':
                    new_val = int(new_val)
                elif type == 'string_list':
                    new_val = str(new_val)
                new_list.append(new_val)
            variable.value_options = new_list
            variable.value = len(new_list)

        elif type == 'image':
            if isinstance(value, np.ndarray):
                variable.value = value

        elif type == 'uuid_dict':
            if value == 'None':
                variable.value = None
            elif isinstance(value, str):
                if value == 'None':
                    variable.value = None
                elif value == '-1':
                    variable.value = -1
                else:
                    variable.value = uuid.UUID(value)

        # TODO: Read value (for format type), then read color values)
        elif type == 'color':
            if isinstance(value, str):
                value = ast.literal_eval(value)
            list_length = len(value)
            new_list = []
            for i in range(list_length):
                new_val = value[i]
                if isinstance(new_val, bytes):
                    new_val = new_val.decode('utf-8')
                new_list.append(int(new_val))
            if len(new_list) == 3:
                variable.value_options = (new_list[0], new_list[1], new_list[2])
            variable.value = 'rgb'
        elif type == 'date':
            # Assuming value is a datetime object
            if hasattr(value, 'isoformat'):
                variable.value = value.isoformat()
            else:
                variable.value = str(value)
        else:
            msg = f'Unsupported data type for object with type: "{type}"'
            print(ValueError(msg))
            return False, unread_variables

        # Units
        if 'units_list' in data_folder:
            units_dataset = data_folder['units_list']
            variable.selected_us_unit = units_dataset.attrs.get('selected_us_unit')
            variable.selected_si_unit = units_dataset.attrs.get('selected_si_unit')

        calculator_var_list[name] = variable

        return True, unread_variables

    def create_class_for_class_name(self, class_input, app_data, model_name=None, project_uuid=None,
                                    is_setting: bool = False):
        """Given a class string, create an instance of the class.

        Args:
            class_input (str): Class string specifying type of class
            app_data: Application data
            model_name: Name of the model
            project_uuid: UUID of the project
            is_setting: Whether this is a setting
        """
        class_string = class_input[8:-2]
        module_name, class_name = class_string.rsplit('.', 1)
        module = importlib.import_module(module_name)
        class_ = getattr(module, class_name)
        try:
            if not is_setting:
                calc = class_(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
            else:
                calc = class_(app_name=app_data.app_name, version=app_data.version, agency=app_data.agency,
                              developed_by=app_data.developed_by)
        except Exception as e:
            print(e)
            calc = class_()

        if hasattr(calc, 'class_name') and calc.class_name is not None and calc.class_name != '':
            if calc.class_name in app_data.model_dict[model_name].calcdata_dict:
                calc_dict_item = app_data.model_dict[model_name].calcdata_dict[calc.class_name]
                calc.icon = calc_dict_item.icon

        return calc
