"""Classes to handle the File IO side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path  # Import pathlib
import uuid

# 2. Third party modules
import h5py
import numpy as np

# 3. Aquaveo modules
from xms.constraint._xmsconstraint.constraint import coReadGridFromFile
from xms.constraint.grid import GridType
from xms.constraint.quadtree_grid_2d import QuadtreeGrid2d
from xms.constraint.quadtree_grid_3d import QuadtreeGrid3d
from xms.constraint.rectilinear_grid_2d import RectilinearGrid2d
from xms.constraint.rectilinear_grid_3d import RectilinearGrid3d
from xms.constraint.ugrid_2d import UGrid2d
from xms.constraint.ugrid_3d import UGrid3d
from xms.constraint.unconstrained_grid import UnconstrainedGrid
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.interface_adapters.view_model.main.tree_model import FolderItem
from xms.FhwaVariable.interface_structures.read_manager import ReadManagerBase

# 4. Local modules
from xms.FhwaVariableFileIO.read.read_calculator import ReadCalculator


class ReadManager(ReadManagerBase):
    """Provides a class that will take calculators and save them to a given file."""

    def __init__(self, app_name, app_version):
        """Initialize the ExportCalculator Class.

        Args:
            app_name (string): name of the application (checks that we are reading the correct file)
            app_version (string): current version of the software being used
        """
        super().__init__(app_name, app_version)

        self.read_calculator = ReadCalculator()

        self.ugrid_filters = ["U-Grid (*.xmc)"]

        self.folder_class_names = ['Project', 'Folder', 'Proposed', 'Existing', 'Geometry Folder', 'Coverage Folder',
                                   'GIS Folder', 'Model Folder']

    # def read(self, filename, app_data=None, model_name=None, is_setting: bool = False):
    #     """Import the calculators to a given filename.

    #     Args:
    #         filename (string): filename to read
    #         app_data (AppData): the application data
    #         model_name (string): the name of the model
    #         is_setting (bool): True if the item is a setting

    #     Return:
    #         calculators (list): the list of calculators to save to a file
    #     """

    #     return self.read_from_hdf5_file(filename, app_data, model_name, is_setting)

    def read(self, filename, app_data=None, model_name=None, is_setting: bool = False):
        """Import the calculators or other supported file types to a given filename.

        Args:
            filename (string): filename to read
            app_data (AppData): the application data
            model_name (string): the name of the model
            is_setting (bool): True if the item is a setting

        Return:
            result (bool): True if the file was successfully read
            geometry_list (list): list of geometry objects read from the file
            cover_list (list): list of cover objects read from the file
            gis_list (list): list of GIS objects read from the file
            calcs (list): the calculator data read from the file
            file_warning (string): any warnings related to the file
            unread_variables (list): list of unread variables
        """
        # Use pathlib to get the file extension
        file_extension = Path(filename).suffix

        result = False
        data = None
        root = None
        file_warning = ''
        unread_variables = []

        # Dispatch based on file type
        if file_extension in [ext.split('*')[-1].strip(')') for ext in self.ugrid_filters]:
            ugrid = self.read_ugrid_file(str(filename))
            if ugrid:
                result = True
                data = {}
                data['ugrid'] = ugrid
                data['ugrid filename'] = filename
        else:
            result, root, file_warning, unread_variables = self.read_from_hdf5_file(
                filename, app_data, model_name, is_setting)

        return result, data, root, file_warning, unread_variables

    def read_ugrid_file(self, file_path):
        """Read a UGrid file and return the corresponding grid object.

        Args:
            file_path (str): Path to the UGrid file.

        Returns:
            A constrained Grid.
        """
        co_grid_types = coReadGridFromFile(file_path)
        if co_grid_types is None:
            return None
        if co_grid_types.GetGridType() == GridType.quadtree_2d:
            return QuadtreeGrid2d(co_grid_types)
        if co_grid_types.GetGridType() == GridType.quadtree_3d:
            return QuadtreeGrid3d(co_grid_types)
        if co_grid_types.GetGridType() == GridType.rectilinear_2d:
            return RectilinearGrid2d(co_grid_types)
        if co_grid_types.GetGridType() == GridType.rectilinear_3d:
            return RectilinearGrid3d(co_grid_types)
        if co_grid_types.GetGridType() == GridType.ugrid_2d:
            return UGrid2d(instance=co_grid_types)
        if co_grid_types.GetGridType() == GridType.ugrid_3d:
            return UGrid3d(instance=co_grid_types)
        if co_grid_types.GetGridType() == GridType.unconstrained:
            return UnconstrainedGrid(instance=co_grid_types)
        raise ValueError('Unable to read CoGrid.')

    def read_from_hdf5_file(self, filename, app_data=None, model_name=None, is_setting: bool = False):
        """Import the calculators to a given filename.

        Args:
            filename (string): filename to read
            app_data (AppData): the application data
            model_name (string): the name of the model
            is_setting (bool): True if the item is a setting

        Return:
            calculators (list): the list of calculators to save to a file
        """
        # Check if the file exists and is locked
        if not os.path.exists(filename):
            print(f'file: {filename} does not exist')
            return False

        if app_data is None:
            app_data = AppData()

        _, unit_system = app_data.app_settings.get_setting('Selected unit system')

        calculators = []
        unread_variables = []
        file_warning = ''

        # TODO: Create logic that checks UUID when read, and if it exists, perhaps create a new UUID

        try:
            with h5py.File(filename, 'r') as file:
                # Check that we are working with a file that was created by the toolbox
                if file.attrs.get('app_name', '') == self.app_name:
                    self.file_version_str = file.attrs.get('app_version', '0.0.0')
                    # self.file_version = float('.'.join(self.file_version_str.split('.')[:2]))
                    result = self.compare_file_version(self.file_version_str, ignore_patch=True)

                    if result == 'File is newer':
                        file_warning = (
                            f'The file was written by a newer version ({self.file_version_str}) '
                            f'than the software you are using ({self.app_version_str}). It is recommended '
                            'that you update your software to be sure that you are not losing data.'
                        )
                    elif result == 'File is older':
                        file_warning = (
                            f'The file was written by an older version ({self.file_version_str}) '
                            f'than the software you are using ({self.app_version_str}). It is recommended '
                            'that you verify that all data was read and converted correctly.'
                        )

                    result, data, unread_var = self._read_item_from_hdf5_file_recursive(
                        'root_items', file, unit_system, app_data, model_name, is_setting=is_setting)
                    unread_variables.extend(unread_var)
                    if result:
                        calculators = data

                    # if 'calculator_order' in file:
                    #     calculator_order_dataset = file['calculator_order']
                    #     calculator_order = calculator_order_dataset[()].tolist()

                    #     for group_name in calculator_order:
                    #         group = file[group_name]
                    #         type_name = group.attrs.get('type')
                    #         class_name = group.attrs.get('class')

                    #         calc = self.read_calculator.create_class_for_class_name(class_name)
                    #         if calc:
                    #             result, data, unread_var = self.read_calculator.read_calculator_from_hdf5(
                    #                 calc, group, unit_system)
                    #             unread_variables.extend(unread_var)
                    #             if result:
                    #                 calculators.append(data)
                    #         else:
                    #             unread_variables.append(type_name)
                    # else:
                    #     return False, '', file_warning, unread_variables
                else:
                    file_warning = f'The file was not created by the {self.app_name}.'
                    return False, '', file_warning, unread_variables

            return True, calculators, file_warning, unread_variables

        except Exception as e:
            print(f"Error occurred while reading the file: {e}")
            file_warning = f"Error occurred while reading the file: {e}"
            return False, [], file_warning, unread_variables

    def _read_item_from_hdf5_file_recursive(self, item_name, hdf_file, unit_system, app_data, model_name,
                                            project_uuid=None, is_setting: bool = False):
        """Exports the item and children to the file.

        Args:
            item_name (?): the name of the dataset we will read
            hdf_file: group within an HDF file
            unit_system: the selected Unit System
            app_data (AppData): the application data
            model_name (string): the name of the model
            project_uuid (string): the project UUID
            is_setting (bool): True if the item is a setting

        Return:
            result (bool): True if the variable was read
            calculator (CalcData): CalcData to have data set to it
            unread_variables (list): list of unread variables
        """
        if item_name not in hdf_file:
            return

        dataset = hdf_file[item_name]

        dataset_type = dataset.attrs.get('datatype', None)

        if dataset_type is None:
            dataset_type = dataset[()].tolist()
        if isinstance(dataset_type, bytes):
            dataset_type = dataset_type.decode('utf-8')

        if dataset_type == 'dict':
            dict_names = hdf_file['dict'][()].tolist()
            dict_names = [item.decode('utf-8') if isinstance(item, bytes) else item for item in dict_names]
            item_dict = {}
            for name in dict_names:
                result, data, unread_var = self._read_item_from_hdf5_file_recursive(
                    name, hdf_file, unit_system, app_data, model_name, project_uuid, is_setting)
                if result:
                    item_dict[name] = data
            return True, item_dict, unread_var
        elif dataset_type == 'list':
            list_names = hdf_file['list'][()].tolist()
            list_names = [item.decode('utf-8') if isinstance(item, bytes) else item for item in list_names]
            item_list = []
            for name in list_names:
                result, data, unread_var = self._read_item_from_hdf5_file_recursive(
                    name, hdf_file, unit_system, app_data, model_name, project_uuid, is_setting)
                if result:
                    item_list.append(data)
            return True, item_list, unread_var
        elif dataset_type in self.folder_class_names:
            unread_var = []
            new_folder = self.read_folder_from_hdf5_file(hdf_file[item_name])
            new_folder.uuid = new_folder.item_uuid  # On project, these are the same
            if dataset_type == 'Project':
                project_uuid = new_folder.uuid
                # Get project_uuid, and project settings
                project_settings_folder = hdf_file[item_name].attrs.get('project_settings', None)
                if project_settings_folder is not None and project_settings_folder in hdf_file:
                    result, data, unread_var = self._read_item_from_hdf5_file_recursive(
                        project_settings_folder, hdf_file, unit_system, app_data, model_name, project_uuid,
                        is_setting=True)
                    if result:
                        new_folder.project_settings = data
                        # We need to add the project settings so we can properly initialize data as it is read
                        app_data.model_dict[model_name].add_or_update_project(new_folder)
                # Need enough update to put the settings in the app_data
            if 'children' in hdf_file[item_name]:
                children = hdf_file[item_name]['children'][()].tolist()
                children = [item.decode('utf-8') if isinstance(item, bytes) else item for item in children]
                for child_name in children:
                    result, data, unread_var = self._read_item_from_hdf5_file_recursive(
                        child_name, hdf_file[item_name], unit_system, app_data, model_name, project_uuid, is_setting)
                    if result:
                        new_folder.children.append(data)
                        data.parent = new_folder.uuid
            return True, new_folder, unread_var
        elif dataset_type == 'calculator':
            unread_variables = []
            group = hdf_file[item_name]
            type_name = group.attrs.get('type')
            class_name = group.attrs.get('class')

            calc = self.read_calculator.create_class_for_class_name(class_name, app_data, model_name,
                                                                    project_uuid, is_setting)
            if calc:
                result, data, unread_var = self.read_calculator.read_calculator_from_hdf5(
                    calc, group, unit_system, app_data, model_name, project_uuid, is_setting)
                unread_variables.extend(unread_var)
            else:
                unread_variables.append(type_name)

            return True, data, unread_variables
        elif dataset_type in hdf_file:
            result, data, unread_var = self._read_item_from_hdf5_file_recursive(
                dataset_type, hdf_file, unit_system, app_data, model_name, project_uuid, is_setting)
            return True, data, unread_var

        # if isinstance(item, dict):
        #     group_name = 'dict'
        #     dict_keys = []
        #     for key in item:
        #         dict_keys.append(key)
        #         dict_key_group = hdf_file.create_group(key)
        #         self._write_item_to_hdf5_file_recursive(item[key], dict_key_group, unit_system)
        #     string_array = np.array(dict_keys, dtype='S')
        #     hdf_file.create_dataset(group_name, data=string_array)
        # elif isinstance(item, list):
        #     group_name = 'list'
        #     list_names = []
        #     for list_item in item:
        #         list_names.append(str(list_item.uuid))
        #         list_item_group = hdf_file.create_group(list_names[-1])
        #         self._write_item_to_hdf5_file_recursive(list_item, list_item_group, unit_system)
        #     string_array = np.array(list_names, dtype='S')
        #     hdf_file.create_dataset(group_name, data=string_array)
        # # Write a folder or calculator
        # elif hasattr(item, 'class_name') and item.class_name in self.folder_class_names:
        #     folder_children = []
        #     group_name, folder_group = self.write_folder_to_hdf5_file(item, hdf_file)
        #     for child in item.children:
        #         child_name, _ = self._write_item_to_hdf5_file_recursive(child, folder_group, unit_system)
        #         folder_children.append(child_name)
        #     string_array = np.array(folder_children, dtype='S')
        #     folder_group.create_dataset('children', data=string_array)
        # else:
        #     group_name = self.write_calculator.write_calculator_to_hdf5(item, hdf_file, unit_system)

        return False, None, [item_name]

    def read_folder_from_hdf5_file(self, folder_group):
        """Exports the folder and children to the file.

        Args:
            folder_group: group within an HDF file
        """
        new_folder = FolderItem('New folder')
        new_folder.name = folder_group.attrs.get('name', 'New folder')
        new_folder.class_name = folder_group.attrs.get('class_name', 'Folder')
        new_folder.is_checked = folder_group.attrs.get('is_checked', False)
        new_folder.is_partially_checked = folder_group.attrs.get('is_partially_checked', 'Folder')
        new_folder.is_enabled = folder_group.attrs.get('is_enabled', True)
        new_folder.is_expanded = folder_group.attrs.get('is_expanded', True)
        # Do not save selection; We do not want selections to persist
        # folder_group.attrs['is_selected'] = folder.is_selected
        new_folder.tool_tip = folder_group.attrs.get('tool_tip', None)
        uuid_str = folder_group.attrs.get('item_uuid', None)
        if uuid_str:
            new_folder.item_uuid = uuid.UUID(uuid_str)
        uuid_str = folder_group.attrs.get('parent_uuid', None)
        if uuid_str:
            new_folder.item_uuid = uuid.UUID(uuid_str)

        new_folder.set_standard_icon()

        return new_folder

    @staticmethod
    def clean_hdf5_dict(obj):
        """Cleans up an HDF5 dictionary read from file."""
        if isinstance(obj, dict):
            return {ReadManager.clean_hdf5_dict(k): ReadManager.clean_hdf5_dict(v) for k, v in obj.items()}
        elif isinstance(obj, (list, tuple)):
            # Remove empty strings from lists/tuples
            cleaned = [ReadManager.clean_hdf5_dict(x) for x in obj if x != '' and x is not None]
            return type(obj)(cleaned)
        elif isinstance(obj, np.ndarray):
            if obj.dtype.kind == 'S':  # byte strings
                return [x.decode('utf-8').replace('#d', '°') if isinstance(x, bytes) else str(x).replace('#d', '°')
                        for x in obj.tolist() if x != b'' and x != '']
            else:
                # Round floating point arrays to remove precision artifacts
                return [x for x in obj.tolist() if x != '' and x is not None]
        elif isinstance(obj, bytes):
            val = obj.decode('utf-8').replace('#d', '°')
            return val if val != '' else None
        elif isinstance(obj, str):
            # HY-8 Shape Database uses the '#d' placeholder for degree symbol
            val = obj.replace('#d', '°')
            return val if val != '' else None
        elif isinstance(obj, float):
            # Round floats to remove precision artifacts
            return round(obj, 6)
        else:
            return obj if obj != '' else None

    @staticmethod
    def hdf5_to_dict(h5file):
        """Convert an HDF5 file to a nested dictionary."""
        def recursively_load(h5obj):
            result = {}
            for key, item in h5obj.items():
                if isinstance(item, h5py.Group):
                    result[key] = recursively_load(item)
                elif isinstance(item, h5py.Dataset):
                    result[key] = item[()]  # Convert dataset to numpy array or value
            return result

        with h5py.File(h5file, 'r') as f:
            file_dict = recursively_load(f)
        return ReadManager.clean_hdf5_dict(file_dict)
