"""Classes to handle the File IO side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
# import cv2
import h5py
import numpy as np

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class WriteCalculator:
    """Provides a class that will take calculators and save them to a given file."""

    def __init__(self):
        """Initialize the ExportCalculator Class.
        """
        # self.debug_count = 0

    def write_calculator_to_hdf5(self, calculator, hdf_file, unit_system):
        """Write a calculator to the file.

        Args:
            calculator (CalcData): CalcData to have data set to it
            hdf_file: group within an HDF file
            unit_system: the selected Unit System

        Return:
            calc_group_name (string): name of the group within the HDF file
        """
        # Create group based on name and UUID
        calc_group_name = f'{calculator.name}_{str(calculator.uuid)}'
        calc_group = hdf_file.create_group(calc_group_name)

        calc_group.attrs['datatype'] = 'calculator'

        calc_group.attrs['name'] = calculator.name
        calc_group.attrs['type'] = calculator.type
        calc_group.attrs['uuid'] = str(calculator.uuid)
        calc_group.attrs['class'] = str(type(calculator))
        if hasattr(calculator, 'tree_data'):
            calc_group.attrs['is_checked'] = calculator.tree_data.is_checked
            calc_group.attrs['is_expanded'] = calculator.tree_data.is_expanded
            # Do not save selections - We do not want selections to persist
            # calc_group.attrs['is_selected'] = calculator.tree_data.is_selected
            calc_group.attrs['is_drag_enabled'] = calculator.tree_data.is_drag_enabled
            calc_group.attrs['is_drop_enabled'] = calculator.tree_data.is_drop_enabled
            calc_group.attrs['is_read_only'] = calculator.tree_data.is_read_only
            calc_group.attrs['parent_uuid'] = str(calculator.tree_data.parent_uuid)

        input_group = calc_group.create_group('input')
        results_group = calc_group.create_group('results')

        if 'get_can_compute' in dir(calculator) and 'compute_data' in dir(calculator) and calculator.get_can_compute():
            try:
                calculator.compute_data()
            except Exception as e:
                print(f'Error computing data for {calculator.name}: {e}')

        for variable in calculator.input:
            self.write_variable_to_hdf5(variable, calculator.input[variable], input_group, unit_system)

        for variable in calculator.results:
            self.write_variable_to_hdf5(variable, calculator.results[variable], results_group, unit_system)

        return calc_group_name

    def write_variable_to_hdf5(self, name, variable, calc_group, unit_system):
        """Write a calculator to the file.

        Args:
            name (string): name of the variable
            variable (Variable): Variable to write to the file
            calc_group: group within an HDF for the calculator
            unit_system: the selected Unit System
        """
        if isinstance(variable, dict):
            if 'dictionaries_group' not in calc_group:
                dict_group = calc_group.create_group('dictionaries_group')
            else:
                dict_group = calc_group['dictionaries_group']
            name_group = dict_group.create_group(name)
            for key in variable:
                var = variable[key]
                self.write_variable_to_hdf5(key, var, name_group, unit_system)
            return

        if variable is None:
            return

        # print(self.debug_count, name, variable.type, variable.get_val())
        # self.debug_count += 1
        value = variable.get_val()
        var_group = calc_group.create_group(name)

        if variable.type in ['calc', 'class', 'UserArray', 'table', 'PersonalDetails']:
            dataset = var_group.create_group(name)
            self.write_calculator_to_hdf5(value, dataset, unit_system)

        elif variable.type == 'calc list':
            dataset = var_group.create_group(name)
            input_group = dataset.create_group('input')
            for key in variable.value.input:
                var = variable.value.input[key]
                self.write_variable_to_hdf5(key, var, input_group, unit_system)
            item_group = dataset.create_group('item_list')
            num_items = variable.value.input['Number of items'].get_val()
            for index in range(num_items):
                self.write_calculator_to_hdf5(variable.value.item_list[index], item_group, unit_system)

        elif variable.type == 'float':
            dataset = var_group.create_dataset(name, data=value)
        elif variable.type == 'int':
            dataset = var_group.create_dataset(name, data=value)
        elif variable.type in Variable.string_types or variable.type == 'file':
            dataset = var_group.create_dataset(name, data=str(value))

        elif variable.type == 'list':
            # Put the list to a numpy array and then write it and the selection to the file
            string_array = np.array(variable.value_options, h5py.string_dtype(encoding='utf-8', length=None))
            dataset = var_group.create_dataset(name, data=string_array)
            dataset.attrs['index'] = value

        elif variable.type == 'bool':
            dataset = var_group.create_dataset(name, data=value)

        elif variable.type == 'float_list':
            # Assuming value_options is a list and value is an integer indicating the number of items to take
            # from value_options
            list_length = int(variable.value)
            float_array = np.array(variable.value_options[:list_length])
            dataset = var_group.create_dataset(name, data=float_array)

        elif variable.type == 'string_list':
            string_array = np.array(value, h5py.string_dtype(encoding='utf-8', length=None))
            dataset = var_group.create_dataset(name, data=string_array)

        elif variable.type == 'uuid_dict':
            if value is None:
                value = 'None'
            string_array = np.array(str(value), h5py.string_dtype(encoding='utf-8', length=None))
            dataset = var_group.create_dataset(name, data=string_array)

        # TODO - Add support for other types
        elif variable.type == 'color':
            # dataset = var_group.create_dataset(name, data=value.name())
            float_array = np.array(variable.value_options)
            dataset = var_group.create_dataset(name, data=float_array)

        elif variable.type == 'image':
            if isinstance(value, np.ndarray):
                dataset = var_group.create_dataset(name, data=value)
            else:
                return  # Improper data

        elif variable.type == 'date':
            # Assuming value is a datetime object
            if hasattr(value, 'isoformat'):
                string_array = np.array(value.isoformat(), h5py.string_dtype(encoding='utf-8', length=None))
                dataset = var_group.create_dataset(name, data=string_array)
            else:
                return

        else:
            msg = f'Unsupported data type for object with name: "{variable.name}" ' \
                f'with type: "{variable.type}" and python type: {type(value)}'
            print(ValueError(msg))
            return

        dataset.attrs['type'] = variable.type

        # Units
        units_list, index = variable.get_units_list_and_index(unit_system)
        if len(units_list) and index is not None and index >= 0:
            string_array = np.array(units_list, h5py.string_dtype(encoding='utf-8', length=None))
            dataset = var_group.create_dataset('units_list', data=string_array)
            dataset.attrs['selected_us_unit'] = variable.selected_us_unit
            dataset.attrs['selected_si_unit'] = variable.selected_si_unit

        if hasattr(variable, 'uuid'):
            str_uuid = variable.uuid
            if not isinstance(str_uuid, str):
                str_uuid = str(variable.uuid)
            dataset.attrs['uuid'] = str_uuid

        return
