"""Classes to handle the File IO side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import os

# 2. Third party modules
import h5py
import numpy as np

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.interface_structures.write_manager import WriteManagerBase

# 4. Local modules
from xms.FhwaVariableFileIO.write.write_calculator import WriteCalculator


class WriteManager(WriteManagerBase):
    """Provides a class that will take calculators and save them to a given file."""

    def __init__(self, app_name, app_version):
        """Initialize the WriteCalculator Class.
        """
        super().__init__(app_name, app_version)

        self.folder_class_names = ['Project', 'Folder', 'Proposed', 'Existing', 'Geometry Folder', 'Coverage Folder',
                                   'GIS Folder', 'Model Folder']

        self.write_calculator = WriteCalculator()

    def write(self, root_items, filename, overwrite):
        """Exports the calculators to a given filename.

        Args:
            root_items (list or tree): the list or tree of items to save to a file
            filename (string): filename to write
            overwrite (bool): Overwrite the file if it exists?
        """
        self.write_to_hdf5_file(root_items, filename, overwrite)

    def write_to_hdf5_file(self, root_items, filename, overwrite):
        """Exports the calculators to a given filename.

        Args:
            root_items (list or tree): the list or tree of items to save to a file
            filename (string): filename to write
            overwrite (bool): Overwrite the file if it exists?
        """
        # Check if the file exists and is locked
        if not self.check_for_file_lock(filename, overwrite):
            return False

        # Check if the base folder of the filename exists and create it if not
        base_folder = os.path.dirname(filename)
        if not os.path.exists(base_folder):
            os.makedirs(base_folder)

        current_datetime = datetime.now()

        _, unit_system = AppData.app_settings.get_setting('Selected unit system')

        with h5py.File(filename, 'w') as file:
            file.clear()

            # Created for the FHWA Hydraulic Resource Group, Created by Eric J. Jones at Aquaveo
            file.attrs['created for the FHWA Hydraulic Resource Center'] = 'Created by Eric J. Jones at Aquaveo'

            file.attrs['app_name'] = self.app_name
            file.attrs['app_version'] = self.app_version
            file.attrs['file creation date'] = current_datetime.strftime('%Y-%m-%d %H:%M:%S')

            # root_items = []

            # for calculator in root_items:
            #     root_items.append(self.write_calculator.write_calculator_to_hdf5(calculator, file, unit_system))

            # string_array = np.array(calculator_order, dtype='S')
            # file.create_dataset('calculator_order', data=string_array)

            group_name = self._write_item_to_hdf5_file_recursive(root_items, file, unit_system)

            file.create_dataset('root_items', data=np.array(group_name, dtype='S'))

    def _write_item_to_hdf5_file_recursive(self, item, hdf_file, unit_system):
        """Exports the item and children to the file.

        Args:
            item (Fo?lder): the item to save to a file
            hdf_file: group within an HDF file
            unit_system: the selected Unit System
        """
        if isinstance(item, dict):  # Write a dictionary
            group_name = 'dict'
            dict_keys = []
            dict_names = []
            for key in item:
                dict_keys.append(key)
                dict_key_group = hdf_file.create_group(key)
                subgroup_name = self._write_item_to_hdf5_file_recursive(item[key], dict_key_group, unit_system)
                dict_names.append(subgroup_name)
            string_array = np.array(dict_keys, dtype='S')
            hdf_file.create_dataset('dict_keys', data=string_array)
            string_array = np.array(dict_names, dtype='S')
            hdf_file.create_dataset('dict_names', data=string_array)
        elif isinstance(item, list):  # Write a list
            group_name = 'list'
            list_names = []
            for list_item in item:
                # list_names.append(str(list_item.uuid))
                # list_item_group = hdf_file.create_group(list_names[-1])
                subgroup_name = self._write_item_to_hdf5_file_recursive(list_item, hdf_file, unit_system)
                list_names.append(subgroup_name)
            string_array = np.array(list_names, dtype='S')
            hdf_file.create_dataset(group_name, data=string_array)
        # Write a folder
        elif hasattr(item, 'class_name') and item.class_name in self.folder_class_names:
            folder_children = []
            group_name, folder_group = self.write_folder_to_hdf5_file(item, hdf_file, unit_system)
            for child in item.children:
                child_name = self._write_item_to_hdf5_file_recursive(child, folder_group, unit_system)
                folder_children.append(child_name)
            string_array = np.array(folder_children, dtype='S')
            folder_group.create_dataset('children', data=string_array)
        else:  # Write a calculator
            group_name = self.write_calculator.write_calculator_to_hdf5(item, hdf_file, unit_system)

        return group_name

    def write_folder_to_hdf5_file(self, folder, hdf_file, unit_system):
        """Exports the folder and children to the file.

        Args:
            folder (Folder): the folder to save to a file
            hdf_file: group within an HDF file
            unit_system: the selected Unit System
        """
        folder_group_name = f'{folder.name}_{str(folder.uuid)}'
        folder_group = hdf_file.create_group(folder_group_name)
        folder_group.attrs['datatype'] = folder.class_name
        folder_group.attrs['name'] = folder.name
        folder_group.attrs['class_name'] = folder.class_name
        folder_group.attrs['is_checked'] = folder.is_checked
        folder_group.attrs['is_partially_checked'] = folder.is_partially_checked
        folder_group.attrs['is_enabled'] = folder.is_enabled
        folder_group.attrs['is_expanded'] = folder.is_expanded
        # Do not save selection; We do not want selections to persist
        # folder_group.attrs['is_selected'] = folder.is_selected
        if folder.tool_tip:
            folder_group.attrs['tool_tip'] = folder.tool_tip
        folder_group.attrs['item_uuid'] = str(folder.item_uuid)
        if folder.parent_uuid:
            folder_group.attrs['parent_uuid'] = str(folder.parent_uuid)
        folder_group.attrs['project_uuid'] = str(folder.project_uuid)

        if hasattr(folder, 'project_settings'):
            # Write project settings
            group = self._write_item_to_hdf5_file_recursive(folder.project_settings, hdf_file, unit_system)
            folder_group.attrs['project_settings'] = str(group)
            # Write profile settings

        return folder_group_name, folder_group

    def check_for_file_lock(self, filename, overwrite):
        """Simple check if the filename is available to write.

        Args:
            filename (string): filename to check
            overwrite (bool): should we overwrite the file?

        Returns:
            True if the file can be written (or overwritten) to.
            False if the file exists (and can't overwrite) or is locked.
        """
        if os.path.exists(filename) and overwrite:
            try:
                with h5py.File(filename, "w"):
                    return True
            except OSError:
                print(f"File '{filename}' is locked.")
                return False
        else:
            return True
