"""Function that runs our application."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from importlib import import_module
from pathlib import Path
from typing import Any, Dict, Optional

# 2. Third party modules
from injector import Injector

# 3. Aquaveo modules
# from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.interface_adapters.controller.action_router import action_router
from xms.FhwaVariable.interface_adapters.interactors.app_interactor import AppInteractor

# 4. Local modules
from xms.HydraulicToolboxCalc.model.hydraulic_toolbox import HydraulicToolbox
# from xms.HydraulicToolboxCalc.model.NeXMS import NeXMS


# # Temporary Fix for VS Code debugger Qt integration issue
# import os  # noqa: E402, I100, I201
# os.environ['QT_API'] = 'pyside6'
# import matplotlib  # noqa: E402, I100, I201
# matplotlib.use('Agg')  # Use non-GUI backend to avoid Qt conflicts


def call_controller_method(module_name: str, class_name: str, method_name: str, params: Optional[Dict[str, Any]],
                           injector: Injector) -> None:
    """
    Call a controller method given strings for the module, class, and method names.

    Args:
        module_name: The module name.
        class_name: The class name.
        method_name: The method name.
        params: The controller params.
        injector: The injector.
    """
    try:
        module = import_module(module_name)
        class_ = getattr(module, class_name)
        controller = class_(injector=injector)
        method = getattr(controller, method_name)
    except AttributeError:
        raise NotImplementedError(
            f"Controller class '{class_name}' does not implement '{method_name}'")
    return method(**params)


def create_injector() -> Injector:
    """
    Assemble dependency injector from provider modules.

    Returns:
        The dependency injector.
    """
    modules = []

    injector = Injector(modules)
    return injector


def setup_action_router(injector: Injector) -> None:
    """
    Setup controllers to be called using class and method name.

    Args:
        injector: The injector.
    """
    def _call_controller(class_name: str, method_name: str, params: Optional[Dict[str, Any]]) -> None:
        return call_controller_method("xms.FhwaVariable.interface_adapters.interactors.app_interactor", class_name,
                                      method_name, params, injector)
    action_router.handler = _call_controller


def main(view=None, graphics_view=None):
    """Main function for the Hydraulic Toolbox GUI."""
    app_folder = Path(__file__).parent
    frozen_app_path = 'xms/HydraulicToolboxCalc'

    hyd_tb = HydraulicToolbox(app_folder=app_folder, frozen_app_path=frozen_app_path, view=view,
                              graphics_view=graphics_view)
    # hyd_tb = NeXMS(app_folder=app_folder, frozen_app_path=frozen_app_path)

    injector = create_injector()
    setup_action_router(injector)

    app = AppInteractor(hyd_tb.model_name)
    # Run after the AppInteractor is created for functors
    hyd_tb.app_data.model_dict[hyd_tb.model_name].initialize_tools(about_icon=hyd_tb.model_icon)

    app.run()
