"""GradationLayerData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class GradationLayerCalc(Calculator):
    """Provides a class that will define the site data of a culvert barrel."""
    def __init__(self, ):
        """Initialize the GradationLayerCalc class."""
        super().__init__()

        self.d50 = 0.0
        self.d84 = 0.0
        self.tau = 0.0
        self.ks = 0.0
        self.d_star = 0.0
        self.noncohesive_results = 'None'
        self.gradation_name = None

        self.soil_type = None
        self.sediment_dia = None
        self.percent_pass = None

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
            (str): warning message
        """
        self.manning_n = 0.012
        _, self.soil_type = self.get_data('Soil type')
        if self.soil_type == 'Noncohesive':
            sediments = self.input_dict['calc_data']['Gradation']['Sediment particle diameter']
            passing = self.input_dict['calc_data']['Gradation']['Percent passing']

            if len(sediments) <= 0:
                self.warnings['Gradation length'] = 'Please enter gradation data.'
                return False

            if len(sediments) != len(passing):
                self.warnings['Gradation length'] = \
                    'Sediment particle diameter and percent passing must be the same length.'
                return False

            # Filter out zero values in passing and their corresponding indices in sediments
            filtered_data = [(p, s) for p, s in zip(passing, sediments) if p != 0.0 and s != 0.0]
            if not filtered_data:
                self.warnings['Gradation length'] = 'Please enter gradation data.'
                return False

            # Sort by passing values (first element of each tuple)
            filtered_data.sort(key=lambda x: x[0])

            # Unzip the sorted data back into separate lists
            self.percent_pass, self.sediment_dia = zip(*filtered_data)

            # Convert back to lists (zip returns tuples)
            self.percent_pass = list(self.percent_pass)
            self.sediment_dia = list(self.sediment_dia)

        else:  # Cohesive
            result = self.check_float_vars_to_greater_zero(['Critical shear stress (τc)'])
            return result
        return True

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.noncohesive_results = 'None'
        self.gradation_name = None

        # plot_options = self.input['Plot options']['Gradation'].get_val()
        layer_plot_dict = None
        if 'Soil layers' in self.plot_dict and 'series' in self.plot_dict['Soil layers']:
            layer_plot_dict = self.plot_dict["Soil layers"]['series'][0]
            _, name = self.get_data('Name')
            layer_plot_dict['Name'] = name

        if self.soil_type == 'Noncohesive':
            self.compute_noncohesive()
            if layer_plot_dict is not None:
                layer_plot_dict['x var'].set_val(self.input_dict['calc_data']['Gradation'][
                    'Sediment particle diameter'])
                layer_plot_dict['y var'].set_val(self.input_dict['calc_data']['Gradation']['Percent passing'])
        else:
            self.compute_cohesive()

        self.tau = self.compute_critical_shear_tau()
        self.manning_n = self.compute_manning_n()

        return True

    def compute_noncohesive(self):
        """Computes the data for noncohesive soils."""
        self.results['Gradation'] = {}
        self.results['Gradation']['Calculate texture'] = False

        self.d84 = 0.0
        d50 = None
        if len(self.percent_pass) > 0:
            if len(self.percent_pass) > 1 and len(self.sediment_dia) >= 2:
                d15 = self.get_diameter_for_passing(0.15)
                d50 = self.get_diameter_for_passing(0.5)
                d84 = self.get_diameter_for_passing(0.84)
                self.d84 = d84
                d85 = self.get_diameter_for_passing(0.85)
                d95 = self.get_diameter_for_passing(0.95)
                d10 = self.get_diameter_for_passing(0.1)
                d60 = self.get_diameter_for_passing(0.6)
                d30 = self.get_diameter_for_passing(0.3)
                if d10 == 0:
                    cu = 0.0
                    cc = 0.0
                else:
                    cu = d60 / d10
                    cc = (d30 ** 2) / (d10 * d60)

                _, uniform_tol = self.get_data('Uniformity coefficient tolerance')

                graded = 'Poorly graded'
                if cu > 4 and 1 < cc < 3:
                    graded = 'Well graded'
                elif abs(cu - 1) < uniform_tol:
                    graded = 'Uniformly graded'

                self.results['Gradation']['D15'] = d15
                self.results['Gradation']['D50'] = d50
                self.results['Gradation']['D85'] = d85
                self.results['Gradation']['D95'] = d95
                self.results['Gradation']['Cu'] = cu
                self.results['Gradation']['Cc'] = cc
                self.results['Gradation']['Graded'] = graded

                self.results['Shear decay parameters'] = {}
                sigma = 0.0
                if d50 > 0:
                    sigma = d84 / d50
                self.results['Shear decay parameters']['Sediment gradation coefficient (σ)'] = sigma

                self.noncohesive_results = 'Full gradation'

            elif 0.5 in self.percent_pass:
                d50 = self.get_diameter_for_passing(0.5)
                self.results['Gradation']['D50'] = d50
                self.noncohesive_results = 'Single D50'

            elif 0.84 in self.percent_pass:
                d84 = self.get_diameter_for_passing(0.84)
                self.results['Gradation']['D84'] = d84
                self.noncohesive_results = 'Single D84'
                self.d84 = d84

            elif len(self.sediment_dia) >= 1:
                self.noncohesive_results = 'Single value'
                name = f'D{int(self.percent_pass[0] * 100.0)}'
                # self.results['Gradation'][name] = copy.deepcopy(self.results['Gradation']['D50'])
                # self.results['Gradation'][name].name = name
                self.results['Gradation'][name] = self.sediment_dia[0]
                self.gradation_name = name

            if d50 is not None:
                self.d50 = d50
                classification = self.compute_soil_category(d50)
                self.results['Gradation']['Classification'] = classification

                # The following commented code allows the user to categorize the gradation into:
                # clay, silt, sand, granular, gravel, cobble, and riprap
                # I'm leaving the code here, as it is possible we may want a more defined gradation system in the future
                # See also code in the gradation_settings.py file in the compute_cohesive method

                # _, max_cohesive_diameter = self.get_data('Max cohesive d50 size')
                # _, max_silt_diameter = self.get_data('Max silt d50 size')
                # _, max_sand_diameter = self.get_data('Max sand d50 size')
                # _, max_granular_diameter = self.get_data('Max granular d50 size')
                # _, max_gravel_diameter = self.get_data('Max gravel d50 size')
                # _, max_cobble_diameter = self.get_data('Max cobble d50 size')

                # material = 'Cohesive'
                # if d50 > max_cobble_diameter:
                #     material = 'Riprap'
                # elif d50 > max_gravel_diameter:
                #     material = 'Cobble'
                # elif d50 > max_granular_diameter:
                #     material = 'gravel'
                # elif d50 > max_sand_diameter:
                #     material = 'Granular'
                # elif d50 > max_silt_diameter:
                #     material = 'sand'
                # elif d50 > max_cohesive_diameter:
                #     material = 'silt'

                # self.results['Gradation']['Category'].set_val(material)
            else:
                self.d50 = 0.0

            if self.noncohesive_results == 'Full gradation':
                self.compute_usda_texture()

    def compute_soil_category(self, d50):
        """Computes the soil category based on the d50 value.

        Args:
            d50 (float): the d50 value
        """
        unit_converter = ConversionCalc()
        _, d50_mm = unit_converter.convert_units('ft', 'mm', d50)
        _, gradation_system = self.get_data('Gradation system')
        classification = 'clay'
        if gradation_system == 'USDA':
            if 0.05 > d50_mm > 0.002:
                classification = 'silt'
            elif 0.1 > d50_mm >= 0.05:
                classification = 'very fine sand'
            elif 0.25 > d50_mm >= 0.1:
                classification = 'fine sand'
            elif 0.5 > d50_mm >= 0.25:
                classification = 'medium sand'
            elif 1.0 > d50_mm >= 0.5:
                classification = 'coarse sand'
            elif 2.0 > d50_mm >= 1.0:
                classification = 'very coarse sand'
            elif 75 > d50_mm >= 2.0:
                classification = 'gravel'
            elif d50_mm > 75:
                classification = 'riprap'
        elif gradation_system == 'UNIFIED':
            if 0.074 > d50_mm:
                classification = 'silt or clay'
            elif 0.42 > d50_mm >= 0.075:
                classification = 'fine sand'
            elif 2.0 > d50_mm >= 0.42:
                classification = 'medium sand'
            elif 5.0 > d50_mm >= 2.0:
                classification = 'coarse sand'
            elif 20 > d50_mm >= 5.0:
                classification = 'fine gravel'
            elif 75 > d50_mm >= 20:
                classification = 'coarse gravel'
            elif d50_mm > 75:
                classification = 'riprap'
        elif gradation_system == 'AASHTO':
            if 0.074 > d50_mm > 0.005:
                classification = 'silt'
            elif 0.42 > d50_mm >= 0.075:
                classification = 'very fine sand'
            elif 2.0 > d50_mm >= 0.42:
                classification = 'coarse sand'
            elif 9.5 > d50_mm >= 2.0:
                classification = 'fine gravel'
            elif 25.4 > d50_mm >= 9.5:
                classification = 'medium gravel'
            elif 75 > d50_mm >= 25.4:
                classification = 'coarse gravel'
            elif d50_mm > 75:
                classification = 'riprap'
        return classification

    def compute_shields_number_ks(self):
        """Computes the Shields number ks."""
        _, rho_w = self.get_data('Water density', 1.94)
        _, rho_s = self.get_data('Sediment density', 5.141)
        s = rho_s / rho_w
        _, v = self.get_data('Kinematic viscosity of water', 1.208e-05)
        _, g = self.get_data('Gravity', 32.2)
        self.d_star = (((s - 1) * g) / v ** 2.0) ** (1.0 / 3.0) * self.d50
        if 'Shear decay parameters' not in self.results:
            self.results['Shear decay parameters'] = {}
        self.results['Shear decay parameters']['Dimensionless grain size diameter (D*)'] = self.d_star
        # Gou's (2002) critical shields number approximation uses the dimensionless grain size diameter
        self.ks = 0.0
        if self.d_star > 0:
            self.ks = 0.23 / self.d_star + 0.054 * (1 - math.exp(-self.d_star ** 0.85 / 23.0))
        self.results['Shear decay parameters']['Shields number (ks)'] = self.ks
        return self.ks, s

    def compute_critical_shear_tau(self):
        """Computes the critical shear tau."""
        self.ks, s = self.compute_shields_number_ks()
        _, gamma_w = self.get_data('Unit weight of water (γw)', 62.4)
        _, min_d50 = self.get_data('Min D50 for critical shear method', 0.000656168)  # 0.2mm
        if self.soil_type == 'Noncohesive' and self.d50 >= min_d50:
            # Equation for the Shields number is rearranged to calculate the critical shear stress (tau_c)
            tau = self.ks * (s - 1) * gamma_w * self.d50
        elif self.soil_type == 'Cohesive':
            _, tau = self.get_data('Critical shear stress (τc)')
        else:
            tau = 0.0
        self.results['Shear decay parameters']['Critical shear stress (τc)'] = tau
        return tau

    # Generate Manning's n from Strickler's eqn (or 0.01 for cohesive-adjust between 0.01 and 0.012)
    # Strickler's equation for n (Noncohesive) (0.041 for SI, 0.034 for English)
    def compute_manning_n(self):
        """Manning's n from Strickler's eqn (or 0.01 for cohesive-adjust between 0.01 and 0.012)."""
        _, min_d50 = self.get_data('Min D50 for critical shear method', 0.000656168)  # 0.2mm
        manning_n = 0.012
        if self.soil_type == 'Noncohesive' and self.d50 >= min_d50:
            manning_n = 0.034 * (self.d50) ** (1.0 / 6.0)
        self.results['Shear decay parameters']['Manning n'] = manning_n
        return manning_n

    def compute_usda_texture(self):
        """Computes the USDA soil texture.

        Args:
            usda_clay (float): Percent passing for the clay diameter
            usda_silt (float): Percent passing for the silt diameter
            usda_sand (float): Percent passing for the sand diameter
        """
        usda_clay = 0.00000656  # 0.002 mm
        usda_silt = 0.00016404  # 0.05 mm
        usda_sand = 0.00656168  # 2.0 mm

        usda_clay_passing = self.get_passing_for_diameter(usda_clay)
        usda_silt_passing = self.get_passing_for_diameter(usda_silt)
        usda_sand_passing = self.get_passing_for_diameter(usda_sand)

        if usda_sand_passing > 0.5:
            usda_clay_small_percent = usda_clay_passing
            usda_silt_small_percent = usda_silt_passing - usda_clay_passing
            usda_sand_small_percent = usda_sand_passing - usda_silt_passing

            total_percent = usda_clay_small_percent + usda_silt_small_percent + usda_sand_small_percent

            usda_clay_percent = usda_clay_small_percent / total_percent
            usda_silt_percent = usda_silt_small_percent / total_percent
            usda_sand_percent = usda_sand_small_percent / total_percent

            if usda_clay_percent >= 0.4:
                if usda_silt_percent >= 0.4:
                    soil_texture = 'silty clay'
                elif usda_sand_percent >= 0.45:
                    soil_texture = 'sandy clay'
                else:
                    soil_texture = 'clay'
            elif 0.275 <= usda_clay_percent <= 0.40 and usda_sand_percent <= 0.2:
                soil_texture = 'silty clay loam'
            elif usda_silt_percent >= 0.5 and usda_clay_percent <= 0.275:
                if usda_silt_percent >= 0.8 and usda_clay_percent <= 0.125:
                    soil_texture = 'silt'
                else:
                    soil_texture = 'silt loam'
            elif 0.35 <= usda_clay_percent <= 0.55 and 0.45 <= usda_sand_percent:
                soil_texture = 'sandy clay'
            elif 0.20 <= usda_clay_percent <= 0.35 and 0.275 >= usda_silt_percent and \
                    0.45 <= usda_sand_percent:
                soil_texture = 'sandy clay loam'
            elif 0.20 >= usda_clay_percent and 0.525 <= usda_sand_percent or 0.75 <= usda_clay_percent and \
                    0.5 <= usda_silt_percent:
                sand_percent_on_line = 0.5 * usda_clay_percent + 0.85
                if usda_sand_percent >= sand_percent_on_line:
                    soil_texture = 'sand'
                elif usda_sand_percent >= usda_clay_percent + 0.70:
                    soil_texture = 'loamy sand'
                else:
                    soil_texture = 'sandy loam'
            else:
                if usda_clay_percent >= 0.275:
                    soil_texture = 'clay loam'
                else:
                    soil_texture = 'loam'

            self.results['Gradation']['Calculate texture'] = True
            self.results['Gradation']['Texture'] = soil_texture

    def compute_cohesive(self):
        """Computes the data for cohesive soils."""
        self.d50 = 0.0
        if 'Gradation' not in self.results:
            self.results['Gradation'] = {}
        self.results['Gradation']['Classification'] = 'clay'

    def get_diameter_for_passing(self, passing):
        """Returns the Percent passing for a given diameter.

        Args:
            Percent passing (float): Percent passing for the given diameter

        Returns:
            diameter (float): diameter of the sediment particle
        """
        if self.percent_pass is None or self.sediment_dia is None:
            return 0.0
        if passing in self.percent_pass:
            index = self.percent_pass.index(passing)
            return self.sediment_dia[index]
        else:
            y_vals = self.sediment_dia
            x_vals = self.percent_pass

            # Check if all values in x_vals or y_vals are zero
            if all(val == 0 for val in x_vals):
                return 0.0
            if all(val == 0 for val in y_vals):
                return 0.0

            paired_lists = list(zip(x_vals, y_vals))
            paired_lists.sort()
            x_vals_sorted, y_vals_sorted = zip(*paired_lists)

            _, null_data = self.get_data('Null data', -9999)
            _, zero_tol = self.get_data('Zero tolerance', 1e-6)
            interpolator = Interpolation(x_vals_sorted, y_vals_sorted, null_data=null_data, zero_tol=zero_tol)

            interped_result = float(interpolator.interpolate_y(passing)[0])
            if interped_result < 0.0:
                interped_result = 0.0
            return interped_result

    def get_passing_for_diameter(self, diameter):
        """Returns the Percent passing for a given diameter.

        Args:
            diameter (float): diameter of the sediment particle

        Returns:
            Percent passing (float): Percent passing for the given diameter
        """
        if diameter in self.sediment_dia:
            index = self.sediment_dia.index(diameter)
            return self.percent_pass[index]
        else:
            y_vals = self.percent_pass
            x_vals = self.sediment_dia

            # Check if all values in x_vals or y_vals are zero
            if all(val == 0 for val in x_vals):
                return 0.0
            if all(val == 0 for val in y_vals):
                return 0.0

            paired_lists = list(zip(x_vals, y_vals))
            paired_lists.sort()
            x_vals_sorted, y_vals_sorted = zip(*paired_lists)

            _, null_data = self.get_data('Null data', -9999)
            _, zero_tol = self.get_data('Zero tolerance', 1e-6)
            interpolator = Interpolation(x_vals_sorted, y_vals_sorted, null_data=null_data, zero_tol=zero_tol)
            passing, _ = interpolator.interpolate_y(diameter, extrapolate=True)
            if passing > 1.0:
                passing = 1.0
            return passing

    def get_critical_shear_stress(self):
        """Returns the critical shear stress.

        Returns:
            critical shear stress (float): critical shear stress
        """
        if self.sediment_dia is None:
            self._get_can_compute()
            self._compute_data()
        return self.results['Shear decay parameters']['Critical shear stress (τc)']

    def get_d50(self):
        """Returns the D50 of the gradation.

        Returns:
            D50 (float): D50 of the gradation
        """
        self._get_can_compute()
        if self.input_dict['calc_data']['Soil type'] == 'Noncohesive':
            return self.get_diameter_for_passing(0.5)
        return 0.0

    def get_d84(self):
        """Returns the D84 of the gradation.

        Returns:
            D84 (float): D84 of the gradation
        """
        self._get_can_compute()
        if self.input_dict['calc_data']['Soil type'] == 'Noncohesive':
            return self.get_diameter_for_passing(0.84)
        return 0.0
