"""SpecifyBoundaryData Class (used in GVF to specify the boundary at both ends."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class SpecifyBoundaryCalc(Calculator):
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """

    def get_starting_water_depth_and_velocity(self, normal_depth, normal_velocity, critical_depth, critical_velocity,
                                              flow, index=0, select=None):
        """Returns the starting water depth and velocity for the given data.

        Args:
            normal_depth (float): computed normal depth for the channel
            normal_velocity (float): computed average velocity at normal depth
            critical_depth (float): computed critical depth for the flow and geometry
            critical_velocity (float): computed average velocity at critical depth
            flow (float): flowrate of the channel for current computations
            index (int): index of the result that is applicable to this computation
            select (str): the selected boundary condition
        """
        if select is None:
            select = self.input_dict['calc_data']['Select']
        if select == 'normal depth':
            return normal_depth, normal_velocity
        elif select == 'critical depth':
            # TODO: Consider adding brink depth for appropriate channel shapes
            return critical_depth, critical_velocity
        elif select == 'constant depth':
            return self.input_dict['calc_data']['Constant depth'], 0.0  # Assume pool condition and zero velocity
        elif select == 'rating curve':
            _, null_data = self.get_data('Null data')
            _, zero_tol = self.get_data('Zero tolerance')
            flow_table = self.input_dict['calc_data']['Rating curve']['Flow']
            wse_table = self.input_dict['calc_data']['Rating curve']['Elevation']
            velocity_table = self.input_dict['calc_data']['Rating curve']['Velocity']

            wse_interp = Interpolation(x_values=flow_table, y_values=wse_table, null_data=null_data, zero_tol=zero_tol)
            depth, _ = wse_interp.interpolate_y(flow)

            vel_interp = Interpolation(x_values=flow_table, y_values=velocity_table, null_data=null_data,
                                       zero_tol=zero_tol)
            velocity, _ = vel_interp.interpolate_y(flow)

            return depth, velocity
        elif select == 'specified depths':
            depths = 0.0
            velocities = 0.0
            if len(self.input_dict['calc_data']['Specified depths']) > index:
                depths = self.input_dict['calc_data']['Specified depths'][index]
            if len(self.input_dict['calc_data']['Specified velocities']) > index:
                velocities = self.input_dict['calc_data']['Specified velocities'][index]
            return depths, velocities
        elif select == 'channel calculations':
            self.manning_n_calc.input_dict['calc_data']['Flows'] = [flow]
            self.manning_n_calc.compute_data()
            depths = self.manning_n_calc.results['Depths'][-1]
            velocities = self.manning_n_calc.results['Average velocity'][-1]
            return depths, velocities

    def _get_can_compute(self):
        """Determine whether we have enough data for a valid computation."""
        if self.input_dict['calc_data']['Select'] in ['normal depth', 'critical depth']:
            return True
        elif self.input_dict['calc_data']['Select'] == 'constant depth':
            if self.input_dict['calc_data']['Constant depth'] >= 0.0:
                return True
        elif self.input_dict['calc_data']['Select'] == 'rating curve':
            result = True
            if len(self.input_dict['calc_data']['Rating curve']['Flow']) <= 0:
                result = False
                self.warnings['rating_curve_1'] = f'Please enter rating curve data {self.boundary_name}'
            found = False
            if 'rating_curve_1' not in self.warnings:
                for flow in self.input_dict['calc_data']['Rating curve']['Flow']:
                    if flow > 0.0:
                        found = True
                        break
                if not found:
                    result = False
                    self.warnings['rating_curve_2'] = \
                        f'Please enter a positive, non-zero value for flows {self.boundary_name}'
            if 'rating_curve_1' not in self.warnings:
                found = False
                for elev in self.input_dict['calc_data']['Rating curve']['Elevation']:
                    if elev > 0.0:
                        found = True
                        break
                if not found:
                    result = False
                    self.warnings['rating_curve_3'] = \
                        f'Please enter a positive, non-zero value for elevations {self.boundary_name}'
            return result
        elif self.input_dict['calc_data']['Select'] == 'specified depths':
            depths = self.input_dict['calc_data']['Specified depths']
            if len(depths) == 0:
                self.warnings['depth'] = f'Please enter a specified depth {self.boundary_name}'
                return False
            found = False
            for depth in depths:
                if depth > 0.0:
                    found = True
                    break
            if not found:
                self.warnings['depth'] = \
                    f'Please enter a positive, non-zero value for a depth {self.boundary_name}'
                return False
            return True
        return False

    def set_specified_depths_and_velocities(self, depths, velocities):
        """Set the specified depths and velocities.

        Args:
            depths (list of depths): water depths
            velocities (list of velocities): water velocities
        """
        if type(depths) is not list:
            depths = [depths]
        if type(velocities) is not list:
            velocities = [velocities]
        self.input_dict['calc_data']['Select'] = 'specified depths'
        self.input_dict['calc_data']['Specified depths'] = depths
        self.input_dict['calc_data']['Specified velocities'] = velocities

    def get_specified_depths_and_velocities(self):
        """Get the specified depths and velocities.

        Returns:
            depths (list of depths): water depths
            velocities (list of velocities): water velocities
        """
        return self.input_dict['calc_data']['Specified depths'], \
            self.input_dict['calc_data']['Specified velocities']
