"""Bridge Geometry Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.calculator.table_data import TableData
# from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.geometry.bridge_deck_table import BridgeDeckTable
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.geometry.pier_geometry import PierGeometry


class BridgeGeometry(VariableGroup):
    """Provides a class that will define the cross-section of a bridge."""

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Class.

        Args:
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Bridge Geometry CalcData'
        self.type = 'BridgeGeometry'

        self.theme = self.get_theme()

        # max_value = self.max_value
        # Input
        self.input = {}

        structure_color = self.theme['Plot structure color']
        structure_fill_color = self.theme['Plot structure fill color']

        soil_color = self.theme['Plot soil color']
        soil_fill_color = self.theme['Plot soil fill color']

        cross_section = {
            'Station': Variable('Station', 'float_list', 0, [0.0], precision=2,
                                limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'Elevation': Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                  limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                  native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        }
        #   Approach Cross-section
        name = 'Approach cross-section'
        self.input[name] = Variable(
            name, 'table', TableData(self.theme, name=name, plot_names=[name], input=copy.deepcopy(cross_section),
                                     min_items=3, app_data=app_data, model_name=model_name, project_uuid=project_uuid),
            note='Approach cross-section should be upstream of the bridge outside of the influence of the bridge.')
        self.input[name].get_val().set_plot_series_options(
            name, related_index=0, index=0, name=name, x_axis='Station', y_axis='Elevation',
            line_color=soil_color, linetype='dashed', line_width=1.5, fill_below_line=False,
            fill_color=soil_fill_color, pattern='sand')

        #   Bridge Cross-section
        name = 'Bridge cross-section'
        self.input[name] = Variable(
            name, 'table', TableData(self.theme, name=name, plot_names=['Bridge cross-section'],
                                     input=copy.deepcopy(cross_section), min_items=3,
                                     app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input[name].get_val().set_plot_series_options(
            name, related_index=0, index=0, name=name, x_axis='Station', y_axis='Elevation',
            line_color=soil_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=soil_fill_color, pattern='sand')

        #   Bridge Deck Geometry
        deck_geometry = {
            'Station': Variable('Station', 'float_list', 0, [0.0], precision=2,
                                limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'High chord': Variable('High chord', 'float_list', 0, [0.0], precision=2,
                                   limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                   native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'Low chord': Variable('Low chord', 'float_list', 0, [0.0], precision=2,
                                  limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                  native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        }
        name = 'Bridge deck geometry'
        self.input[name] = Variable(name, 'table', BridgeDeckTable(
            self.theme, name=name, plot_names=[name], input=deck_geometry, min_items=2, app_data=app_data,
            model_name=model_name, project_uuid=project_uuid))
        self.input[name].get_val().set_plot_series_options(
            name, related_index=0, index=0, name=name, x_axis='Station', y_axis='Bridge deck',
            line_color=structure_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=structure_fill_color, pattern='concrete')

        #   Channel Bank Stations
        self.input['Channel left bank station'] = Variable(
            'Channel left bank station',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Channel right bank station'] = Variable(
            'Channel right bank station',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        # Abutment Toe Stations (or do we make full geometry?)
        self.input['Left abutment toe station'] = Variable(
            'Left abutment toe station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Right abutment toe station'] = Variable(
            'Right abutment toe station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Piers
        self.input['Piers'] = Variable(
            'Piers', 'calc_list',
            CalcOrVarlist(PierGeometry, app_data=app_data, model_name=model_name, project_uuid=project_uuid,
                          show_define_btn=False, default_name='pier', default_plural_name='piers', select_one=False,
                          min_number_of_items=0, show_number=True,
                          show_name=True, show_duplicate=False, show_delete=False))
        self.input['Piers'].value.input['Number of items'].set_val(0)  # Start with zero piers; let the user add them

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)
        self.plots = {}

    # def _get_can_compute(self):
    #     """Determine whether we have enough data to compute.

    #     Returns:
    #         True, if we can compute; otherwise, False
    #     """
    #     return True

    def get_val(self):
        """Computes and returns the results.

        Returns:
            self.results['result'] (variable): result of the computations.
        """
        self.compute_data()
        if 'result' in self.results:
            return self.results['result']
        return None

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        input_vars = self.input

        return input_vars

    # def _compute_data(self):
    #     """Computes the data possible; stores results in self.

    #     Returns:
    #         bool: True if successful
    #     """
    #     pass
