"""Pier Geometry Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.calculator.table_calc import TableCalc
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class PierCalc(TableCalc):
    """A class that defines the bridge deck table calculator."""

    def _compute_data(self):
        """Compute the data.

        Returns:
            True, if successful; otherwise, False
        """
        super()._compute_data()

        # self.check_list_length()
        # size = self.input['Table options'].get_val().input['Number of items'].get_val()
        # elevations = self.input['Data input'].get_val().input['Elevation'].get_val()
        # widths = self.input['Data input'].get_val().input['Width'].get_val()
        size = self.input_dict['calc_data']['Table options']['Number of items']
        elevations = self.input_dict['calc_data']['Data input']['Elevation']
        widths = self.input_dict['calc_data']['Data input']['Width']

        x_data = []
        y_data = []
        for i in range(size):
            x_data.append(self.centerline + widths[i] / 2)
            y_data.append(elevations[i])

        for i in range(size - 1, -1, -1):
            x_data.append(self.centerline - widths[i] / 2)
            y_data.append(elevations[i])

        if len(x_data) > 0:
            x_data.append(x_data[0])
            y_data.append(y_data[0])

        self.results['Width'] = x_data
        self.results['Elevation'] = y_data

        plot_name = 'Pier geometry'
        # plot_options = self.input['Plot options'][plot_name].get_val()
        # pier_plot_options = plot_options.get_item_by_name(plot_name)
        # if pier_plot_options:
        #     pier_plot_options.x_var = self.results['Width']
        #     pier_plot_options.y_var = self.results['Elevation']
        #     plot_options.set_item_by_name(pier_plot_options)
        self.plot_dict[plot_name]['series'][0]['x var'].set_val(self.results['Width'])
        self.plot_dict[plot_name]['series'][0]['y var'].set_val(self.results['Elevation'])

        return True

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        _, zero_tol = self.get_data('Zero tolerance', 1e-6)
        result = True

        if self.centerline == 0.0:
            self.warnings['Centerline station'] = 'Centerline station is zero.'

        elevations = self.input_dict['calc_data']['Data input']['Elevation']
        widths = self.input_dict['calc_data']['Data input']['Width']

        zero_width = False
        for width in widths:
            if width < zero_tol:
                zero_width = True
        if zero_width:
            self.warnings['width & elevations'] = 'Please specify a width for all elevations.'
            result = False

        if len(elevations) != len(set(elevations)):
            self.warnings['elev unique'] = 'Elevation values are not unique.'
            result = False

        return result


class PierTable(TableData):
    """Override the table class, so we can draw plots based on widths."""
    def __init__(self, theme, name, plot_names, name_of_items='points', stand_alone_calc=True, input=None, min_items=1,
                 max_items=sys.maxsize, show_increment=False, app_data=None, model_name=None, project_uuid=None):
        """Initializes the calculator.

        Args:
            theme (dict): the theme
            name (str): the name of the calculator
            plot_names (list of str): list of plot names
            name_of_items (str): name of the items
            stand_alone_calc (bool): whether the calculator is a stand-alone calculator
            input (dict): dictionary of input variables
            min_items (int): minimum number of items
            max_items (int): maximum number of items
            show_increment (bool): whether to show the increment
            app_data (AppData): the application data
            model_name (str): the name of the model
            project_uuid (str): the project UUID
        """
        super().__init__(theme=theme, name=name, plot_names=plot_names, name_of_items=name_of_items,
                         stand_alone_calc=stand_alone_calc, input=input, min_items=min_items, max_items=max_items,
                         show_increment=show_increment, app_data=app_data, model_name=model_name,
                         project_uuid=project_uuid)

        self.centerline = 0.0
        self.calculator = PierCalc()

        self.intermediate_to_copy.extend(['centerline'])


class PierGeometry(VariableGroup):
    """Provides a class that will define the geometry of a bridge pier."""

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Class.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Pier Geometry'
        self.type = 'PierGeometry'

        self.theme = self.get_theme()

        structure_color = self.theme['Plot structure color']
        structure_fill_color = self.theme['Plot structure fill color']

        # max_value = self.max_value
        # Input
        self.input = {}

        self.scour_reference = ['Thalweg', 'Local streambed']

        # self.input['Name'] = Variable('Name', 'string', 'pier', [])
        self.input['Centerline'] = Variable('Centerline station', 'float', 0.0, [],
                                            limits=(-sys.float_info.max, sys.float_info.max), precision=2,
                                            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
                                            si_units=self.si_mid_length)
        #  Geometry
        pier_geometry = {
            'Width': Variable('Width', 'float_list', 0, [0.0], precision=2,
                              limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                              native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'Elevation': Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                  limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                  native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        }
        name = 'Pier geometry'
        self.input[name] = Variable(
            name, 'table', PierTable(self.theme, name=name, plot_names=[name], input=pier_geometry, min_items=2,
                                     app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input[name].get_val().set_plot_series_options(
            name, related_index=0, index=0, name=name, x_axis='Width', y_axis='Elevation',
            line_color=structure_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=structure_fill_color, pattern='concrete')

        self.input['Scour reference point'] = Variable(
            'Pier scour reference point', "list", 0, self.scour_reference,
            note='Specify the depth and velocity at the max unit discharge for thalweg;'
            ' Specify local depth and velocity for local streambed')

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        # Update the centerline on the pier geometry
        self.input['Pier geometry'].get_val().centerline = self.input['Centerline'].get_val()

        input_vars = self.input

        return input_vars
