"""CalcData for performing Channel geometry operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.distance import xy_distance
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation
from xms.HydraulicToolboxCalc.util.intersection import find_intersection_and_closest_index
from xms.HydraulicToolboxCalc.util.list_utils import remove_nans
from xms.HydraulicToolboxCalc.util.smoothing import Smoothing


class ChannelGeometryCalc(Calculator):
    """A class that defines a channel and performs Channel geometry computations."""

    def __init__(self, stand_alone_calc=True, allowed_shapes='both', hide_depth=False,
                 zero_tol=None, open_shapes=None, closed_shapes=None, all_shapes=None, shapes=None):
        """Initializes the Channel geometry calculator.

        Args:
            stand_alone_calc (bool): Is the calculator a standalone calc or part of another class?
            allowed_shapes (string): sets which channel shapes the user can select: 'open', 'closed', 'both'
            hide_depth (bool): Is this class being used for tailwater calculations?
            zero_tol (float): The zero tolerance for calculations.
            open_shapes (list of str): List of open channel shapes.
            closed_shapes (list of str): List of closed channel shapes.
            all_shapes (list of str): List of shapes that can be either open or closed.
            shapes (list of str): List of all channel shapes.
        """
        super().__init__()

        self.hide_depth = hide_depth

        self.zero_tol = zero_tol

        self.update_cross_section = True

        # Input
        self.open_shapes = open_shapes
        self.closed_shapes = closed_shapes
        self.all_shapes = all_shapes
        self.shapes = shapes

    def compute_channel_geometry_parameters(self, embedment_depth):
        """Determines the low and and high of a given set of elevations.

        Args:
            embedment_depth (float): The depth of embedment to apply to the channel geometry.
        """
        if self.shape_is_closed:
            if 'calc_data' in self.input_dict and 'Shape' in self.input_dict['calc_data'] and \
                    self.input_dict['calc_data']['Shape'] == 'circle':
                self.unembedded_rise_to_crown = self.input_dict['calc_data']['Diameter']
            elif 'calc_data' in self.input_dict and 'Rise' in self.input_dict['calc_data']:
                self.unembedded_rise_to_crown = self.input_dict['calc_data']['Rise']
            if hasattr(self, 'unembedded_rise_to_crown') and self.unembedded_rise_to_crown is not None:
                self.rise_to_crown = self.unembedded_rise_to_crown - embedment_depth

        if len(self.elevations) == 0:
            return None

        self.thalweg_elevation = min(self.elevations)
        thalweg_index = self.elevations.index(self.thalweg_elevation)
        self.thalweg_station = self.stations[thalweg_index]

        self.crown_elevation = None
        self.crown_station = None
        self.left_max_bank = None
        self.left_max_bank_station = None
        self.right_max_bank = None
        self.right_max_bank_station = None

        if self.shape_is_closed:
            self.crown_elevation = max(self.elevations)
            crown_index = self.elevations.index(self.crown_elevation)
            self.crown_station = self.stations[crown_index]
            self.reorder_closed_shape_to_start_at_crown(crown_index)
            if 'calc_data' in self.input_dict and 'Shape' in self.input_dict['calc_data'] and self.input_dict[
                    'calc_data']['Shape'] in ['circle', 'elliptical', 'pipe arch', 'box', 'horseshoe']:
                pass  # If we have a closed shape with defined rise or diameter, we already set the rise_to_crown above
            else:
                self.rise_to_crown = self.crown_elevation - self.thalweg_elevation
                if not hasattr(self, 'unembedded_rise_to_crown') or self.unembedded_rise_to_crown < self.rise_to_crown:
                    self.unembedded_rise_to_crown = self.rise_to_crown
            return None

        else:
            self.reorder_open_shape_to_left_to_right()
            if thalweg_index == 0:
                self.warnings['Thalweg'] = 'Thalweg is at the left end of the channel.'
                self.left_max_bank = self.thalweg_elevation
                self.right_max_bank = max(self.elevations[thalweg_index:])
            elif thalweg_index == len(self.elevations) - 1:
                self.warnings['Thalweg'] = 'Thalweg is at the right end of the channel.'
                self.left_max_bank = max(self.elevations[0:thalweg_index])
                self.right_max_bank = self.thalweg_elevation
            else:
                self.left_max_bank = max(self.elevations[0:thalweg_index])
                self.left_max_bank_station = self.stations[self.elevations.index(self.left_max_bank)]
                self.right_max_bank = max(self.elevations[thalweg_index:])
                self.right_max_bank_station = self.stations[self.elevations.index(self.right_max_bank)]
                reverse_index = len(self.elevations) - 1 - self.elevations[::-1].index(self.right_max_bank)
                self.right_max_bank_station = self.stations[reverse_index]
            return min(self.left_max_bank, self.right_max_bank)

    def reorder_open_shape_to_left_to_right(self):
        """Reorders the stations and elevations of an open shape from left to right."""
        if self.stations[0] < self.stations[-1]:
            return

        self.stations.reverse()
        self.elevations.reverse()

    def reorder_closed_shape_to_start_at_crown(self, crown_index):
        """
        Reorders stations and elevations to start at the crown index.

        Args:
            crown_index (int): Index of the crown point
        """
        if crown_index != 0:
            # Remove duplicate last point temporarily
            stations_work = self.stations[:-1]
            elevations_work = self.elevations[:-1]

            # Reorder to start at crown
            self.stations = stations_work[crown_index:] + stations_work[:crown_index]
            self.elevations = elevations_work[crown_index:] + elevations_work[:crown_index]

            # Close the shape again if it was originally closed
            self.stations.append(self.stations[0])
            self.elevations.append(self.elevations[0])

        if len(self.stations) > 2 and self.stations[0] < self.stations[1]:
            # Assume that the shape is clockwise and we want ccw order
            self.stations.reverse()
            self.elevations.reverse()

    def vertical_wall_needed(self, max_wse):
        """Adds vertical walls to a set of stations and elevations to a specified elevation.

        Args:
            max_wse (float): Max WSE that we must handle

        Returns:
            tuple (list, list): The stations and elevations with a vertical wall (lists of floats)
        """
        if self.shape_is_closed:
            return False
        # if self.shape_grows_no_channel_depth_needed():
        #     return False
        _, only_fill_thalweg_areas = self.get_data('Only fill flow areas with thalweg locations')
        if only_fill_thalweg_areas:
            if max_wse > self.left_max_bank:
                return True
            if max_wse > self.right_max_bank:
                return True
        else:
            if max_wse > self.elevations[0]:
                return True
            if max_wse > self.elevations[-1]:
                return True
        return False

    # def shape_grows_no_channel_depth_needed(self):
    #     """Determines if the shape grows without needing channel depth adjustments."""
    #     if len(self.input_dict) == 0 or 'calc_data' not in self.input_dict or \
    #             'Shape' not in self.input_dict['calc_data']:
    #         return False
    #     if self.input_dict['calc_data']['Shape'] != 'cross-section' and not self.shape_is_closed:
    #         return True  # We don't need vertical walls on shapes that grow....
    #         # TODO: Check channel depth, perhaps?  Probably parabola shapes too?
    #     return False

    def add_vertical_wall_to_elevation(self, max_wse):
        """Adds vertical walls to a set of stations and elevations to a specified elevation.

        Args:
            max_wse (float): Max WSE that we must handle

        Returns:
            tuple (list, list): The stations and elevations with a vertical wall (lists of floats)
        """
        # if self.shape_grows_no_channel_depth_needed():
        if self.shape_is_closed:
            return

        self.left_vertical_station = None
        self.left_vertical_elevation = None
        self.right_vertical_station = None
        self.right_vertical_elevation = None
        _, only_fill_thalweg_areas = self.get_data('Only fill flow areas with thalweg locations')
        if only_fill_thalweg_areas:
            if max_wse > self.left_max_bank:
                self.left_vertical_station = self.stations[0]
                self.left_vertical_elevation = max_wse
            if max_wse > self.right_max_bank:
                self.right_vertical_station = self.stations[-1]
                self.right_vertical_elevation = max_wse
        else:
            if max_wse > self.elevations[0]:
                self.left_vertical_station = self.stations[0]
                self.left_vertical_elevation = max_wse
            if max_wse > self.elevations[-1]:
                self.right_vertical_station = self.stations[-1]
                # Make sure this piont is dry, not equal to the max_wse
                self.right_vertical_elevation = max_wse + self.zero_tol

    def initialize_geometry(self):
        """Initialize the geometry of the cross-section to prepare for calculations.

        First, correct for negative values.
        Second, Change the stations to begin at zero.
        Third, Add a vertical wall to even up the sides.
        Note, that this is done on a new station/elevation set so the originals remain unchanged.
        """
        self.channel_is_embedded = False
        self.embedment_is_evenly_across = False
        _, null_data = self.get_data('Null data')

        shapes_to_compute_first = copy.copy(self.shapes_to_compute_geom)
        if 'cross-section' in shapes_to_compute_first:
            shapes_to_compute_first.remove('cross-section')

        if self.input_dict['calc_data']['Shape'] in ['cross-section']:
            if self.update_cross_section:
                self.channel_stations = copy.copy(self.input_dict['calc_data']['Cross-section']['Station'])
                self.channel_elevations = copy.copy(self.input_dict['calc_data']['Cross-section']['Elevation'])
                self.channel_mannings_n = copy.copy(self.input_dict['calc_data']['Cross-section']["Manning's n"][:-1])
            else:
                self.channel_stations = copy.copy(self.stations)
                self.channel_elevations = copy.copy(self.elevations)
                # self.channel_mannings_n = copy.copy(self.mannings_n)  # No self.mannings_n; channel n already updated
            self.channel_stations, self.channel_elevations, self.channel_mannings_n = remove_nans(
                self.channel_stations, self.channel_elevations, null_data, n_data=self.channel_mannings_n)
            self.stations = copy.copy(self.channel_stations)
            self.elevations = copy.copy(self.channel_elevations)
            self.mannings_n = copy.copy(self.channel_mannings_n)
            self.low_elev, self.high_elev = self.determine_low_and_high_of_elevations()

        # Update embedment variable with key info
        self.determine_if_shape_is_closed()
        embedment_depth = self.update_determine_embedment_depth()
        if self.input_dict['calc_data']['Shape'] in shapes_to_compute_first or embedment_depth > 0.0:
            # self.update_determine_embedment_depth()
            self.compute_cross_sections_for_shapes(0.0, initializing=True)

        self.compute_channel_geometry_parameters(embedment_depth)
        self.ensure_ineffective_flow_stations_in_channel_stations()

        # Determine vertical walls
        if not self.shape_is_closed:
            # track min overtopping before vertical walls
            wses = self.input_dict['calc_data']['WSE']
            if wses is not None and len(wses) > 0:
                max_wse = (max(wses) - self.thalweg_elevation) * 2.0 + self.thalweg_elevation

                # Add vertical wall to make both sides equal to max elev
                self.add_vertical_wall_to_elevation(max_wse)

        self.channel_stations = copy.copy(self.stations)
        self.channel_elevations = copy.copy(self.elevations)

        self.determine_embedded_geometry(embedment_depth)
        self.low_elev, self.high_elev = self.determine_low_and_high_of_elevations()

        self.determine_if_shape_is_closed()

        # Determine the full flow area
        if self.shape_is_closed:
            self.full_flow_area = self._compute_area(self.high_elev - self.low_elev)

    def ensure_ineffective_flow_stations_in_channel_stations(self):
        """Ensure that ineffective flow stations are included in the channel stations."""
        if self.shape_is_closed:
            return
        self.ineffective_flow_stations = []
        ineffective_flow_stations = self.input_dict['calc_data']['Ineffective flow stations']
        if not ineffective_flow_stations or len(ineffective_flow_stations) == 0:
            return
        first_cs_station = min(self.stations)
        last_cs_station = max(self.stations)
        ineffective_flow_stations = self.consolidate_ineffective_flow_stations(
            first_cs_station, last_cs_station, ineffective_flow_stations)
        # Check if the ineffective flow stations are directly specified in channel stations
        # (This is important so we can separate the polygons)
        _, null_data = self.get_data('Null data', -9999.0)
        channel_interp = Interpolation(self.stations, self.elevations, null_data=null_data, zero_tol=self.zero_tol)
        if ineffective_flow_stations:
            for pair in ineffective_flow_stations:
                if len(pair) == 2:
                    next_start, next_end = pair

                    if next_start not in self.stations:
                        inef_y, inef_index = channel_interp.interpolate_y(next_start, False)
                        self.stations.insert(inef_index + 1, next_start)
                        self.elevations.insert(inef_index + 1, inef_y)
                    if next_end not in self.stations:
                        inef_y, inef_index = channel_interp.interpolate_y(next_end, False)
                        self.stations.insert(inef_index + 1, next_end)
                        self.elevations.insert(inef_index + 1, inef_y)
                    self.ineffective_flow_stations.append((next_start, next_end))

    def consolidate_ineffective_flow_stations(self, first_cs_station, last_cs_station, ineffective_flow_stations):
        """Sort and combine overlapping ineffective flow station pairs."""
        # Sort by start station (first value in each pair)
        ineffective_flow_stations.sort(key=lambda x: x[0])

        add_pair = True
        consolidated = []
        indices_to_skip = []
        for i in range(0, len(ineffective_flow_stations)):
            if i in indices_to_skip:
                continue
            current_start, current_end = ineffective_flow_stations[i]
            current_start, current_end, current_add_pair, first_cs_station, last_cs_station = \
                self.verify_ineffective_flow_stations(
                    current_start, current_end, first_cs_station, last_cs_station)

            if not current_add_pair:
                continue

            for j in range(i + 1, len(ineffective_flow_stations)):
                if j in indices_to_skip:
                    continue

                next_start, next_end = ineffective_flow_stations[j]

                next_start, next_end, add_pair, first_cs_station, last_cs_station = \
                    self.verify_ineffective_flow_stations(
                        next_start, next_end, first_cs_station, last_cs_station)

                if not add_pair:
                    indices_to_skip.append(j)
                    continue

                # Check if current and next overlap or are adjacent
                if next_start <= current_end:
                    # Overlap found - extend the current range, skip the index
                    indices_to_skip.append(j)
                    current_end = max(current_end, next_end)

            # Don't forget to add the last range (if not intentionally skipped)
            if current_add_pair:
                consolidated.append([current_start, current_end])

        return consolidated

    def verify_ineffective_flow_stations(self, start, end, first_cs_station, last_cs_station):
        """Verify that ineffective flow stations are valid."""
        add_pair = True
        # Check if stations are specified backwards
        if end < start:
            start, end = end, start
        # check if is completely outside the bounds of the channel stations
        if end <= first_cs_station or start >= last_cs_station:
            add_pair = False
            return start, end, add_pair, first_cs_station, last_cs_station
        # Ensure stations are within the channel bounds
        if start < first_cs_station:
            start = first_cs_station
        if end > last_cs_station:
            end = last_cs_station

        return start, end, add_pair, first_cs_station, last_cs_station

    def update_determine_embedment_depth(self):
        """Update the embedment depth and determine if the channel is embedded.

        Returns:
            float: The embedment depth
        """
        embed = self.input_dict['calc_data']['Embedment']
        embedment_rise = self.input_dict['calc_data']['Channel depth']
        if self.input_dict['calc_data']['Shape'] in [
                'elliptical',
                'pipe arch',
                'box',
                'horseshoe',
        ]:
            embedment_rise = self.input_dict['calc_data']['Rise']
        elif self.input_dict['calc_data']['Shape'] in ['circle']:
            embedment_rise = self.input_dict['calc_data']['Diameter']
        embed['calculator'].rise = embedment_rise
        embed['open_shape'] = not self.shape_is_closed
        embed['calculator'].input_dict = copy.copy(self.input_dict)
        embed['calculator'].input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Embedment'])
        embed_depth = embed['calculator'].get_embedment_depth()

        # Determine the embedment depth
        self.min_embedment_depth = embed_depth - embed['Low flow channel depth']
        _, self.zero_tol = self.get_data('Zero tolerance', 1e-5)
        if embed_depth < self.zero_tol:
            self.channel_is_embedded = False
            return 0.0
        self.channel_is_embedded = True
        self.embedment_is_evenly_across = embed['Embedment entry'] == 'embed evenly across channel'
        return embed_depth

    def determine_embedded_geometry(self, embedment_depth=None):
        """Compute the geometry after channel has been embedded.

        Args:
            embedment_depth (float): embedment depth
            embedment_depth (float): embedment depth
        """
        embed_elev = 0.0
        embed = self.input_dict['calc_data']['Embedment']
        if embedment_depth is None:
            embedment_depth = embed['Embedment depth']

        self.embed_stations = []
        self.embed_elevations = []
        self.embedment_stations = []
        self.embedment_elevations = []
        if embedment_depth <= self.zero_tol:
            # # No embedment, just copy the stations and elevations
            # self.embedment_stations = copy.copy(self.stations)
            # self.embedment_elevations = copy.copy(self.elevations)
            return
        stations = copy.copy(self.stations)
        elevations = copy.copy(self.elevations)
        self.stations = []
        self.elevations = []
        start_stations = []
        end_stations = []
        embed_poly_w_lowflow_stations = []  # This is to make adding a low-flow channel easier
        embed_poly_w_lowflow_elevations = []
        start_index = 0
        currently_embedding = False
        handle_embedment_poly = False
        embed_by_thalweg = embed['Embedment entry'] != 'embed evenly across channel'
        input_dict = copy.copy(self.input_dict)
        input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Composite n'])
        self.input_dict['calc_data']['Composite n']['calculator'].input_dict = input_dict
        channel_n = self.input_dict['calc_data']['Composite n']['calculator'].get_channel_n()
        embed_n = self.input_dict['calc_data']['Composite n']['calculator'].get_embedment_n()
        n_list = []

        self.embedment_stations = []
        self.embedment_elevations = []

        if embed_by_thalweg:
            thalweg = min(elevations)
            embed_elev = thalweg + embedment_depth
            for index in range(len(stations)):
                if elevations[index] < embed_elev:
                    if not currently_embedding:
                        if len(self.embedment_stations):
                            self.embed_stations.append('nan')
                            self.embed_elevations.append('nan')
                            self.embedment_stations.append('nan')
                            self.embedment_elevations.append('nan')
                        start_index = index
                        currently_embedding = True
                        handle_embedment_poly = True
                        # Determine the intersection of embedment to channel geom
                        prev_station = stations[index - 1]
                        prev_elevation = elevations[index - 1]
                        int_station = (prev_station + stations[index]) / 2.0
                        if (prev_elevation - elevations[index]) != 0.0:
                            int_station = (prev_station - stations[index]) / (prev_elevation - elevations[index]) * \
                                          (embed_elev - elevations[index]) + stations[index]
                        self.embedment_stations.append(int_station)
                        self.embedment_elevations.append(embed_elev)
                        self.embed_stations.append(int_station)
                        self.embed_elevations.append(embed_elev)
                        embed_poly_w_lowflow_stations.append(int_station)
                        embed_poly_w_lowflow_elevations.append(embed_elev)
                        start_stations.append(int_station)
                        self.stations.append(int_station)
                        self.elevations.append(embed_elev)
                        n_list.append(channel_n)

                    self.embedment_stations.append(stations[index])
                    self.embedment_elevations.append(elevations[index])
                    self.embed_stations.append(stations[index])
                    self.embed_elevations.append(embed_elev)
                    embed_poly_w_lowflow_stations.append(stations[index])
                    embed_poly_w_lowflow_elevations.append(elevations[index])
                    self.stations.append(stations[index])
                    self.elevations.append(embed_elev)
                    n_list.append(embed_n)
                else:
                    if currently_embedding:
                        # Determine the intersection of embedment to channel geom
                        prev_station = stations[index - 1]
                        prev_elevation = elevations[index - 1]
                        int_station = (prev_station - stations[index]) / (prev_elevation - elevations[index]) * \
                                      (embed_elev - elevations[index]) + stations[index]
                        self.embedment_stations.append(int_station)
                        self.embedment_elevations.append(embed_elev)
                        self.embed_stations.append(int_station)
                        self.embed_elevations.append(embed_elev)
                        embed_poly_w_lowflow_stations.append(int_station)
                        embed_poly_w_lowflow_elevations.append(embed_elev)
                        if handle_embedment_poly:
                            for poly_index in range(index - 1, start_index - 1, -1):
                                self.embedment_stations.append(stations[poly_index])
                                self.embedment_elevations.append(embed_elev)
                            handle_embedment_poly = False
                        end_stations.append(int_station)
                        currently_embedding = False
                        self.stations.append(int_station)
                        self.elevations.append(embed_elev)
                        n_list.append(embed_n)
                    self.stations.append(stations[index])
                    self.elevations.append(elevations[index])
                    if index > 0:
                        n_list.append(channel_n)

        else:
            self.elevations = [elevation + embedment_depth for elevation in elevations]
            self.stations = copy.copy(stations)
            start_stations.append(stations[0])
            end_stations.append(elevations[0])
            n_list.append(channel_n)

            self.embedment_stations.extend(self.stations)
            self.embedment_elevations.extend(self.elevations)
            self.embedment_stations.extend(reversed(stations))
            self.embedment_elevations.extend(reversed(elevations))

        low_flow_channel_depth = embed['Low flow channel depth']

        if low_flow_channel_depth > 0.0:
            # Initialize and check variables
            if low_flow_channel_depth > embedment_depth:
                low_flow_channel_depth = embedment_depth
            low_flow_channel_side_slope = embed['Low flow channel side slope']
            top_width_half = low_flow_channel_side_slope * low_flow_channel_depth
            # determine the length of the embedment so we can center the low flow channel
            embedment_length = 0.0
            for index in range(len(start_stations)):
                embedment_length += end_stations[index] - start_stations[index]
            halfway_length = embedment_length / 2.0
            channel_start = halfway_length - top_width_half
            channel_end = halfway_length + top_width_half
            prev_length = 0.0

            # Now we need to determine where the center of the embedment is in the channel's stationing
            start_station = 0.0
            thalweg_station = 0.0
            end_station = 0.0
            n_list = []

            for index in range(len(start_stations)):
                cur_station = start_stations[index] + prev_length + channel_end
                if cur_station < end_stations[index]:
                    #  We found where the low flow chanel will fit
                    start_station = start_stations[index] + prev_length + channel_start
                    thalweg_station = start_stations[index] + prev_length + halfway_length
                    end_station = start_stations[index] + prev_length + channel_end
                else:
                    prev_length = end_stations[index] - start_stations[index]

            # Now we can modify the geometry to have a low flow channel
            embedded_stations = []
            embedded_elevations = []
            self.embed_stations = []
            self.embed_elevations = []
            channel_added = False
            first_station_after_embed = False
            _, tol = self.get_data('Zero tolerance')
            for index in range(len(self.stations)):
                if self.elevations[index] < embed_elev + tol:
                    if self.stations[index] >= start_station:
                        if not channel_added:
                            embedded_stations.append(start_station)
                            elevation, new_index = self.determine_elevation_at_station_between_points(
                                index, start_station)
                            embedded_elevations.append(elevation)
                            self.embed_stations.append(start_station)
                            self.embed_elevations.append(elevation)
                            n_list.append(embed_n)

                            embedded_stations.append(thalweg_station)
                            elevation, new_index = self.determine_elevation_at_station_between_points(
                                new_index, thalweg_station)
                            embedded_elevations.append(elevation - low_flow_channel_depth)
                            self.embed_stations.append(thalweg_station)
                            self.embed_elevations.append(
                                elevation - low_flow_channel_depth)
                            n_list.append(embed_n)

                            embedded_stations.append(end_station)
                            elevation, new_index = self.determine_elevation_at_station_between_points(
                                new_index, end_station)
                            embedded_elevations.append(elevation)
                            self.embed_stations.append(end_station)
                            self.embed_elevations.append(elevation)
                            n_list.append(embed_n)

                            channel_added = True
                        elif self.stations[index] < start_station or self.stations[index] > end_station:
                            embedded_stations.append(self.stations[index])
                            embedded_elevations.append(self.elevations[index])
                            self.embed_stations.append(self.stations[index])
                            self.embed_elevations.append(self.elevations[index])
                            n_list.append(embed_n)
                            first_station_after_embed = True
                    elif self.stations[index] < start_station or self.stations[index] > end_station:
                        embedded_stations.append(self.stations[index])
                        embedded_elevations.append(self.elevations[index])
                        self.embed_stations.append(self.stations[index])
                        self.embed_elevations.append(self.elevations[index])
                        first_station_after_embed = True
                        n_list.append(embed_n)

                else:
                    embedded_stations.append(self.stations[index])
                    embedded_elevations.append(self.elevations[index])
                    if first_station_after_embed:
                        # Because we are going point by point, but assign the n value to segments, we need to
                        # correct the number of n values. I've decided to drop this point, as the n per segment
                        # are correct prior to this point.
                        n_list = n_list[:-1]
                        first_station_after_embed = False
                    n_list.append(channel_n)

            embed_poly_w_lowflow_stations.reverse()
            embed_poly_w_lowflow_elevations.reverse()
            self.embedment_stations = embed_poly_w_lowflow_stations
            self.embedment_elevations = embed_poly_w_lowflow_elevations
            self.embedment_stations.extend(self.embed_stations)
            self.embedment_elevations.extend(self.embed_elevations)

            self.stations = embedded_stations
            self.elevations = embedded_elevations

        if self.embedment_stations[0] != self.embedment_stations[-1] or \
                self.embedment_elevations[0] != self.embedment_elevations[-1]:
            self.embedment_stations.append(self.embedment_stations[0])
            self.embedment_elevations.append(self.embedment_elevations[0])

        if self.input_dict['calc_data']['Composite n']['n entry'] == 'specify channel and embedment n values':
            self.input_dict['calc_data']['Composite n']['n per geometry segment'] = n_list
            self.mannings_n = n_list

    def determine_elevation_at_station_between_points(self, index, mid_station):
        """Determine the elevation at a station that is between two points on the embedment profile.

        Args:
            index (int): the index of the current location in the embedment points
            mid_station (float): The station between the two points that we want the elevation of

        Returns:
            elevation (float): the elevation between the two points
            index (int): updated index to the station following the mid point
        """
        while self.stations[index] < mid_station:
            index += 1
        next_elevation = self.elevations[index]
        if index > 0:
            prev_station = self.stations[index - 1]
            prev_elevation = self.elevations[index - 1]
            elevation = (mid_station - prev_station) / (mid_station - prev_station) * \
                        (next_elevation - prev_elevation) + prev_elevation
        else:
            elevation = self.elevations[index]
        return elevation, index

    def check_for_vertical_wall_warning(self, wses, critical_wses):
        """Checks if vertical walls were necessary to determine normal and critical flow depths.

        Note that is function is run in the Check for Warnings function call.

        Returns:
            str: Warning string indicating which flows needed vertical walls
        """
        warning = ""
        if self.input_dict['calc_data']['Shape'] != 'cross-section':
            return warning  # No vertical walls needed; these shapes will grow or will be closed

        embedment_depth = self.update_determine_embedment_depth()
        max_elev_without_vertical_walls = self.compute_channel_geometry_parameters(embedment_depth)
        if max_elev_without_vertical_walls is None:
            return warning  # No vertical walls needed

        wall_for_normal = False
        wall_for_critical = False
        for wse in wses:
            if wse > max_elev_without_vertical_walls:
                wall_for_normal = True
        for wse in critical_wses:
            if wse > max_elev_without_vertical_walls:
                wall_for_critical = True
        if wall_for_normal:
            if wall_for_critical:
                warning = "Vertical walls were added for normal and critical depth computations"
            else:
                warning = "Vertical walls were added for normal depth computations"
        elif wall_for_critical:
            warning = "Vertical walls were added for critical depth computations"
        return warning  # No vertical walls needed

    def _get_can_compute(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): The variable that is unknown and being calculated

        Returns:
            bool: True if can compute
        """
        # This _get_can_compute function has an optional 'unknown' for the manning's n calc to use
        # It is intentionally different than the established pattern.
        result = True

        if unknown is None:
            if self.unknown is None:
                self.unknown = 'Head'
            unknown = self.unknown
        else:
            self.unknown = unknown

        # if 'Composite n computation method' in self.results:
        #     self.input_dict['calc_data']['Composite n'].results['Composite n computation method'] = \
        #         self.results['Composite n computation method']
        # if 'Composite n value' in self.results:
        #     self.input_dict['calc_data']['Composite n'].results['Composite n value'] = self.results[
        #         'Composite n value']

        if self.stand_alone_calc:
            found = False
            depths = self.input_dict['calc_data']['Depths']
            wses = self.input_dict['calc_data']['WSE']
            if self.input_dict['calc_data']['Head'] == 'Depth':
                if hasattr(depths, '__len__') and len(depths) < 1:
                    self.warnings['depths'] = "Please enter a depth"
                    result = False
                else:
                    for depth in depths:
                        if depth > 0.0:
                            found = True
                            break
                    if not found:
                        self.warnings['depths2'] = "Please enter a positive, non-zero depth"
                        result = False
            elif self.input_dict['calc_data']['Head'] == 'Elevation':
                if hasattr(wses, '__len__') and len(wses) < 1:
                    self.warnings['wse'] = "Please enter a water surface elevation"
                    result = False
                else:
                    for wse in wses:
                        if wse > 0.0:
                            found = True
                            break
                    if not found:
                        self.warnings['wse2'] = "Please enter a positive, non-zero water surface elevation"
                        result = False

        shape_result = self.get_can_compute_shape(unknown)

        self.can_compute_shape = result
        self.can_compute = result and shape_result

        return self.can_compute

    def get_can_compute_shape(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): The variable that is unknown and being calculated

        Returns:
            bool: True if can compute
        """
        result = True

        if self.input_dict['calc_data']['Shape'] == 'cross-section':
            pass
        elif self.input_dict['calc_data']['Shape'] == 'circle':
            if unknown != 'Diameter' and self.input_dict['calc_data']['Diameter'] <= 0.0:
                self.warnings['diameter'] = "Please enter the diameter"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'triangle':
            if unknown != 'Side slope' and self.input_dict['calc_data']['Side slope 1'] + \
                    self.input_dict['calc_data']['Side slope 2'] <= 0.0:
                self.warnings['side slope'] = "Please enter a side slope"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'round-bottomed triangle':
            if unknown != 'Side slope' and self.input_dict['calc_data']['Side slope'] <= 0.0:
                self.warnings['side slope'] = "Please enter a side slope"
                result = False
            if unknown != 'Corner radius' and self.input_dict['calc_data']['Corner radius'] <= 0.0:
                self.warnings['corner radius'] = "Please enter the corner radius"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'trapezoidal':
            if unknown != 'Side slope' and self.input_dict['calc_data']['Side slope 1'] + \
                    self.input_dict['calc_data']['Side slope 2'] <= 0.0:
                self.warnings['side slope'] = "Please enter a side slope"
                result = False
            if unknown != 'Bottom width' and self.input_dict['calc_data']['Bottom width'] <= 0.0:
                self.warnings['bottom width'] = "Please enter the bottom width"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'parabola':
            if unknown != 'Top width' and self.input_dict['calc_data']['Top width'] <= 0.0:
                self.warnings['top width'] = "Please enter the top width"
                result = False
            if unknown != 'Channel depth' and self.input_dict['calc_data']['Channel depth'] <= 0.0:
                self.warnings['channel depth'] = "Please enter the channel depth"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'rectangle':
            if unknown != 'Bottom width' and self.input_dict['calc_data']['Bottom width'] <= 0.0:
                self.warnings['bottom width'] = "Please enter the bottom width"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'box':
            if unknown != 'Rise' and self.input_dict['calc_data']['Rise'] <= 0.0:
                self.warnings['rise'] = "Please enter the rise"
                result = False
            if unknown != 'Span' and self.input_dict['calc_data']['Span'] <= 0.0:
                self.warnings['span'] = "Please enter the span"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'round-cornered rectangle':
            if unknown != 'Bottom width' and self.input_dict['calc_data']['Bottom width'] <= 0.0:
                self.warnings['bottom width'] = "Please enter the bottom width"
                result = False
            if unknown != 'Corner radius' and self.input_dict['calc_data']['Corner radius'] <= 0.0:
                self.warnings['corner radius'] = "Please enter the corner radius"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'curb and gutter':
            # Curb height, gutter width, and road width are not required for the calculation,
            # though, it may be required for the display of the curb and gutter, as well as reporting the data
            # gutter width could be zero, and we are just designing the roadway as a gutter

            # if unknown != 'Curb height' and self.input_dict['calc_data']['Curb height'] <= 0.0:
            #     self.warnings['curb height'] = "Please enter the curb height"
            #     result = False
            if unknown != 'Cross-slope pavement' and self.input_dict['calc_data']['Cross-slope pavement'] <= 0.0:
                self.warnings['cross-slope pavement'] = "Please enter the cross slope pavement"
                result = False
            if self.input_dict['calc_data']['Define gutter cross-slope'] and unknown != 'Cross-slope gutter' and \
                    self.input_dict['calc_data']['Cross-slope gutter'] <= 0.0:
                self.warnings['cross-slope gutter'] = "Please enter the cross slope gutter"
                result = False
            # if unknown != 'Gutter width' and self.input_dict['calc_data']['Gutter width'] <= 0.0:
            #     self.warnings['gutter width'] = "Please enter the gutter width"
            #     result = False
            # if unknown != 'Road width' and self.input_dict['calc_data']['Road width'] <= 0.0:
            #     self.warnings['road width'] = "Please enter the road width half"
            #     result = False
        elif self.input_dict['calc_data']['Shape'] == 'elliptical':
            if unknown != 'Rise' and self.input_dict['calc_data']['Rise'] <= 0.0:
                self.warnings['rise'] = "Please enter the rise of the elliptical channel"
                result = False
            if unknown != 'Span' and self.input_dict['calc_data']['Span'] <= 0.0:
                self.warnings['span'] = "Please enter the span of the elliptical channel"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'pipe arch':
            if self.input_dict['calc_data']['Rise'] <= 0.0:
                self.warnings['rise'] = "Please enter the rise of the pipe arch"
                result = False
            if self.input_dict['calc_data']['Span'] <= 0.0:
                self.warnings['span'] = "Please enter the span of the pipe arch"
                result = False
            if self.input_dict['calc_data']['Top radius'] <= 0.0:
                self.warnings['top radius'] = "Please enter the top radius of the pipe arch"
                result = False
            if self.input_dict['calc_data']['Corner radius'] <= 0.0:
                self.warnings['corner radius'] = "Please enter the corner radius of the pipe arch"
                result = False
            if self.input_dict['calc_data']['Bottom radius'] <= 0.0:
                self.warnings['bottom radius'] = "Please enter the bottom radius of the pipe arch"
                result = False
            if self.input_dict['calc_data']['b'] <= 0.0:
                self.warnings['b'] = "Please enter the b of the pipe arch"
                result = False
        elif self.input_dict['calc_data']['Shape'] == 'horseshoe':
            if unknown != 'Rise' and self.input_dict['calc_data']['Rise'] <= 0.0:
                self.warnings['rise'] = "Please enter the rise of the horseshoe channel"
                result = False
        # elif self.input_dict['calc_data']['Shape'] == 'oblong':
        #     if self.input_dict['calc_data']['Corner radius'] <= 0.0:
        #         self.warnings['corner radius'] = "Please enter the corner radius"
        #         result = False
        #     if unknown != 'Oblong width' and self.input_dict['calc_data']['Oblong width'] <= 0.0:
        #         self.warnings['oblong width'] = "Please enter the oblong width"
        #         result = False

        self.can_compute_shape = result
        return result

    def _check_elevation_to_max_depth(self, wse):
        """Check if an elevation is above the maximum depth.

        Args:
            wse (float): The elevation to check
        """
        if self.low_elev is None or self.low_elev == self.high_elev:
            self.low_elev, self.high_elev = self.determine_low_and_high_of_elevations()
        # if self.shape_grows_no_channel_depth_needed():
        max_depth = self.high_elev - self.low_elev

        wse_depth = wse - self.low_elev
        if wse_depth > max_depth:
            if not self.shape_is_closed:
                # Add vertical walls
                new_high_elevation = wse_depth * 2.0 + self.low_elev
                self.add_vertical_wall_to_elevation(new_high_elevation)
                self.low_elev, self.high_elev = self.determine_low_and_high_of_elevations()
            else:
                self.warnings['max_depth'] = "The water surface elevation is above the maximum depth of the channel." \
                    " Please enter a lower water surface elevation."
                wse = self.high_elev
        return wse

    def determine_low_and_high_of_elevations(self):
        """Determines the low and and high of a given set of elevations.

        Returns:
            tuple (float, float): See description
        """
        if not self.elevations or len(self.elevations) < 1:
            return 0.0, self.rise_to_crown
        self.low_elev = min(self.elevations)
        if self.shape_is_closed:
            self.high_elev = max(self.elevations)
        else:
            if self.left_vertical_elevation:
                self.high_elev = self.left_vertical_elevation
                return self.low_elev, self.high_elev
            elif self.right_vertical_elevation:
                self.high_elev = self.right_vertical_elevation
                return self.low_elev, self.high_elev
            if self.input_dict['calc_data']['Only fill flow areas with thalweg locations']:
                low_index = self.elevations.index(self.low_elev)
                left_high = max(self.elevations[:low_index])
                right_high = max(self.elevations[low_index:])
            else:
                left_high = self.elevations[0]
                right_high = self.elevations[-1]
            self.high_elev = min(left_high, right_high)
        return self.low_elev, self.high_elev

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.results['Wetted perimeter'] = []
        self.results['Flow area'] = []
        self.results['Top width'] = []
        self.results['Gutter depression'] = []
        self.results['Eo'] = []
        self.results['Section factor'] = []
        self.results['Hydraulic radius'] = []
        self.results['Hydraulic depth'] = []
        self.results['Equivalent depth'] = []

        self.results['Horton n'] = []
        self.results['Pavlovskii n'] = []
        self.results['Lotter n'] = []

        # if 'Composite n computation method' in self.results:
        #     self.results['Composite n computation method'] = []
        # if 'Composite n value' in self.results:
        #     self.results['Composite n value'] = []

        self.wse_stations = []
        self.wse_elevations = []

        self.channel_stations = []
        self.channel_elevations = []

        self.embedment_stations = []
        self.embedment_elevations = []

        self.stations = []
        self.elevations = []

        self.flow_stations = []
        self.flow_elevations = []

        self.thalweg_elevation = 0.0
        self.thalweg_station = 0.0
        self.crown_elevation = 0.0
        self.crown_station = 0.0
        self.left_max_bank = 0.0
        self.left_max_bank_station = 0.0
        self.right_max_bank = 0.0
        self.right_max_bank_station = 0.0

        self.left_vertical_station = None
        self.left_vertical_elevation = None
        self.right_vertical_station = None
        self.right_vertical_elevation = None

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        if self.stand_alone_calc:
            self.clear_results()

        self.initialize_geometry()

        if not self.can_compute_shape:
            return False

        self._compute_data_after_initializations()

    def set_plotting_data(self, index):
        """Sets the plotting data for the cross-section.

        Args:
            index (int): The index of the cross-section to plot.
        """
        if 'Cross-section' not in self.plot_dict:
            return
        flow_stations = self.flow_stations
        flow_elevations = self.flow_elevations
        if index >= len(flow_stations) or index == -1 and len(flow_stations) > 0:
            self.plot_dict['Cross-section']['series'][0]['x var'].set_val(flow_stations[index])
            self.plot_dict['Cross-section']['series'][0]['y var'].set_val(flow_elevations[index])

        self.plot_dict['Cross-section']['series'][1]['x var'].set_val(self.embedment_stations)
        self.plot_dict['Cross-section']['series'][1]['y var'].set_val(self.embedment_elevations)

        self.plot_dict['Cross-section']['series'][2]['x var'].set_val(self.channel_stations)
        self.plot_dict['Cross-section']['series'][2]['y var'].set_val(self.channel_elevations)
        self.plot_dict['Cross-section']['series'][2]['Fill color'] = None

    def _compute_data_after_initializations(self):
        """Computes the data after initializations; stores results in self."""
        # Backup settings before we change them for this calculation
        depth_original = copy.copy(self.input_dict['calc_data']['Depths'])

        if not self.can_compute:
            return False

        # Convert depths/elevations
        if self.input_dict['calc_data']['Head'] == 'Elevation':
            depths = []
            for wse in self.input_dict['calc_data']['WSE']:
                depths.append(wse - self.low_elev)
            self.input_dict['calc_data']['Depths'] = depths
        else:
            wses = []
            for depth in self.input_dict['calc_data']['Depths']:
                wses.append(depth + self.low_elev)
            self.input_dict['calc_data']['WSE'] = wses

        # Compute WSE values
        for wse in self.input_dict['calc_data']['WSE']:
            self.compute_channel_geometry(wse)

        depth = 0.0
        top_width = 0.0
        area = 0.0
        perimeter = 0.0
        if len(self.input_dict['calc_data']['Depths']):
            depth = self.input_dict['calc_data']['Depths'][-1]

        if 'Flow area' not in self.results:
            self.results['Flow area'] = []
        if len(self.results['Flow area']):
            area = self.results['Flow area'][-1]

        if 'Wetted perimeter' not in self.results:
            self.results['Wetted perimeter'] = []
        if len(self.results['Wetted perimeter']):
            perimeter = self.results['Wetted perimeter'][-1]

        if 'Top width' not in self.results:
            self.results['Top width'] = []
        if len(self.results['Top width']):
            top_width = self.results['Top width'][-1]

        top_width = self.return_governing_top_width_or_equivalent_depth(
            depth, top_width, area)

        if 'Hydraulic radius' not in self.results:
            self.results['Hydraulic radius'] = []
        if perimeter > 0.0:
            self.results['Hydraulic radius'].append(area / perimeter)
        else:
            self.results['Hydraulic radius'].append(0.0)

        if 'Hydraulic depth' not in self.results:
            self.results['Hydraulic depth'] = []
        if top_width > 0.0:
            hydraulic_depth = area / top_width
            self.results['Hydraulic depth'].append(hydraulic_depth)
        else:
            self.results['Hydraulic depth'].append(0.0)

        # Restore settings now that it is complete
        self.input_dict['calc_data']['Depths'] = depth_original

        return True

    def compute_channel_geometry(self, wse):
        """
        Computes the channel geometry based on the current shape and elevation data.
        """
        if self.input_dict['calc_data']['Shape'] in self.shapes_to_compute_geom or \
                (self.channel_is_embedded and not self.embedment_is_evenly_across):
            self._compute_cross_section_geometry_from_elevation(wse)
            self.append_flow_polys()

        else:
            new_wse = copy.copy(wse)
            if self.channel_is_embedded and self.embedment_is_evenly_across:
                new_wse -= self.min_embedment_depth
            self._compute_area(new_wse)
            self._compute_perimeter(new_wse, True)
            self._compute_top_width(new_wse)
            self._compute_section_factor(new_wse)
            if self.input_dict['calc_data']['Compute cross-section']:
                self.compute_cross_sections_for_shapes(wse)
            # self.append_flow_polys()

    def append_flow_polys(self):
        """Appends the current flow polygons to the respective lists."""
        self.flow_stations.append(self.cur_flow_stations)
        self.flow_elevations.append(self.cur_flow_elevations)

        self.wse_stations.append(self.cur_wse_stations)
        self.wse_elevations.append(self.cur_wse_elevations)

        self.ineffective_flow_poly_stations.append(self.cur_ineffective_flow_poly_stations)
        self.ineffective_flow_poly_elevations.append(self.cur_ineffective_flow_poly_elevations)

    def compute_wse(self, wse, top_width, velocity, channel_x, channel_y, froude=None):
        """Computes the water surface elevation for a given depth.

        Args:
            depth (float): The depth to compute the water surface elevation for
            top_width (float): The top width to use for the computation (optional)
            velocity (float): The flow velocity to use for the computation
            channel_x (list): The x-coordinates of the channel
            channel_y (list): The y-coordinates of the channel

        Returns:
            float: The computed water surface elevation
        """
        delta_y, transition_length = self.compute_super_elevations(top_width, velocity, froude)
        if self.input_dict['calc_data']['Curvature direction'] == 'Left':
            wse_y = [wse - delta_y / 2.0, wse + delta_y / 2.0]
        else:
            wse_y = [wse + delta_y / 2.0, wse - delta_y / 2.0]

        _, null_data = self.get_data('Null data', -999.0)
        _, zero_tol = self.get_data('Zero tolerance', 1e-6)
        channel_interp = Interpolation(channel_y, channel_x, null_data=null_data, zero_tol=zero_tol)
        first_station, _ = channel_interp.interpolate_y(wse_y[0])
        rev_channel_interp = Interpolation(list(reversed(channel_y)), list(reversed(channel_x)), null_data=null_data,
                                           zero_tol=zero_tol)
        last_station, _ = rev_channel_interp.interpolate_y(wse_y[-1])
        wse_x = [first_station, last_station]

        if 'Super elevation' not in self.results:
            self.results['Super elevation'] = {}
        if 'Depth change' not in self.results['Super elevation']:
            self.results['Super elevation']['Depth change'] = []
        self.results['Super elevation']['Depth change'].append(delta_y)

        if 'Transition length' not in self.results['Super elevation']:
            self.results['Super elevation']['Transition length'] = []
        self.results['Super elevation']['Transition length'].append(transition_length)

        return wse_x, wse_y, transition_length

    def compute_super_elevations(self, top_width, velocity, froude=None):
        """Computes the super elevations for the channel.

        Args:
            froude (float): The Froude number
            top_width (float): The top width of the channel
            velocity (float): The flow velocity

        Returns:
            tuple: (delta_y, transition_length) where delta_y is the super elevation and transition_length is the
            length over which the super elevation occurs.
        """
        _, g = self.get_data('Gravity', 32.2)
        radius_of_curvature = self.input_dict['calc_data']['Radius of curvature']
        delta_y = top_width * velocity ** 2.0 / (g * radius_of_curvature)
        if froude is None or froude >= 1.0:
            delta_y *= 2.0
        # Alternative formula
        # delta_y = unit_q ** 2.0 / (g * radius_of_curvature * flow_depth)
        transition_length = 15.0 * velocity ** 2.0 * top_width / (g * radius_of_curvature)
        return delta_y, transition_length

    def determine_if_shape_is_closed(self):
        """Determines if the shape is closed and the rise to the crown.

        Returns:
            shape_is_closed (bool): True if the shape is closed
        """
        # _, zero_tol = self.get_setting('Zero tolerance')
        self.shape_is_closed = False
        shape = self.input_dict['calc_data']['Shape']
        # embedment_depth = 0.0
        # Closed shapes by definition
        if shape in self.closed_shapes:
            if self.channel_is_embedded or shape == 'cross-section':
                # embedment_depth = self.input_dict['calc_data']['Embedment']['Embedment depth']
                stations = self.stations
                elevations = self.elevations
                if len(stations) and len(elevations):
                    if math.isclose(stations[0], stations[-1], abs_tol=self.zero_tol) and \
                            math.isclose(elevations[0], elevations[-1], abs_tol=self.zero_tol):
                        self.shape_is_closed = True
            else:
                self.shape_is_closed = True

        return self.shape_is_closed

    def compute_area_perimeter_top_width(self, depth):
        """Computes the flow area, wetted perimeter, and top width for any shape given a depth.

        Args:
            depth (float): specified depth

        Returns:
            flow_area (float): cross-sectional flow area
            wetted_perimeter (float): wetted perimeter
            top_width (float): top width
        """
        flow_area = 0.0
        wetted_perimeter = 0.0
        top_width = 0.0

        shapes_computed_without_cross_section = [
            'circle', 'triangle', 'round-bottomed triangle', 'trapezoidal',
            'parabola', 'rectangle', 'box', 'round-cornered rectangle'
        ]

        if self.input_dict['calc_data']['Shape'] not in shapes_computed_without_cross_section or \
                self.channel_is_embedded:

            if len(self.elevations) == 0 and not self.input_dict['calc_data']['Shape'] == 'cross-section':
                self.compute_cross_sections_for_shapes(depth)
            if self.low_elev is not None or self.low_elev == self.high_elev:
                self.initialize_geometry()
            flow_area, wetted_perimeter, top_width = self._compute_cross_section_geometry_from_elevation(
                depth + self.low_elev + self.min_embedment_depth)
        else:
            flow_area = self._compute_area(depth)
            wetted_perimeter = self._compute_perimeter(depth)
            top_width = self._compute_top_width(depth)

        return flow_area, wetted_perimeter, top_width

    def _compute_area(self, depth):
        """Compute the cross-sectional flow area of the geometry, given a depth.

        Args:
            depth (float): specified depth
        """
        area = 0.0
        if 'Flow area' not in self.results:
            self.results['Flow area'] = []
        y = depth
        if y < self.zero_tol:
            self.results['Flow area'].append(area)
            return area
        # If we have a closed shape, don't compute anything above the crown
        if self.shape_is_closed:
            if y > self.rise_to_crown:
                y = self.rise_to_crown

        if self.input_dict['calc_data']['Shape'] == 'circle':
            diameter = self.input_dict['calc_data']['Diameter']
            self.input_dict['calc_data']['Rise'] = diameter
            self.input_dict['calc_data']['Span'] = diameter
            if y > diameter:
                y = diameter
            r = diameter / 2.0
            self.radius = r
            intermediate_step = 0.0
            if r > 0.0:
                intermediate_step = (r - y) / r
            if intermediate_step <= -1.0:
                intermediate_step = -1.0
            self.theta = 2.0 * math.acos(intermediate_step)
            area = 1.0 / 8.0 * (self.theta - math.sin(self.theta)) * diameter**2

        elif self.input_dict['calc_data']['Shape'] == 'triangle':
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            area = 0.5 * (y * z1 + y * z2) * y

        elif self.input_dict['calc_data']['Shape'] == 'round-bottomed triangle':
            r = self.input_dict['calc_data']['Corner radius']
            z = self.input_dict['calc_data']['Side slope']
            _, pi = self.get_data('Pi')
            # from colalg.math.cxusb.edu, accessed on July 10, 2020
            # cot -1(x) = pi/2 - tan -1(x)
            # https://colalg.math.csusb.edu/~devel/IT/main/m06_inverse/src/s02_tanflip.html
            self.acot_z = pi / 2.0 - math.atan(z)
            area = (2 * (z * (y - r) + r * (1 + z**2)**0.5))**2 / (4 * z) - r**2 / z * (1 - z * self.acot_z)

        elif self.input_dict['calc_data']['Shape'] == 'trapezoidal':
            b = self.input_dict['calc_data']['Bottom width']
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            area = (b + (z1 * y + z2 * y) * 0.5) * y

        elif self.input_dict['calc_data']['Shape'] == 'parabola':
            # equation of a parabola From web.nmsu.edu on July 9, 2020
            # https://web.nmsu.edu/~pbaggett/ebookofcalculus/HandsOnUnitsInvolvingIntegrals/AreaUnderParabola/AreaUnderParabolaWebPage01.html
            # equation of parabola:
            # Y = H - (H/R^2)*X^2
            # Where H = channel depth
            # and R = half of the Top width
            # X = SQRT((H - Y) / (H/R^2))
            top_width_of_channel = self.input_dict['calc_data']['Top width']
            r = top_width_of_channel / 2.0
            h = self.input_dict['calc_data']['Channel depth']
            y_from_base = h - y
            x = ((h - y_from_base) / (h / r**2.0))**0.5
            self.top_width_of_water = 2.0 * x
            area = 2.0 / 3.0 * self.top_width_of_water * y

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            b = self.input_dict['calc_data']['Bottom width']
            if self.input_dict['calc_data']['Shape'] in ['box']:
                b = self.input_dict['calc_data']['Span']
                rise = self.input_dict['calc_data']['Rise']
                if y > rise:
                    y = rise
            area = b * y

        elif self.input_dict['calc_data']['Shape'] == 'round-cornered rectangle':
            b = self.input_dict['calc_data']['Bottom width']
            r = self.input_dict['calc_data']['Corner radius']
            _, pi = self.get_data('Pi')
            area = (pi / 2 - 2) * r**2 + (b + 2 * r) * y

        else:
            if self.stations is None or len(self.stations) < 2:
                self.compute_cross_sections_for_shapes(depth)
            area = 0.0
            wse1 = depth + self.low_elev
            wse2 = depth + self.low_elev
            for i in range(1, len(self.stations)):
                # both points below water
                if (self.elevations[i - 1] < wse1) and (self.elevations[i] < wse2):
                    trap_area, _ = self._compute_trapezoidal_flow_area(
                        self.stations[i - 1], self.stations[i], self.elevations[i - 1], self.elevations[i], wse1,
                        wse2)
                    area += trap_area
                # one point dry, one point wet -or- one point at water level and other point is wet
                elif (self.elevations[i - 1] < wse1) or (self.elevations[i] < wse2) or \
                        (self.elevations[i - 1] == wse1) and (self.elevations[i] < wse2):
                    tri_area, _, _, _, _ = self._compute_triangular_flow_area(
                        self.stations[i - 1], self.stations[i], self.elevations[i - 1], self.elevations[i], wse1, wse2)

                    # Update current polygon area, perimeter, and top width
                    area += tri_area

        self.results['Flow area'].append(area)
        return area

    def _compute_perimeter(self, depth, compute_hydraulic_radius=False):
        """Compute the wetted perimeter of the geometry, given a depth.

        Args:
            depth (float): The depth at which to compute the wetted perimeter.
            compute_hydraulic_radius (bool): Whether to compute the hydraulic radius.

        Returns:
            float: The wetted perimeter of the geometry.
        """
        wetted_perimeter = 0.0
        if 'Wetted perimeter' not in self.results:
            self.results['Wetted perimeter'] = []
        if compute_hydraulic_radius and 'Hydraulic radius' not in self.results:
            self.results['Hydraulic radius'] = []
        y = depth
        if y < self.zero_tol:
            self.results['Wetted perimeter'].append(wetted_perimeter)
            if compute_hydraulic_radius:
                self.results['Hydraulic radius'].append(0.0)
            return wetted_perimeter
        # If we have a closed shape, don't compute anything above the crown
        if self.shape_is_closed:
            if y > self.rise_to_crown:
                y = self.rise_to_crown

        if self.input_dict['calc_data']['Shape'] == 'circle':
            diameter = self.input_dict['calc_data']['Diameter']
            wetted_perimeter = 0.5 * self.theta * diameter

        elif self.input_dict['calc_data']['Shape'] == 'triangle':
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            wetted_perimeter = 2 * y * (1 + z1 * z2)**0.5

        elif self.input_dict['calc_data']['Shape'] == 'round-bottomed triangle':
            r = self.input_dict['calc_data']['Corner radius']
            z = self.input_dict['calc_data']['Side slope']

            wetted_perimeter = (2 * (z * (y - r) + r * (1 + z**2)**0.5)) / z * (
                (1 + z**2) ** 0.5) - 2 * r / z * (1 - z * self.acot_z)

        elif self.input_dict['calc_data']['Shape'] == 'trapezoidal':
            b = self.input_dict['calc_data']['Bottom width']
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            wetted_perimeter = b + 2 * y * (1 + z1 * z2)**0.5

        elif self.input_dict['calc_data']['Shape'] == 'parabola':
            # equation of parabola:
            # Y = H - (H/R^2)*X^2
            # Where H = channel depth
            # and R = half of the Top width
            # X = SQRT((H - Y) / (H/R^2))
            # top_width_of_channel = self.input['top_width'].get_val()
            # r = top_width_of_channel / 2.0
            # h = self.input['channel_depth'].get_val()
            top_width = self.top_width_of_water
            x = 4.0 * y / top_width
            wetted_perimeter = top_width / 2.0 * ((1 + x**2)**0.5 + 1 / x * math.log(x + (1 + x**2)**0.5))

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            b = self.input_dict['calc_data']['Bottom width']
            rise = self.input_dict['calc_data']['Rise']
            if self.input_dict['calc_data']['Shape'] in ['box']:
                b = self.input_dict['calc_data']['Span']
                if y > rise:
                    y = rise
            wetted_perimeter = b + 2 * y
            if self.input_dict['calc_data']['Shape'] in ['box']:
                if y >= rise - self.zero_tol:
                    wetted_perimeter = 2 * (y + b)

        elif self.input_dict['calc_data']['Shape'] == 'round-cornered rectangle':
            b = self.input_dict['calc_data']['Bottom width']
            r = self.input_dict['calc_data']['Corner radius']
            _, pi = self.get_data('Pi')
            wetted_perimeter = (pi - 2) * r + b + 2 * y

        self.results['Wetted perimeter'].append(wetted_perimeter)

        if compute_hydraulic_radius:
            hydraulic_radius = 0.0
            if wetted_perimeter > 0.0:
                flow_area = self.results['Flow area'][-1] if 'Flow area' in self.results else 0.0
                hydraulic_radius = flow_area / wetted_perimeter
            self.results['Hydraulic radius'].append(hydraulic_radius)
        return wetted_perimeter

    def _compute_top_width(self, depth, append_to_results=True):
        """Compute the top width of the geometry, given a depth.

        Args:
            depth (float): specified depth

        Returns:
            top_width (float):
        """
        top_width = 0.0
        if append_to_results:
            if 'Top width' not in self.results:
                self.results['Top width'] = []
        y = depth
        if y < self.zero_tol:
            if append_to_results:
                self.results['Top width'].append(top_width)
            return top_width
        # If we have a closed shape, don't compute anything above the crown
        if self.shape_is_closed:
            if y > self.rise_to_crown:
                y = self.rise_to_crown

        if self.input_dict['calc_data']['Shape'] == 'circle':
            d = self.input_dict['calc_data']['Diameter']
            top_width = d * math.sin(self.theta * 0.5)

        elif self.input_dict['calc_data']['Shape'] == 'triangle':
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            top_width = y * z1 + y * z2

        elif self.input_dict['calc_data']['Shape'] == 'round-bottomed triangle':
            r = self.input_dict['calc_data']['Corner radius']
            z = self.input_dict['calc_data']['Side slope']
            top_width = 2 * (z * (y - r) + r * (1 + z**2)**0.5)
            # self.results['top_width'].add_result(top_width)

            # This will give in bad results when the depth is less than the corner radius
            # TODO: add logic to handle depth < corner radius

        elif self.input_dict['calc_data']['Shape'] == 'trapezoidal':
            b = self.input_dict['calc_data']['Bottom width']
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            top_width = b + y * z1 + y * z2

        elif self.input_dict['calc_data']['Shape'] == 'parabola':
            area = self.results['Flow area'][-1]
            top_width = 3.0 * area / (2.0 * y)
            # equation of parabola:
            # Y = H - (H/R^2)*X^2
            # Where H = channel depth
            # and R = half of the Top width
            # X = SQRT((H - Y) / (H/R^2))

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            b = self.input_dict['calc_data']['Bottom width']
            if self.input_dict['calc_data']['Shape'] in ['box']:
                b = self.input_dict['calc_data']['Span']
            top_width = b

        elif self.input_dict['calc_data']['Shape'] == 'round-cornered rectangle':
            b = self.input_dict['calc_data']['Bottom width']
            r = self.input_dict['calc_data']['Corner radius']
            top_width = b + 2 * r
            # This will give in bad results when the depth is less than the corner radius
            # TODO: add logic to handle depth < corner radius

        if append_to_results:
            self.results['Top width'].append(top_width)
        return top_width

    def return_governing_top_width_or_equivalent_depth(self, depth, top_width, flow_area):
        """Returns the governing top width or equivalent depth for the geometry.

        Args:
            depth (float): specified depth
            top_width (float): top width
            flow_area (float): cross-sectional flow area

        Returns:
            governing_top_width (float): governing top width
        """
        governing_top_width = top_width
        _, use_depth_e = self.get_data('Use equivalent depth', 'When depth > rise / 2')
        rise = self.input_dict['calc_data']['Rise']
        span = self.input_dict['calc_data']['Span']
        if self._need_to_compute_equivalent_depth(depth, use_depth_e,
                                                  top_width, rise, span):
            equivalent_depth = self.compute_equivalent_depth(flow_area)
            equivalent_top_width = 0.0
            if equivalent_depth > 0.0:
                equivalent_top_width = flow_area / equivalent_depth
            if equivalent_top_width > top_width:
                governing_top_width = equivalent_top_width
        if use_depth_e == 'use hydraulic depth < rise':
            if top_width == 0.0 or flow_area / top_width > rise:
                governing_top_width = flow_area / rise
        return governing_top_width

    def _need_to_compute_equivalent_depth(self, depth, use_depth_e, top_width, rise, span):
        """Returns true if the equivalent depth should be computed.

        Args:
            depth (float): specified depth
            use_depth_e (string): use equivalent depth setting
            top_width (float): top width
            rise (float): rise
            span (float): span

        Returns:
            bool: True if the equivalent depth should be computed
        """
        shape = self.input_dict['calc_data']['Shape']
        if shape in self.closed_shapes:
            if use_depth_e == 'For shapes other than box' and shape not in ['rectangle', 'box']:
                return True
            if top_width < span / 2.0:
                if use_depth_e == 'Always':
                    return True
                elif use_depth_e == 'When depth > rise / 2':
                    if depth > rise / 2.0:
                        return True

        return False

    def compute_equivalent_depth(self, area):
        """Use Hager method for estimating top width.

        Args:
            area (float): cross-sectional flow area

        Returns:
            equivalent_depth (float): equivalent depth
        """
        # From Hager 1992; NDOR Research Project Number SPR-PL-1(33) P498
        # Hydraulic Analysis of Broken-Back Culverts
        equivalent_depth = math.sqrt(area / 2.0)
        return equivalent_depth

    def compute_cross_sections_for_shapes(self, depth, initializing=False):
        """Compute the cross-section from shape specification as well as the normal depth stations.

        Args:
            depth (float): specified Normal depth
            initializing (bool): True if this is called from initialize_geometry
        """
        # Compute plotting geometry?
        y = depth
        if self.input_dict['calc_data']['Shape'] not in ['cross-section']:
            self.stations = []
            self.elevations = []
        wse_stations = []
        wse_elevations = []
        flow_stations = []
        flow_elevations = []

        # If we have a closed shape, don't compute anything above the crown
        if self.shape_is_closed:
            if y > self.rise_to_crown:
                y = self.rise_to_crown

        # take one off of the number of points as we divide it, it will have that number of points
        _, num_points = self.get_data('Number of points computed for shapes', 40)

        # if the user provided us a depth of channel, use that; otherwise 125 percent of the depth for visualizing
        channel_depth = self.input_dict['calc_data']['Channel depth']
        if channel_depth <= self.zero_tol:
            if y <= self.zero_tol:
                channel_depth = 1.0  # Gotta have something... so just assume 1.0
            else:
                channel_depth = 1.25 * y

        self.high_elev = 0.0
        self.low_elev = self.input_dict['calc_data']['Thalweg invert elevation']

        # used to determine the stations of the WSE
        # and updated in some cross-section calculations
        top_width = None

        if self.input_dict['calc_data']['Shape'] in ['circle', 'elliptical']:
            d = self.input_dict['calc_data']['Diameter']
            vertical_radius = d / 2.0
            horizontal_radius = d / 2.0
            if self.input_dict['calc_data']['Shape'] == 'elliptical':
                horizontal_radius = self.input_dict['calc_data']['Span'] / 2.0
                vertical_radius = self.input_dict['calc_data']['Rise'] / 2.0
                d = self.input_dict['calc_data']['Rise']
            self.high_elev = self.low_elev + d
            if self.input_dict['calc_data']['Channel depth'] <= 0.0:
                self.input_dict['calc_data']['Channel depth'] = d
            arc_increment = 360.0 / num_points
            cur_angle = 360.0
            while cur_angle >= 0.0:
                self.stations.append(horizontal_radius
                                     * math.sin(math.radians(cur_angle)))
                self.elevations.append(vertical_radius
                                       * math.cos(math.radians(cur_angle)))
                cur_angle -= arc_increment
            self.stations.append(self.stations[0])
            self.elevations.append(self.elevations[0])
            # TODO: See if we can reliably determine that the top width was already computed and available in results
            # It is easy to grab it, but hard to know that it relates to the current depth
            top_width = self._compute_top_width(y, append_to_results=False)
            wse_stations = [-top_width / 2.0, top_width / 2.0]

        elif self.input_dict['calc_data']['Shape'] == 'triangle':
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            # top left corner
            self.stations.append(-z1 * channel_depth)
            self.elevations.append(channel_depth)
            wse_stations.append(-z1 * y)
            # bottom point
            self.stations.append(0.0)
            self.elevations.append(0.0)
            # top right corner
            self.stations.append(z2 * channel_depth)
            self.elevations.append(channel_depth)
            wse_stations.append(z2 * y)
            self.high_elev = self.low_elev + channel_depth

        elif self.input_dict['calc_data']['Shape'] == 'round-bottomed triangle':
            r = self.input_dict['calc_data']['Corner radius']
            z = self.input_dict['calc_data']['Side slope']

            num_points -= 2
            alpha_in_radians = -math.atan(r / ((1 + z**2)**0.5))
            corner_x = r * math.sin(alpha_in_radians)
            corner_y = r * math.cos(alpha_in_radians)
            # top left corner
            self.stations.append(-z * (channel_depth - corner_y) + corner_x)
            self.elevations.append(channel_depth)
            wse_stations.append(-z * (y - corner_y) + corner_x)
            # Determine the rounded bottom
            cur_angle = alpha_in_radians
            alpha_increment = abs(alpha_in_radians) / (num_points / 2.0)
            while cur_angle <= abs(alpha_in_radians):
                self.stations.append(r * math.sin(cur_angle))
                self.elevations.append(r - r * math.cos(cur_angle))
                cur_angle += alpha_increment
            # Top right corner
            corner_x = abs(corner_x)  # change the sign
            self.stations.append(z * (channel_depth - corner_y) + corner_x)
            self.elevations.append(channel_depth)
            wse_stations.append(z * (y - corner_y) + corner_x)
            self.high_elev = self.low_elev + channel_depth

            # This will give in bad results when the depth is less than the corner radius

        elif self.input_dict['calc_data']['Shape'] == 'trapezoidal':
            b = self.input_dict['calc_data']['Bottom width']
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            self.stations.append(-z1 * channel_depth - b / 2.0)
            self.elevations.append(channel_depth)
            wse_stations.append(-z1 * y - b / 2.0)
            self.stations.append(-b / 2.0)
            self.elevations.append(0.0)
            self.stations.append(b / 2.0)
            self.elevations.append(0.0)
            self.stations.append(z2 * channel_depth + b / 2.0)
            self.elevations.append(channel_depth)
            wse_stations.append(z2 * y + b / 2.0)
            self.high_elev = self.low_elev + channel_depth

        elif self.input_dict['calc_data']['Shape'] == 'parabola':
            # equation of parabola:
            # Y = H - (H/R^2)*X^2
            # Where H = channel depth
            # and R = half of the Top width
            # X = SQRT((H - Y) / (H/R^2))
            # t = self.top_width_of_water
            r = self.input_dict['calc_data']['Top width'] / 2.0
            if r <= 0.0:
                return
            x_increment = r / (num_points / 2.0)
            cur_location = -r
            while cur_location <= r:
                self.stations.append(cur_location)
                self.elevations.append(channel_depth - (channel_depth - (channel_depth / (r**2)) * cur_location**2))
                cur_location += x_increment
            if self.stations[-1] <= r:
                self.stations.append(cur_location)
                self.elevations.append(channel_depth)
            # Needs hydraulic computations performed before this can be determined
            if len(self.results['Top width']) > 0:
                top_width = self.results['Top width'][-1]
                wse_stations = [-top_width / 2.0, top_width / 2.0]
            self.high_elev = self.low_elev + channel_depth

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            b = self.input_dict['calc_data']['Bottom width']
            rise = self.input_dict['calc_data']['Rise']
            if self.input_dict['calc_data']['Shape'] in ['box']:
                b = self.input_dict['calc_data']['Span']
                if y > rise:
                    y = rise
                channel_depth = rise
            # determine top left corner
            self.stations.append(-b / 2.0)
            self.elevations.append(channel_depth)
            wse_stations.append(-b / 2.0)
            # determine bottom left corner
            self.stations.append(-b / 2.0)
            self.elevations.append(0.0)
            # determine bottom right corner
            self.stations.append(b / 2.0)
            self.elevations.append(0.0)
            # determine top right corner
            self.stations.append(b / 2.0)
            self.elevations.append(channel_depth)
            wse_stations.append(b / 2.0)
            if self.input_dict['calc_data']['Shape'] in ['box']:
                self.stations.append(-b / 2.0)
                self.elevations.append(channel_depth)
            self.high_elev = self.low_elev + channel_depth

        elif self.input_dict['calc_data']['Shape'] == 'round-cornered rectangle':
            b = self.input_dict['calc_data']['Bottom width']
            r = self.input_dict['calc_data']['Corner radius']

            # This will give in bad results when the depth is less than the corner radius
            num_points -= 2
            num_points /= 2.0

            self.stations = []
            self.elevations = []

            # determine top left corner
            self.stations.append(-b / 2.0 - r)
            self.elevations.append(channel_depth)
            wse_stations.append(-b / 2.0 - r)
            # determine bottom left corner
            angle_increment = 90.0 / num_points
            cur_angle = 270.0
            point_of_curvature_x = -b / 2.0
            point_of_curvature_y = r
            while cur_angle >= 180:
                self.stations.append(r * math.sin(math.radians(cur_angle)) + point_of_curvature_x)
                self.elevations.append(r * math.cos(math.radians(cur_angle)) + point_of_curvature_y)
                cur_angle -= angle_increment
            # determine bottom right corner
            cur_angle = 180.0
            point_of_curvature_x = b / 2.0
            while cur_angle >= 90:
                self.stations.append(r * math.sin(math.radians(cur_angle)) + point_of_curvature_x)
                self.elevations.append(r * math.cos(math.radians(cur_angle)) + point_of_curvature_y)
                cur_angle -= angle_increment
            # determine top right corner
            self.stations.append(b / 2.0 + r)
            self.elevations.append(channel_depth)
            wse_stations.append(b / 2.0 + r)
            self.high_elev = self.low_elev + channel_depth

        elif self.input_dict['calc_data']['Shape'] == 'curb and gutter':
            curb_height = self.input_dict['calc_data']['Curb height']
            if curb_height <= self.zero_tol:
                curb_height = 0.5  # just assume 6" curb
            cross_slope_pavement = self.input_dict['calc_data']['Cross-slope pavement']
            cross_slope_gutter = self.input_dict['calc_data']['Cross-slope pavement']
            if self.input_dict['calc_data']['Define gutter cross-slope']:
                cross_slope_gutter = self.input_dict['calc_data']['Cross-slope gutter']
            gutter_width = self.input_dict['calc_data']['Gutter width']
            road_width = self.input_dict['calc_data']['Road width']
            if road_width <= self.zero_tol:
                road_width = 12.0  # just assume 12' lane
            # Start at curb face and layout the curb section
            self.stations.append(0.0)
            self.elevations.append(curb_height)
            self.stations.append(0.0)
            self.elevations.append(0.0)
            #  assuming widths are horizontal and not at slope
            self.stations.append(gutter_width)
            self.elevations.append(gutter_width * cross_slope_gutter)
            self.stations.append(gutter_width + road_width)
            self.elevations.append(gutter_width * cross_slope_gutter + road_width * cross_slope_pavement)

        elif self.input_dict['calc_data']['Shape'] == 'pipe arch':
            rise = self.input_dict['calc_data']['Rise']
            span = self.input_dict['calc_data']['Span']
            half_span = span / 2.0
            b_r = self.input_dict['calc_data']['Bottom radius']
            t_r = self.input_dict['calc_data']['Top radius']
            c_r = self.input_dict['calc_data']['Corner radius']
            b = self.input_dict['calc_data']['b']
            half_points = num_points / 2
            num_top_points = half_points * 0.6
            num_corner_points = half_points * 0.2
            num_bottom_points = half_points * 0.4
            top_origin_x = 0.0
            top_origin_y = rise - t_r
            corner_origin_x = (span / 2.0) - c_r
            corner_origin_y = b
            bottom_origin_x = 0.0
            bottom_origin_y = b_r

            # Degree Assumption helps us get close to the number of points goal
            degrees_assumption = 30  # Assume the top radius will sweep from straight up to about 60 degrees
            angle_increment = degrees_assumption / num_top_points
            cur_angle = 0.0
            top_stations = []
            top_elevations = []
            while cur_angle <= 120:
                station = t_r * math.sin(math.radians(cur_angle)) + top_origin_x
                if station > half_span:
                    break
                top_stations.append(station)
                top_elevations.append(t_r * math.cos(math.radians(cur_angle)) + top_origin_y)
                cur_angle += angle_increment

            # Degree Assumption helps us get close to the number of points goal
            degrees_assumption = 15  # Assume the corner radius to sweep about 30 degrees between intersections
            angle_increment = degrees_assumption / num_corner_points
            cur_angle = 15.0
            corner_stations = []
            corner_elevations = []
            while cur_angle <= 190:
                corner_stations.append(c_r * math.sin(math.radians(cur_angle)) + corner_origin_x)
                corner_elevations.append(c_r * math.cos(math.radians(cur_angle)) + corner_origin_y)
                cur_angle += angle_increment

            # Degree Assumption helps us get close to the number of points goal
            degrees_assumption = 5  # Assume the bottom radius to sweep about 10 degrees (usually long arcs)
            angle_increment = degrees_assumption / num_bottom_points
            cur_angle = 100.0
            bottom_stations = []
            bottom_elevations = []
            # tolerance for last point
            tol_last_point = 180 - angle_increment
            while cur_angle <= 180:
                station = b_r * math.sin(math.radians(cur_angle)) + bottom_origin_x
                if station < half_span:
                    bottom_stations.append(station)
                    bottom_elevations.append(b_r * math.cos(math.radians(cur_angle)) + bottom_origin_y)
                cur_angle += angle_increment
                # We want the last point to be 180
                if tol_last_point < cur_angle < 180:
                    cur_angle = 180

            # Stitch the radia together
            self._pipe_arch_stitch_radia_together(top_stations, top_elevations,
                                                  corner_stations,
                                                  corner_elevations,
                                                  bottom_stations,
                                                  bottom_elevations)

        elif self.input_dict['calc_data']['Shape'] == 'horseshoe':
            radius = self.input_dict['calc_data']['Rise'] / 2.0
            arc_increment = 360.0 / num_points
            cur_angle = 270.0
            self.stations = []
            self.elevations = []
            while cur_angle >= 270.0 or cur_angle <= 90.0:
                if cur_angle >= 360.0:
                    cur_angle -= 360.0
                self.stations.append(radius * math.sin(math.radians(cur_angle)))
                self.elevations.append(radius * math.cos(math.radians(cur_angle)))
                cur_angle += arc_increment

            cur_angle = 90.0
            while cur_angle <= 114.0:
                self.stations.append(2 * radius * math.sin(math.radians(cur_angle)) - radius)
                self.elevations.append(2 * radius * math.cos(math.radians(cur_angle)))
                cur_angle += (arc_increment / 2.0)

            cur_angle = 155.75
            while cur_angle <= 204.25:
                self.stations.append(2 * radius * math.sin(math.radians(cur_angle)))
                self.elevations.append(2 * radius * math.cos(math.radians(cur_angle)) + radius)
                cur_angle += (arc_increment / 2.0)

            cur_angle = 246.0
            while cur_angle <= 270.0:
                self.stations.append(2 * radius * math.sin(math.radians(cur_angle)) + radius)
                self.elevations.append(2 * radius * math.cos(math.radians(cur_angle)))
                cur_angle += (arc_increment / 2.0)

            # close the shape
            cur_angle = 270.0
            self.stations.append(2 * radius * math.sin(math.radians(cur_angle)) + radius)
            self.elevations.append(2 * radius * math.cos(math.radians(cur_angle)))

        # elif self.input_dict['calc_data']['Shape'] == 'oblong':
        #     radius = self.input_dict['calc_data']['Corner radius']
        #     oblong_width = self.input_dict['calc_data']['Oblong width']
        #     arc_increment = 360.0 / num_points
        #     self.stations = []
        #     self.elevations = []
        #     if self.input_dict['calc_data']['Alignment'] == 'vertical':
        #         cur_angle = 270.0
        #         while cur_angle >= 270.0 or cur_angle <= 90.0:
        #             if cur_angle >= 360.0:
        #                 cur_angle -= 360.0
        #             self.stations.append(radius * math.sin(math.radians(cur_angle)))
        #             self.elevations.append(radius * math.cos(math.radians(cur_angle)) + oblong_width)
        #             cur_angle += arc_increment
        #         while cur_angle <= 270.0:
        #             self.stations.append(radius * math.sin(math.radians(cur_angle)))
        #             self.elevations.append(radius * math.cos(math.radians(cur_angle)))
        #             cur_angle += arc_increment
        #     if self.input_dict['calc_data']['Alignment'] == 'horizontal':
        #         cur_angle = 0.0
        #         while cur_angle >= 270.0 or cur_angle <= 90.0:
        #             if cur_angle >= 360.0:
        #                 cur_angle -= 360.0
        #             self.stations.append(radius * math.sin(math.radians(cur_angle)) + oblong_width)
        #             self.elevations.append(radius * math.cos(math.radians(cur_angle)))
        #             cur_angle += arc_increment
        #         while cur_angle <= 270.0:
        #             self.stations.append(radius * math.sin(math.radians(cur_angle)))
        #             self.elevations.append(radius * math.cos(math.radians(cur_angle)))
        #             cur_angle += arc_increment

        # # If we don't have wse stations, but we do have a top width...
        # if top_width is not None and len(wse_stations) <= 0.0 < top_width:
        #     wse_stations.append(-(top_width / 2.0))
        #     wse_stations.append((top_width / 2.0))

        # Now we need to normalize the stations and convert depths to elevations
        lowest_elevation = min(self.elevations)
        self.lowest_station = min(self.stations)

        if lowest_elevation < 0.0:
            for index in range(len(self.elevations)):
                self.elevations[index] += abs(lowest_elevation)

        if self.lowest_station < 0.0:
            for index in range(len(self.stations)):
                self.stations[index] += abs(self.lowest_station)
            for index in range(len(wse_stations)):
                wse_stations[index] += abs(self.lowest_station)

        thalweg_invert = self.input_dict['calc_data']['Thalweg invert elevation']
        for index, _ in enumerate(self.elevations):
            self.elevations[index] += thalweg_invert

        # lowest_elevation = min(self.elevations)
        self.lowest_station = min(self.stations)

        self.high_elev = max(self.elevations)
        self.low_elev = min(self.elevations)

        # # Determine the flow polygon
        if not initializing and depth > self.zero_tol and len(flow_stations) <= 0:
            wse = depth + self.low_elev
            # Assume that we are NOT between intersections (should be valid if we have invisible walls being added)
            # Unless we are in a closed shape, then it may be possible...
            currently_underwater = False
            start_int_station = -999.99
            if not self.shape_is_closed:
                if self.elevations[0] < wse:
                    currently_underwater = True
            for geom_index in range(1, len(self.elevations)):
                # Find intersections with the WSE
                if self.elevations[geom_index] < wse:
                    # intersection found
                    if not currently_underwater:
                        if len(self.embedment_stations):
                            flow_stations.append('nan')
                            flow_elevations.append('nan')
                        currently_underwater = True
                        # Determine the intersection of embedment to channel geom
                        prev_station = self.stations[geom_index - 1]
                        prev_elevation = self.elevations[geom_index - 1]
                        start_int_station = (prev_station - self.stations[geom_index]) / \
                            (prev_elevation - self.elevations[geom_index]) * \
                            (wse - self.elevations[geom_index]) + self.stations[geom_index]
                        flow_stations.append(start_int_station)
                        flow_elevations.append(wse)
                    flow_stations.append(self.stations[geom_index])
                    flow_elevations.append(self.elevations[geom_index])
                else:
                    if currently_underwater:
                        # Determine the intersection of embedment to channel geom
                        prev_station = self.stations[geom_index - 1]
                        prev_elevation = self.elevations[geom_index - 1]
                        int_station = (prev_station - self.stations[geom_index]) / \
                                      (prev_elevation - self.elevations[geom_index]) * \
                                      (wse - self.elevations[geom_index]) + self.stations[geom_index]
                        flow_stations.append(int_station)
                        flow_elevations.append(wse)
                        flow_stations.append(start_int_station)
                        flow_elevations.append(wse)
                        currently_underwater = False
            if len(flow_stations) > 0 and \
                    (flow_stations[0] != flow_stations[-1] or flow_elevations[0] != flow_elevations[-1]):
                flow_stations.append(flow_stations[0])
                flow_elevations.append(flow_elevations[0])

        self.channel_stations = copy.copy(self.stations)
        self.channel_elevations = copy.copy(self.elevations)

        if not initializing and len(wse_stations):
            self.wse_stations.append(wse_stations)
            self.wse_elevations.append(wse_elevations)
            self.flow_stations.append(flow_stations)
            self.flow_elevations.append(flow_elevations)

    compute_cross_sections_for_shapes.counter = 0

    def _pipe_arch_stitch_radia_together(self, top_stations, top_elevations, corner_stations, corner_elevations,
                                         bottom_stations, bottom_elevations):
        """Stitches the radii together for the pipe arch shape.

        Args:
            top_stations (list): stations for the top radius
            top_elevations (list): elevations for the top radius
            corner_stations (list): stations for the corner radius
            corner_elevations (list): elevations for the corner radius
            bottom_stations (list): stations for the bottom radius
            bottom_elevations (list): elevations for the bottom radius
        """
        self.stations = [top_stations[0]]
        self.elevations = [top_elevations[0]]
        # _, zero_tol = self.app_data.get_setting('Zero tolerance')

        _, intersect_x, intersect_y, tr_index, cr_index_1 = find_intersection_and_closest_index(
            poly_line1_x=top_stations, poly_line1_y=top_elevations, poly_line2_x=corner_stations,
            poly_line2_y=corner_elevations, tol=self.zero_tol)

        # Add the top radius and intersection
        self.stations = top_stations[0:tr_index]
        self.elevations = top_elevations[0:tr_index]

        smooth_tr_cr_point = False

        if top_stations[tr_index] != intersect_x:
            self.stations.append(intersect_x)
            self.elevations.append(intersect_y)
        else:
            smooth_tr_cr_point = True

        _, intersect_x, intersect_y, cr_index_2, br_index = find_intersection_and_closest_index(
            corner_stations, corner_elevations, bottom_stations,
            bottom_elevations, tol=self.zero_tol)

        # Add the corner radius and intersection
        self.stations.extend(corner_stations[cr_index_1:cr_index_2])
        self.elevations.extend(corner_elevations[cr_index_1:cr_index_2])

        smooth_cr_br_point = False

        if corner_stations[cr_index_2] != intersect_x:
            self.stations.append(intersect_x)
            self.elevations.append(intersect_y)
        else:
            smooth_cr_br_point = True

        # Add the bottom radius
        self.stations.extend(bottom_stations[br_index:-1])
        self.elevations.extend(bottom_elevations[br_index:-1])

        smoothing = Smoothing(self.stations, self.elevations)
        if smooth_tr_cr_point:
            # Smooth with several passes with increasing distance but with less weight (adjustment).
            # This will smooth the jump and then smooth it into the shape
            distance = xy_distance(self.stations[tr_index - 1],
                                   self.elevations[tr_index - 1],
                                   self.stations[tr_index],
                                   self.elevations[tr_index])
            smoothing.smooth_out_discontinuity(tr_index - 1, tr_index,
                                               distance, 0.1)
            smoothing.smooth_out_discontinuity(tr_index - 1, tr_index,
                                               2 * distance)
            smoothing.smooth_out_discontinuity(tr_index - 1, tr_index,
                                               4 * distance, 0.5)
        if smooth_cr_br_point:
            smoothing.distance_factor_to_smooth = 3.0
            cr_br_index = tr_index + (cr_index_2 - cr_index_1)
            distance = xy_distance(self.stations[cr_br_index - 1],
                                   self.elevations[cr_br_index - 1],
                                   self.stations[cr_br_index],
                                   self.elevations[cr_br_index])
            smoothing.smooth_out_discontinuity(cr_br_index - 1, cr_br_index,
                                               distance, 0.1)
            smoothing.smooth_out_discontinuity(tr_index - 1, tr_index,
                                               2 * distance)
            smoothing.smooth_out_discontinuity(tr_index - 1, tr_index,
                                               4 * distance, 0.5)
        if smooth_tr_cr_point or smooth_cr_br_point:
            self.stations = smoothing.x
            self.elevations = smoothing.y

        size = len(self.stations) - 1
        # If the last point is equivalent to zero, don't add him on again
        # _, zero_tol = self.get_setting('Zero tolerance', 0.00001)
        if self.stations[size] <= self.zero_tol:
            size -= 1
        else:
            self.stations.append(0.0)
            self.elevations.append(0.0)

        # Now complete the shape (we only calculated half)
        for index in range(size, 0, -1):
            self.stations.append(-self.stations[index])
            self.elevations.append(self.elevations[index])

        self.stations.append(self.stations[0])
        self.elevations.append(self.elevations[0])

    def _compute_section_factor(self, depth):
        """Computes the section factor (z) of a geometry.

        Args:
             depth (float): specified Normal depth
        """
        section = 0.0
        if 'Section factor' not in self.results:
            self.results['Section factor'] = []
        y = depth
        if y < self.zero_tol:
            self.results['Section factor'].append(section)
            return section

        # If we have a closed shape, don't compute anything above the crown
        if self.shape_is_closed:
            if y > self.rise_to_crown:
                y = self.rise_to_crown

        if self.input_dict['calc_data']['Shape'] == 'circle':
            d = self.input_dict['calc_data']['Diameter']
            numerator = 2.0 * (self.theta - math.sin(self.theta))**1.5
            denominator = 32.0 * math.sin(self.theta / 2.0)
            if denominator != 0.0:
                section = numerator / denominator * d**2.5

        elif self.input_dict['calc_data']['Shape'] == 'triangle':
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            z = (z1 + z2) / 2.0
            section = 2.0**0.5 / 2.0 * z * y**2.5

        elif self.input_dict['calc_data']['Shape'] == 'round-bottomed triangle':
            area = self.results['Flow area'][-1]
            top_width = self.results['Top width'][-1]
            section = area * (area / top_width)**0.5

        elif self.input_dict['calc_data']['Shape'] == 'trapezoidal':
            b = self.input_dict['calc_data']['Bottom width']
            z1 = self.input_dict['calc_data']['Side slope 1']
            z2 = self.input_dict['calc_data']['Side slope 2']
            z = (z1 + z2) / 2.0
            section = (((b + z * y) * y)**1.5) / ((b + 2 * z * y)**0.5)

        elif self.input_dict['calc_data']['Shape'] == 'parabola':
            top_width_of_flow = self.results['Top width'][-1]
            section = 36.0 * 6.0**0.5 * top_width_of_flow * y**1.5

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            b = self.input_dict['calc_data']['Bottom width']
            rise = self.input_dict['calc_data']['Rise']
            if self.input_dict['calc_data']['Shape'] in ['box']:
                b = self.input_dict['calc_data']['Span']
                if y > rise:
                    y = rise
            section = b * y**1.5

        elif self.input_dict['calc_data']['Shape'] == 'round-cornered rectangle':
            b = self.input_dict['calc_data']['Bottom width']
            r = self.input_dict['calc_data']['Corner radius']
            _, pi = self.get_data('Pi', 3.14159)
            section = (((pi / 2 - 2) * r**2 + (b + 2 * r) * y)**1.5) / ((b + 2 * r)**0.5)

        self.results['Section factor'].append(section)
        return section

    def compute_sequent_depth_and_froude(self, depth, flow):
        """Computes the sequent depth of a geometry given a depth AND flow.

        Args:
             depth (float): specified depth
             flow (float): specified flow
        """
        sequent_depth = depth
        froude = 0.0
        y = depth

        # If we have a closed shape, don't compute anything above the crown
        if self.shape_is_closed:
            if y > self.rise_to_crown:
                y = self.rise_to_crown

        flow_area, _, top_width = self.compute_area_perimeter_top_width(y)

        top_width = self.return_governing_top_width_or_equivalent_depth(depth, top_width, flow_area)
        _, null_data = self.get_data('Null data', -999.0)

        shape = self.input_dict['calc_data']['Shape']
        if shape == 'elliptical':
            bottom_radius = self.input_dict['calc_data']['Bottom radius']
            corner_radius = self.input_dict['calc_data']['Corner radius']
            if bottom_radius <= 0.0 or corner_radius <= 0.0:
                shape = 'cross-section'
                self.compute_cross_sections_for_shapes(depth)

        # Circle
        if not self.channel_is_embedded and shape == 'circle':
            d = self.input_dict['calc_data']['Diameter']

            # Step 1: Calculate dimensionless parameters
            y_prime = y / d
            top_width_prime = top_width / d
            area_prime = self.compute_area_prime(y_prime)

            # Step 2: Calculate upstream Froude number
            froude = self._compute_upstream_froude_prime(
                flow, d, d, area_prime, top_width_prime)
            if froude <= 1.0:  # No Jump
                return null_data, froude

            z_area_prime = self.compute_centroid_area_prime(y_prime)
            full_flow_area_prime = self.compute_area_prime(1.0)
            full_flow_z_area_prime = self.compute_centroid_area_prime(1.0)

            froude_transitional = 0.0

            # Step 3: Calculate transitional upstream Froude number
            if area_prime > 0.0 and full_flow_z_area_prime - z_area_prime > 0.0 and \
                    full_flow_area_prime - area_prime > 0.0:
                froude_transitional = math.sqrt(
                    (top_width_prime * full_flow_area_prime / area_prime**2)
                    * (full_flow_z_area_prime - z_area_prime)
                    / (full_flow_area_prime - area_prime))

            # Step 4: Calculate downstream depth
            y_prime_2 = 0.0
            if froude < froude_transitional:
                # Use Complete Jump Equation
                y_prime_2 = self._compute_sequent_depth_prime(
                    shape, y_prime, froude, top_width_prime, area_prime, z_area_prime)
            else:
                # Use Incomplete Jump Equation
                y_prime_2 = self._compute_sequent_depth_prime_incomplete(
                    y_prime, froude, top_width_prime, area_prime, z_area_prime, full_flow_area_prime,
                    full_flow_z_area_prime)
            sequent_depth = y_prime_2 * d

        elif not self.channel_is_embedded and self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            span = self.input_dict['calc_data']['Bottom width']
            rise = self.input_dict['calc_data']['Channel depth']
            if self.input_dict['calc_data']['Shape'] in ['box']:
                span = self.input_dict['calc_data']['Span']
                rise = self.input_dict['calc_data']['Rise']

            # Step 1: Calculate dimensionless parameters
            y_prime = 0.0
            if rise > 0.0:
                y_prime = y / rise
            top_width_prime = 1.0
            area_prime = self.compute_area_prime(y_prime)
            # Step 2: Calculate upstream Froude number
            froude = self._compute_upstream_froude_prime(flow, span, rise, area_prime, top_width_prime)
            if froude <= 1.0:  # No Jump - subcritical flow
                return null_data, froude

            # Step 3: Calculate transitional upstream Froude number
            froude_transitional = math.sqrt((1 + y_prime) / (2 * y_prime**2))

            # Step 4: Calculate downstream depth
            y_prime_2 = 0.0
            if froude < froude_transitional:
                # Use Complete Jump Equation
                y_prime_2 = self._compute_sequent_depth_prime(
                    shape, y_prime, froude, top_width_prime, area_prime, None)
            else:
                # Use Incomplete Jump Equation
                y_prime_2 = self._compute_sequent_depth_prime_incomplete(
                    y_prime, froude, top_width_prime, area_prime, None, None, None)
            sequent_depth = y_prime_2 * rise

        elif not self.channel_is_embedded and shape in ['pipe arch', 'elliptical']:
            span = self.input_dict['calc_data']['Span']
            rise = self.input_dict['calc_data']['Rise']
            top_radius = self.input_dict['calc_data']['Bottom radius']
            # The only change needed to compute elliptical as pipe arch
            if shape == 'pipe arch':
                top_radius = self.input_dict['calc_data']['Top radius']
            bottom_radius = self.input_dict['calc_data']['Bottom radius']
            corner_radius = self.input_dict['calc_data']['Corner radius']
            # b = self.input['b'].get_val()

            nuetral_axis_height = bottom_radius - math.sqrt(
                ((bottom_radius - corner_radius)**2)
                - (span / 2.0 - corner_radius)**2)
            bottom_transition_height = bottom_radius * (nuetral_axis_height - corner_radius) / \
                (bottom_radius - corner_radius)
            top_transition_height = rise - top_radius * (
                1 - math.sqrt(((top_radius - corner_radius)**2)
                              - ((span / 2 - corner_radius)**2))
                / (top_radius - corner_radius))
            # Step 2: Calculate upstream Froude number
            span_prime = span / rise
            bottom_radius_prime = bottom_radius / rise
            corner_radius_prime = corner_radius / rise
            top_radius_prime = top_radius / rise
            bottom_transition_height_prime = bottom_transition_height / rise
            nuetral_axis_height_prime = nuetral_axis_height / rise
            top_transition_height_prime = top_transition_height / rise

            y_prime = y / rise
            top_width_prime = self._compute_pipearch_top_width_prime(
                y_prime, bottom_transition_height_prime,
                top_transition_height_prime, span_prime, bottom_radius_prime,
                top_radius_prime, nuetral_axis_height_prime)
            area_prime = self._compute_pipearch_area_prime(
                y_prime, bottom_transition_height_prime,
                top_transition_height_prime, span_prime, bottom_radius_prime,
                corner_radius_prime, nuetral_axis_height_prime,
                top_radius_prime)

            # Step 2: Calculate upstream Froude number
            froude = self._compute_upstream_froude_prime(
                flow, span, rise, area_prime, top_width_prime)
            if froude <= 1.0:  # No Jump
                return null_data, froude

            # Step 1.5: Finish Calculating data for transitional froude number
            z_area_prime = self._compute_pipearch_zarea_prime(
                y_prime, bottom_transition_height_prime,
                top_transition_height_prime, span_prime, bottom_radius_prime,
                corner_radius_prime, nuetral_axis_height_prime,
                top_radius_prime)

            full_flow_area_prime = self._compute_pipearch_area_prime(
                1.0, bottom_transition_height_prime,
                top_transition_height_prime, span_prime, bottom_radius_prime,
                corner_radius_prime, nuetral_axis_height_prime,
                top_radius_prime)
            full_flow_z_area_prime = self._compute_pipearch_zarea_prime(
                1.0, bottom_transition_height_prime,
                top_transition_height_prime, span_prime, bottom_radius_prime,
                corner_radius_prime, nuetral_axis_height_prime,
                top_radius_prime)

            # Step 3: Calculate transitional upstream Froude number
            froude_transitional = math.sqrt(
                (top_width_prime * full_flow_area_prime
                 / (area_prime**2)) * (full_flow_z_area_prime - z_area_prime)
                / (full_flow_area_prime - area_prime))

            # Step 4: Calculate downstream depth
            y_prime_2 = 0.0
            if froude < froude_transitional:
                # Use Complete Jump Equation
                y_prime_2 = self._compute_sequent_depth_prime(
                    shape, y_prime, froude, top_width_prime, area_prime, z_area_prime,
                    bottom_transition_height_prime=bottom_transition_height_prime,
                    top_transition_height_prime=top_transition_height_prime, span_prime=span_prime,
                    bottom_radius_prime=bottom_radius_prime, corner_radius_prime=corner_radius_prime,
                    nuetral_axis_height_prime=nuetral_axis_height_prime, top_radius_prime=top_radius_prime)
            else:
                # Use Incomplete Jump Equation
                y_prime_2 = self._compute_sequent_depth_prime_incomplete(
                    y_prime, froude, top_width_prime, area_prime, z_area_prime,
                    full_flow_area_prime, full_flow_z_area_prime)
            sequent_depth = y_prime_2 * rise

        # Use irregular geometry!
        # elif shape in ['cross-section', 'triangle', 'round-bottomed triangle', 'trapezoidal', 'parabola',
        #                'round-cornered rectangle', 'curb and gutter', 'horseshoe', 'oblong']:
        else:
            min_station = min(self.stations)
            span = max(self.stations) - min(self.stations)
            rise = self.high_elev - self.low_elev
            station_prime = [(station - min_station) / span
                             for station in self.stations]
            elev_prime = [(station - self.low_elev) / rise
                          for station in self.elevations]

            y_prime = y / rise
            area_prime, z_area_prime, top_width_prime = self._user_defined_area_zarea_topwidth_prime(
                y_prime, station_prime, elev_prime)

            full_flow_area_prime, full_flow_z_area_prime, _ = self._user_defined_area_zarea_topwidth_prime(
                1.0, station_prime, elev_prime)

            # Step 2: Calculate upstream Froude number
            froude = self._compute_upstream_froude_prime(
                flow, span, rise, area_prime, top_width_prime)
            if froude <= 1.0:  # No Jump
                return null_data, froude

            # Jump may occur

            # Step 3: Calculate transitional upstream Froude number
            froude_transitional = 0.0
            if area_prime != 0.0 and full_flow_area_prime - area_prime != 0.0:
                froude_transitional = math.sqrt(
                    (top_width_prime * full_flow_area_prime / pow(area_prime, 2))
                    * (full_flow_z_area_prime - z_area_prime)
                    / (full_flow_area_prime - area_prime))

            # Step 4: Calculate downstream depth
            y_prime_2 = 0.0
            if froude < froude_transitional:  # complete jump
                y_prime_2 = self._compute_sequent_depth_prime(
                    shape, y_prime, froude, top_width_prime, area_prime,
                    z_area_prime, station_prime, elev_prime)
            else:  # incomplete jump
                y_prime_2 = self._compute_sequent_depth_prime_incomplete(
                    y_prime, froude, top_width_prime, area_prime, z_area_prime,
                    full_flow_area_prime, full_flow_z_area_prime)

            sequent_depth = y_prime_2 * rise

            # end of "Jump may occur" else

        return sequent_depth, froude

    def _compute_upstream_froude_prime(self, flow, span, rise, area_prime, top_width_prime):
        """Compute upstream froude' value. Necessary for hydraulic jump calculations.

        Args:
            flow (float): the flow rate through the channel
            span (float): the span of the channel
            rise (float): the rise of the channel
            top_width_prime (float): top width' value

        Returns:
            froude (float): Froude Number of the flow upstream of a hydraulic jump
        """
        _, g = self.get_data('Gravity', 32.2)
        _, null_data = self.get_data('Null data', -999.0)

        froude = null_data
        if top_width_prime > 0.0:
            denominator = math.sqrt(g * span**2 * rise**3 * area_prime**3 / top_width_prime)
            if denominator > 0.0:
                froude = flow / denominator
        return froude

    def compute_area_prime(self, y_prime):
        """Computes the sequent depth of a geometry given a depth AND flow.

        Args:
             y_prime (float): specified dimensionless depth

        Returns:
            a_prime (float): area' value for hydraulic jump calculations
        """
        a_prime = 0.0

        # Circle
        if self.input_dict['calc_data']['Shape'] == 'circle':
            if y_prime > 1.0:
                # We have a real issue here!
                # We cannot be deeper than the circle's diameter!
                y_prime = 1.0
            if y_prime <= 0.0:
                return 0.0
            theta_prime = 2 * math.acos(1 - 2 * y_prime)
            a_prime = 0.125 * (theta_prime - math.sin(theta_prime))

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            a_prime = y_prime

        # elif self.input_dict['calc_data']['Shape'] in ['pipe arch', 'elliptical']:
        #     pass

        # # Use irregular geometry!
        # elif self.input_dict['calc_data']['Shape'] in [
        #         'cross-section', 'triangle', 'round-bottomed triangle',
        #         'trapezoidal', 'parabola', 'round-cornered rectangle', 'curb and gutter',
        #         'horseshoe',  # 'oblong'
        # ]:
        #     pass

        return a_prime

    def _compute_pipearch_top_width_prime(
            self, y_prime, bottom_transition_height_prime, top_transition_height_prime, span_prime, bottom_radius_prime,
            top_radius_prime, nuetral_axis_height_prime):
        """Compute Pipe arch top width'. Used for hydraulic jump calculations.

        Args:
            y_prime (float): depth' variable
            bottom_transition_height_prime (float): variable that is defined by Nathan Lowe's thesis on hydraulic jumps
            top_transition_height_prime (float): variable that is defined by hydraulic jump equations.
            span_prime (float): normalized span width
            bottom_radius_prime (float): variable that is defined by hydraulic jump equations.
            top_radius_prime (float): variable that is defined by hydraulic jump equations.
            nuetral_axis_height_prime (float): variable that is defined by hydraulic jump equations.

        Returns:
            (float): the top width' of the pipe arch
        """
        tb_prime = 0.0
        tm1_prime = 0.0
        tm2_prime = 0.0
        tt_prime = 0.0

        if y_prime <= bottom_transition_height_prime:
            tb_prime = self._compute_pipearch_tb_prime(y_prime, span_prime,
                                                       bottom_radius_prime)
        elif y_prime <= top_transition_height_prime:
            tm1_prime = self._compute_pipearch_tm1_prime(
                span_prime, top_radius_prime)
            tm2_prime = self._compute_pipearch_tm2_prime(
                y_prime, span_prime, top_radius_prime,
                nuetral_axis_height_prime)
        elif y_prime <= 1:
            tt_prime = self._compute_pipearch_tt_prime(y_prime, span_prime,
                                                       top_radius_prime)

        return tb_prime + tm1_prime + tm2_prime + tt_prime

    @staticmethod
    def _compute_pipearch_tb_prime(y_prime, span_prime, bottom_radius_prime):
        """Compute pipe arch tb'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            bottom_radius_prime (float): normalized bottom radius' variable

        Returns:
            (float): pipearch tb'
        """
        theta_b = 2 * math.acos(1 - y_prime / bottom_radius_prime)
        return 2.0 * bottom_radius_prime / span_prime * math.sin(theta_b / 2.0)

    @staticmethod
    def _compute_pipearch_tm1_prime(span_prime, corner_radius_prime, ):
        """Compute pipe arch tm1'.  Necessary for hydraulic jump calculations.

        Args:
            span_prime (float): normalized span' variable
            corner_radius_prime (float): normalized corner radius' variable

        Returns:
            (float): pipe arch tm1'
        """
        return 1.0 / span_prime * (span_prime - 2 * corner_radius_prime)

    @staticmethod
    def _compute_pipearch_tm2_prime(y_prime, span_prime, corner_radius_prime,
                                    nuetral_axis_height_prime):
        """Compute pipe arch tm2'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable

        Returns:
            (float): pipe arch tm2'
        """
        theta_m = 2 * math.acos(
            1 - 1 / corner_radius_prime
            * (y_prime - nuetral_axis_height_prime + corner_radius_prime))
        return (2 * corner_radius_prime) / span_prime * math.sin(theta_m / 2.0)

    @staticmethod
    def _compute_pipearch_tt_prime(y_prime, span_prime, top_radius_prime):
        """Compute pipe arch tt'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            top_radius_prime (float): normalized top radius' variable

        Returns:
            (float): pipe arch tt'
        """
        theta_t = 2 * math.acos((1 - y_prime) / top_radius_prime - 1)
        return (2 * top_radius_prime) / span_prime * math.sin(theta_t / 2.0)

    def _compute_pipearch_area_prime(self, y_prime,
                                     bottom_transition_height_prime,
                                     top_transition_height_prime, span_prime,
                                     bottom_radius_prime, corner_radius_prime,
                                     nuetral_axis_height_prime,
                                     top_radius_prime):
        """Compute pipe arch area'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            bottom_transition_height_prime (float): bottom transition height' variable
            top_transition_height_prime (float): top transition height' variable
            span_prime (float): normalized span' variable
            bottom_radius_prime (float): normalized bottom radius' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable
            top_radius_prime (float): normalized top radius' variable

        Returns:
            (float): pipe arch area'
        """
        ab_prime = 0
        am1_prime = 0
        am2_prime = 0
        at_prime = 0

        if y_prime <= bottom_transition_height_prime:
            ab_prime = self._compute_pipearch_ab_prime(y_prime, span_prime,
                                                       bottom_radius_prime)
        elif y_prime <= top_transition_height_prime:
            ab_prime = self._compute_pipearch_ab_prime(
                bottom_transition_height_prime, span_prime,
                bottom_radius_prime)
            am1_prime = self._compute_pipearch_am1_prime(
                y_prime, span_prime, corner_radius_prime, nuetral_axis_height_prime) - \
                self._compute_pipearch_am1_prime(bottom_transition_height_prime, span_prime,
                                                 corner_radius_prime, nuetral_axis_height_prime)
            at_prime = self._compute_pipearch_am2_prime(
                y_prime, span_prime, corner_radius_prime, nuetral_axis_height_prime) - \
                self._compute_pipearch_am2_prime(bottom_transition_height_prime, span_prime, corner_radius_prime,
                                                 nuetral_axis_height_prime)
        elif y_prime <= 1:
            ab_prime = self._compute_pipearch_ab_prime(
                bottom_transition_height_prime, span_prime,
                bottom_radius_prime)
            am1_prime = self._compute_pipearch_am1_prime(
                top_transition_height_prime, span_prime, corner_radius_prime, nuetral_axis_height_prime) - \
                self._compute_pipearch_am1_prime(bottom_transition_height_prime, span_prime,
                                                 corner_radius_prime, nuetral_axis_height_prime)
            am2_prime = self._compute_pipearch_am2_prime(
                top_transition_height_prime, span_prime, corner_radius_prime, nuetral_axis_height_prime) - \
                self._compute_pipearch_am2_prime(bottom_transition_height_prime, span_prime,
                                                 corner_radius_prime, nuetral_axis_height_prime)
            at_prime = self._compute_pipearch_at_prime(y_prime, span_prime, top_radius_prime) - \
                self._compute_pipearch_at_prime(top_transition_height_prime, span_prime, top_radius_prime)
        return ab_prime + am1_prime + am2_prime + at_prime

    @staticmethod
    def _compute_pipearch_ab_prime(y_prime, span_prime, bottom_radius_prime):
        """Compute pipe arch ab'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            bottom_radius_prime (float): normalized bottom radius' variable

        Returns:
            (float): pipe arch ab'
        """
        theta_b = 2 * math.acos(1 - y_prime / bottom_radius_prime)
        return (bottom_radius_prime
                ** 2) / (2 * span_prime) * (theta_b - math.sin(theta_b))

    @staticmethod
    def _compute_pipearch_am1_prime(y_prime, span_prime, corner_radius_prime,
                                    nuetral_axis_height_prime):
        """Compute pipe arch am1'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable

        Returns:
            (float): pipe arch am1'
        """
        return 1.0 / span_prime * (span_prime - 2 * corner_radius_prime) * (
            y_prime - nuetral_axis_height_prime + corner_radius_prime)

    @staticmethod
    def _compute_pipearch_am2_prime(y_prime, span_prime, corner_radius_prime,
                                    nuetral_axis_height_prime):
        """Compute pipe arch am2'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable

        Returns:
            (float): pipe arch am2'
        """
        theta_m = 2 * math.acos(
            1 - 1 / corner_radius_prime
            * (y_prime - nuetral_axis_height_prime + corner_radius_prime))
        return (corner_radius_prime
                ** 2) / (2 * span_prime) * (theta_m - math.sin(theta_m))

    @staticmethod
    def _compute_pipearch_at_prime(y_prime, span_prime, top_radius_prime):
        """Compute pipe arch at'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            top_radius_prime (float): normalized top radius' variable

        Returns:
            (float): pipe arch at'
        """
        theta_t = 2 * math.acos((1 - y_prime) / top_radius_prime - 1)
        return (top_radius_prime
                ** 2) / (2 * span_prime) * (theta_t - math.sin(theta_t))

    def _compute_pipearch_zarea_prime(self, y_prime,
                                      bottom_transition_height_prime,
                                      top_transition_height_prime, span_prime,
                                      bottom_radius_prime, corner_radius_prime,
                                      nuetral_axis_height_prime,
                                      top_radius_prime):
        """Compute pipe arch z area' (centroid).  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            bottom_transition_height_prime (float): bottom transition height' variable
            top_transition_height_prime (float): top transition height' variable
            span_prime (float): normalized span' variable
            bottom_radius_prime (float): normalized bottom radius' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable
            top_radius_prime (float): normalized top radius' variable

        Returns:
            (float): pipe arch z area'
        """
        # Assign variables
        zab_prime = 0.0
        ab_prime = 0.0
        zam1_nahp_tthp = 0.0
        zam1_nahp_bthp = 0.0
        am1_nahp_tthp = 0.0
        am1_nahp_bthp = 0.0
        zam2_nahp_tthp = 0.0
        zam2_nahp_bthp = 0.0
        am2_nahp_tthp = 0.0
        am2_nahp_bthp = 0.0
        zat_y = 0.0
        zat_tthp = 0.0
        at_tthp = 0.0

        # Shorthand
        tthp = top_transition_height_prime
        bthp = bottom_transition_height_prime
        nahp = nuetral_axis_height_prime

        if y_prime <= bottom_transition_height_prime:
            ab_prime = self._compute_pipearch_ab_prime(y_prime, span_prime,
                                                       bottom_radius_prime)
            zab_prime = self._compute_pipearch_zab_prime(
                y_prime, span_prime, bottom_radius_prime)
        elif y_prime <= top_transition_height_prime:
            ab_prime = self._compute_pipearch_ab_prime(bthp, span_prime,
                                                       bottom_radius_prime)
            zab_prime = self._compute_pipearch_zab_prime(
                bthp, span_prime, bottom_radius_prime)

            am1_nahp_tthp = self._compute_pipearch_am1_prime(
                y_prime, span_prime, corner_radius_prime, nahp)
            zam1_nahp_tthp = self._compute_pipearch_zam1_prime(
                y_prime, span_prime, corner_radius_prime, nahp)
            am1_nahp_bthp = self._compute_pipearch_am1_prime(
                bthp, span_prime, corner_radius_prime, nahp)
            zam1_nahp_bthp = self._compute_pipearch_zam1_prime(
                bthp, span_prime, corner_radius_prime, nahp)

            am2_nahp_tthp = self._compute_pipearch_am2_prime(
                y_prime, span_prime, corner_radius_prime, nahp)
            zam2_nahp_tthp = self._compute_pipearch_zam2_prime(
                y_prime, span_prime, corner_radius_prime, nahp)
            am2_nahp_bthp = self._compute_pipearch_am2_prime(
                bthp, span_prime, corner_radius_prime, nahp)
            zam2_nahp_bthp = self._compute_pipearch_zam2_prime(
                bthp, span_prime, corner_radius_prime, nahp)

            # Remove these parameters, if the y_prime is less than this height
            if (y_prime - tthp) < 0.0:
                tthp = y_prime

        elif y_prime <= 1:
            ab_prime = self._compute_pipearch_ab_prime(bthp, span_prime,
                                                       bottom_radius_prime)
            zab_prime = self._compute_pipearch_zab_prime(
                bottom_transition_height_prime, span_prime,
                bottom_radius_prime)

            am1_nahp_tthp = self._compute_pipearch_am1_prime(
                tthp, span_prime, corner_radius_prime, nahp)
            zam1_nahp_tthp = self._compute_pipearch_zam1_prime(
                tthp, span_prime, corner_radius_prime, nahp)
            am1_nahp_bthp = self._compute_pipearch_am1_prime(
                bthp, span_prime, corner_radius_prime, nahp)
            zam1_nahp_bthp = self._compute_pipearch_zam1_prime(
                bthp, span_prime, corner_radius_prime, nahp)

            am2_nahp_tthp = self._compute_pipearch_am2_prime(
                tthp, span_prime, corner_radius_prime, nahp)
            zam2_nahp_tthp = self._compute_pipearch_zam2_prime(
                tthp, span_prime, corner_radius_prime, nahp)
            am2_nahp_bthp = self._compute_pipearch_am2_prime(
                bthp, span_prime, corner_radius_prime, nahp)
            zam2_nahp_bthp = self._compute_pipearch_zam2_prime(
                bthp, span_prime, corner_radius_prime, nahp)

            zat_y = self._compute_pipearch_zat_prime(y_prime, span_prime,
                                                     top_radius_prime)
            at_tthp = self._compute_pipearch_at_prime(tthp, span_prime,
                                                      top_radius_prime)
            zat_tthp = self._compute_pipearch_zat_prime(
                tthp, span_prime, top_radius_prime)

        za = zab_prime + (y_prime - bthp) * ab_prime + (zam1_nahp_tthp - zam1_nahp_bthp) + \
            (y_prime - tthp) * am1_nahp_tthp - (y_prime - bthp) * am1_nahp_bthp + \
            (zam2_nahp_tthp - zam2_nahp_bthp) + (y_prime - tthp) * am2_nahp_tthp - \
            (y_prime - bthp) * am2_nahp_bthp + (zat_y - zat_tthp) - \
            (y_prime - tthp) * at_tthp
        return za

    @staticmethod
    def _compute_pipearch_zab_prime(y_prime, span_prime, bottom_radius_prime):
        """Compute pipe arch zab'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            bottom_radius_prime (float): normalized bottom radius' variable

        Returns:
            (float): pipe arch zab'
        """
        theta_b = 2 * math.acos(1 - y_prime / bottom_radius_prime)
        return bottom_radius_prime**3 / (3 * span_prime) * (
            3 * math.sin(theta_b / 2) - math.sin(theta_b / 2)**3 - 3
            * (theta_b / 2) * math.cos(theta_b / 2))

    @staticmethod
    def _compute_pipearch_zam1_prime(y_prime, span_prime, corner_radius_prime,
                                     nuetral_axis_height_prime):
        """Compute pipe arch zam1'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable

        Returns:
            (float): pipe arch zam1'
        """
        return 1.0 / (2.0 * span_prime) * (span_prime - 2 * corner_radius_prime) * \
            (y_prime - nuetral_axis_height_prime + corner_radius_prime) ** 2

    @staticmethod
    def _compute_pipearch_zam2_prime(y_prime, span_prime, corner_radius_prime,
                                     nuetral_axis_height_prime):
        """Compute pipe arch zam2'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            corner_radius_prime (float): normalized corner radius' variable
            nuetral_axis_height_prime (float): nuetral axes height' variable

        Returns:
            (float): pipe arch zam2'
        """
        theta_m = \
            2 * math.acos(1 - 1 / corner_radius_prime * (y_prime - nuetral_axis_height_prime + corner_radius_prime))
        return (corner_radius_prime
                ** 3) / (3 * span_prime) * (3 * math.sin(theta_m / 2)
                                            - (math.sin(theta_m / 2)**3) - 3
                                            * (theta_m / 2) * math.cos(theta_m / 2))

    @staticmethod
    def _compute_pipearch_zat_prime(y_prime, span_prime, top_radius_prime):
        """Compute pipe arch zat'.  Necessary for hydraulic jump calculations.

        Args:
            y_prime (float): normalized depth' variable
            span_prime (float): normalized span' variable
            top_radius_prime (float): normalized top radius' variable

        Returns:
            (float): pipe arch zat'
        """
        theta_t = 2 * math.acos((1 - y_prime) / top_radius_prime - 1)
        return (top_radius_prime ** 3) / (3 * span_prime) * (3 * math.sin(theta_t / 2) - (
            math.sin(theta_t / 2)**3) - 3 * (theta_t / 2) * math.cos(theta_t / 2))

    def compute_centroid_area_prime(self, y_prime):
        """Compute the centroid area'.

        Args:
             y_prime (float): specified dimensionless depth
        """
        # Note: This is only used for circular channels
        z_area_prime = 0.0

        theta = 2.0 * math.acos(1.0 - 2.0 * y_prime)
        half_theta = 0.5 * theta
        sin_term = math.sin(half_theta)
        z_area_prime = (3.0 * sin_term - sin_term**3 - 3 * half_theta * math.cos(half_theta)) / 24.0

        return z_area_prime

    def _compute_sequent_depth_prime(
            self, shape, y_prime, froude, top_width_prime, area_prime, z_area_prime, station_prime=None,
            elev_prime=None, bottom_transition_height_prime=None, top_transition_height_prime=None, span_prime=None,
            bottom_radius_prime=None, corner_radius_prime=None, nuetral_axis_height_prime=None, top_radius_prime=None):
        """Computes the sequent depth of a geometry given a depth AND flow.

        Args:
            shape (str): shape of the geometry
            y_prime (float): specified dimensionless depth
            froude (float): Froude number
            top_width_prime (float): top width' value
            area_prime (float): area' value
            z_area_prime (float): centroid area' value
            station_prime (list): normalized station values
            elev_prime (list): normalized elevation values
            bottom_transition_height_prime (float): variable that is defined by Nathan Lowe's thesis on hydraulic jumps
            top_transition_height_prime (float): variable that is defined by hydraulic jump equations.
            span_prime (float): normalized span width
            bottom_radius_prime (float): variable that is defined by hydraulic jump equations.
            corner_radius_prime (float): variable that is defined by hydraulic jump equations.
            nuetral_axis_height_prime (float): variable that is defined by hydraulic jump equations.
            top_radius_prime (float): variable that is defined by hydraulic jump equations.
        """
        sequent_depth_prime = 0.0
        _, error = self.get_data('Sequent error', 0.01)
        y2_prime_1 = y_prime  # lower sequent depth estimate
        func1 = 1 - froude**2  # function at lower estimate
        y2_prime_2 = 1.0  # Upper sequent depth estimate
        # func2 = froude_transitional ** 2 - froude ** 2  # Function at upper estimate
        y2_prime_3 = 0.0  # Intermediate estimate
        # func3 = 0.0  # Function at intermediate estimate
        # area2_prime = 0.0  # downstream dimensionless area
        # z_area2_prime = 0.0  # downstream dimensionless centroid area

        count = 0
        _, max_iterations = self.get_data('Max number of iterations', 500)

        if shape == 'circle':
            while y2_prime_2 - y2_prime_1 > error:
                # Intermediate estimate
                y2_prime_3 = (y2_prime_1 + y2_prime_2) / 2.0
                # Downstream dimensionless area
                area2_prime = self.compute_area_prime(y2_prime_3)
                # Downstream dimensionless centroid - area
                z_area2_prime = self.compute_centroid_area_prime(y2_prime_3)
                # Function at intermediate estimate
                func3 = self._function_at_intermediate_estimate(
                    area_prime, z_area_prime, top_width_prime, area2_prime,
                    z_area2_prime, froude)
                if (func1 * func3) > 0.0:
                    # y2_prime_3 is new lower estimate
                    y2_prime_1 = y2_prime_3
                    func1 = func3
                else:
                    # y2_prime_3 is new upper estimate
                    y2_prime_2 = y2_prime_3
                    # func2 = func3
                count += 1
            sequent_depth_prime = y2_prime_3

        elif self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            sequent_depth_prime = 0.5 * y_prime * (math.sqrt(1.0 + 8.0 * froude**2) - 1.0)

        elif shape in ['pipe arch', 'elliptical']:
            while y2_prime_2 - y2_prime_1 > error and count < max_iterations:
                # Intermediate estimate
                y2_prime_3 = (y2_prime_1 + y2_prime_2) / 2.0
                # Downstream dimensionless area
                area2_prime = self._compute_pipearch_area_prime(
                    y2_prime_3, bottom_transition_height_prime,
                    top_transition_height_prime, span_prime,
                    bottom_radius_prime, corner_radius_prime,
                    nuetral_axis_height_prime, top_radius_prime)
                # Downstream dimensionless centroid - area
                z_area2_prime = self._compute_pipearch_zarea_prime(
                    y2_prime_3, bottom_transition_height_prime,
                    top_transition_height_prime, span_prime,
                    bottom_radius_prime, corner_radius_prime,
                    nuetral_axis_height_prime, top_radius_prime)
                # Function at intermediate estimate
                func3 = self._function_at_intermediate_estimate(
                    area_prime, z_area_prime, top_width_prime, area2_prime,
                    z_area2_prime, froude)

                if func1 * func3 > 0:  # y2_prime_3 is new upper estimate
                    y2_prime_1 = y2_prime_3
                    func1 = func3
                else:  # m_y2_prime3 is new lower estimate
                    y2_prime_2 = y2_prime_3
                    # func2 = func3
                count += 1
            sequent_depth_prime = y2_prime_3

        # Use irregular geometry!
        elif shape in [
                'cross-section', 'triangle', 'round-bottomed triangle',
                'trapezoidal', 'parabola', 'round-cornered rectangle', 'curb and gutter',
                'horseshoe',  # 'oblong'
        ]:
            while y2_prime_2 - y2_prime_1 > error and count < max_iterations:
                # Intermediate estimate
                y2_prime_3 = (y2_prime_1 + y2_prime_2) / 2.0
                # Downstream dimensionless area & centroid - area
                area2_prime, z_area2_prime, _ = self._user_defined_area_zarea_topwidth_prime(
                    y2_prime_3, station_prime, elev_prime)
                # Function at intermediate estimate
                func3 = self._function_at_intermediate_estimate(
                    area_prime, z_area_prime, top_width_prime, area2_prime,
                    z_area2_prime, froude)

                if func1 * func3 > 0:  # y2_prime_3 is new upper estimate
                    y2_prime_1 = y2_prime_3
                    func1 = func3
                else:  # m_y2_prime3 is new lower estimate
                    y2_prime_2 = y2_prime_3
                    # func2 = func3
                count += 1
            sequent_depth_prime = y2_prime_3

        return sequent_depth_prime

    @staticmethod
    def _function_at_intermediate_estimate(area_prime, z_area_prime,
                                           top_width_prime, area2_prime,
                                           z_area2_prime, froude):
        """Function used iteratively in the complete jump equation for all shapes.

        Args:
            area_prime (float): area' variable
            z_area_prime (float): z area' (centroid)
            top_width_prime (float): top width'
            area2_prime (float): area' at location 2
            z_area2_prime (float): z area' (centroid) at location 2
            froude (float): froude number

        Return:
            (float): result of 'function'
        """
        return (top_width_prime * area2_prime / area_prime ** 2) * (z_area_prime - z_area2_prime) / \
               (area_prime - area2_prime) - froude ** 2

    def _compute_sequent_depth_prime_incomplete(self, y_prime, froude,
                                                top_width_prime, area_prime,
                                                z_area_prime,
                                                full_flow_area_prime,
                                                full_flow_z_area_prime):
        """Computes the sequent depth of a geometry given a depth AND flow.

        Args:
            y_prime (float): area' variable
            froude (float): froude number
            top_width_prime (float): top width'
            area_prime (float): area' variable
            z_area_prime (float): z area' (centroid)
            full_flow_area_prime (float): area' under full flow
            full_flow_z_area_prime (float): z area' (centroid) under full flow

        Return:
            sequent_depth_prime (float): sequent depth'
        """
        sequent_depth_prime = 0.0
        # _, error = SettingsCalc().get_setting('Sequent error', 0.01)
        # y2_prime1 = y_prime  # lower sequent depth estimate
        # func1 = 1 - froude ** 2  # function at lower estimate
        # y2_prime_2 = 1.0  # Upper sequent depth estimate

        if not self.channel_is_embedded and self.input_dict['calc_data']['Shape'] == 'circle':
            sequent_depth_prime = (
                (froude**2 * area_prime**2 * (area_prime - area_prime)
                 - top_width_prime * full_flow_area_prime
                 * (full_flow_z_area_prime - area_prime))
                / (top_width_prime * full_flow_area_prime**2) + 1)

        elif not self.channel_is_embedded and self.input_dict['calc_data']['Shape'] in ['rectangle', 'box']:
            sequent_depth_prime = 0.5 + (
                froude**2 + 0.5) * y_prime**2 - froude**2 * pow(y_prime, 3)

        elif not self.channel_is_embedded and self.input_dict['calc_data']['Shape'] in ['pipe arch', 'elliptical']:
            sequent_depth_prime = 1 + 1 / (
                top_width_prime * full_flow_area_prime**2) * (
                    froude**2 * area_prime**2
                    * (full_flow_area_prime - area_prime)
                    - top_width_prime * full_flow_area_prime
                    * (full_flow_z_area_prime - z_area_prime))

        # Use irregular geometry!
        else:
            sequent_depth_prime = (
                froude ** 2 * area_prime ** 2 * (full_flow_area_prime - area_prime)
                - top_width_prime * full_flow_area_prime * (full_flow_z_area_prime - z_area_prime)
            ) / (top_width_prime * full_flow_area_prime ** 2) + 1

        return sequent_depth_prime

    def _user_defined_area_zarea_topwidth_prime(self, y_prime, station_prime, elev_prime):
        """User defined channel calc for area, z area (centroid), and top width'.

        Args:
            y_prime (float): area' variable
            station_prime (list of floats): station' variable
            elev_prime (list of floats): elev_prime' variable

        Return:
            a_prime (float): area'
            z_a_prime (float): z area' (centroid)
            top_width_prime (float): top width'
        """
        a_prime = 0.0
        z_a_prime = 0.0
        top_width_prime = 0.0

        h1 = 0.0
        h2 = 0.0
        _, null_data_val = self.get_data('Null data', -999.0)
        x_first = null_data_val

        for index in range(len(station_prime) - 1):
            b = station_prime[index + 1] - station_prime[index]
            if elev_prime[index] < elev_prime[index + 1]:
                if y_prime < elev_prime[index]:
                    # Depth is below both
                    h1 = 0.0
                    h2 = 0.0
                    b = 0.0
                elif y_prime > elev_prime[index + 1]:
                    # Depth is above both
                    h1 = y_prime - elev_prime[index]
                    h2 = y_prime - elev_prime[index + 1]
                else:
                    # Depth is between
                    h1 = y_prime - elev_prime[index]
                    h2 = 0.0
                    # Determine new x distance
                    elev_change = elev_prime[index + 1] - elev_prime[index]
                    # Unable to hit elev_change == 0.0 because of previous if statement
                    # if elev_change == 0.0:
                    #     b = 0.0
                    # else:
                    b = h1 / (elev_change) * b
                    if x_first == null_data_val:
                        x_first = station_prime[index] + b
                    else:
                        top_width_prime += abs(station_prime[index] + b - x_first)
                        x_first = null_data_val
            else:
                if y_prime < elev_prime[index + 1]:
                    # Depth is below both
                    h1 = 0.0
                    h2 = 0.0
                elif y_prime > elev_prime[index]:
                    # Depth is above both
                    h1 = y_prime - elev_prime[index]
                    h2 = y_prime - elev_prime[index + 1]
                else:
                    # Depth is between
                    h1 = 0.0
                    h2 = y_prime - elev_prime[index + 1]
                    # Determine new x distance
                    elev_change = elev_prime[index] - elev_prime[index + 1]
                    if elev_change == 0.0:
                        b = 0.0
                    else:
                        b = h2 / (elev_change) * b
                    if x_first == null_data_val:
                        x_first = station_prime[index] + b
                    else:
                        top_width_prime += abs(station_prime[index] + b - x_first)
                        x_first = null_data_val

            # Area for a trapezoid (from Nathan Lowe's thesis, page 152; pdf page 178):
            # 1/2 * b * (h1 + h2)
            a_prime_i = 1.0 / 2.0 * b * (h1 + h2)
            a_prime += a_prime_i

            # Centroid Area for a trapezoid (from Nathan Lowe's thesis, page 152; pdf page 178):
            # 1/6 * b * (h1^2 + h1*h2+h2^2)
            z_a_prime_i = 1 / 6 * b * (h1**2 + h1 * h2 + h2**2)
            z_a_prime += z_a_prime_i

        if a_prime < 0.0:
            a_prime = abs(a_prime)
            z_a_prime = abs(z_a_prime)

        return a_prime, z_a_prime, top_width_prime

    def _compute_cross_section_geometry_from_elevation(self, wse):
        """Determines the geometry variables (area and perimeter mainly, wet stations too) for a given elevation.

        Args:
            wse (float): Elevation of the water surface (wse)
        """
        flow_stations = []
        flow_elevations = []

        wse = self._check_elevation_to_max_depth(wse)
        vertical_wall_needed = self.vertical_wall_needed(wse)

        flow_area = 0.0
        top_width = 0.0
        perimeter = 0.0

        flow_stations_by_poly = []
        flow_elevations_by_poly = []

        flow_area_by_poly = []
        perimeter_by_poly = []
        top_width_by_poly = []
        thalweg_in_poly = []
        poly_has_ineffective_flow = []

        wse_stations_in_poly = []
        wse_elevations_in_poly = []

        inef_index = None
        if self.ineffective_flow_stations and len(self.ineffective_flow_stations) > 0:
            inef_index = 0
        have_ineffective_flow = False
        have_thalweg = False
        thalwegs = self.input_dict['calc_data']['Thalweg station(s)']
        check_thalwegs = not self.shape_is_closed and self.input_dict['calc_data'][
            'Only fill flow areas with thalweg locations']
        if not thalwegs or len(thalwegs) == 0:
            thalwegs = [self.thalweg_station]

        left_int_x = None
        left_int_y = None
        left_int_wse = None

        # Manning's n values
        mannings_n_horton = 0.0
        mannings_n_pavlovskii = 0.0
        horton_perimeter = []
        horton_area = []
        horton_mannings_n = []

        mannings_n_horton_by_poly = []
        mannings_n_pavlovskii_by_poly = []
        horton_perimeter_by_poly = []
        horton_area_by_poly = []
        horton_mannings_n_by_poly = []

        calc_mannings_n = False
        if len(self.mannings_n):
            calc_mannings_n = True

        # Perform superelevation calculations
        wse_1 = wse
        wse_2 = wse
        super_wse_interp = None
        if self.input_dict['calc_data']['Curvature direction'] != 'None':
            null_data = self.get_data('Null data', -999.0)[1]
            if self.superelev_top_width is None:
                channel_interp = Interpolation(self.elevations, self.stations, null_data=null_data,
                                               zero_tol=self.zero_tol)
                first_station, _ = channel_interp.interpolate_y(wse)
                rev_channel_interp = Interpolation(list(reversed(self.elevations)), list(reversed(self.stations)),
                                                   null_data=null_data, zero_tol=self.zero_tol)
                last_station, _ = rev_channel_interp.interpolate_y(wse)
                self.superelev_top_width = abs(last_station - first_station)
            if self.superelev_velocity is None:
                # We have no reasonable method to approximate the velocity, so just take a guess of 2
                self.superelev_velocity = 2.0
            if self.superelev_froude is None:
                # We have no reasonable method to approximate the Froude number, so just take a guess of 0.5
                self.superelev_froude = 0.5
            super_wse_x, super_wse_y, _ = self.compute_wse(
                wse, self.superelev_top_width, self.superelev_velocity, self.stations, self.elevations,
                self.superelev_froude)
            super_wse_interp = Interpolation(super_wse_x, super_wse_y, null_data=null_data,
                                             zero_tol=self.zero_tol)

        # handle vertical walls
        stations = []
        elevations = []
        mannings_n = []
        if vertical_wall_needed and self.left_vertical_elevation:
            stations.append(self.left_vertical_station)
            elevations.append(self.left_vertical_elevation)
            if self.mannings_n and len(self.mannings_n) > 0:
                mannings_n.append(self.mannings_n[0])
        stations.extend(self.stations)
        elevations.extend(self.elevations)
        if self.mannings_n and len(self.mannings_n) > 0:
            mannings_n.extend(self.mannings_n)
        if vertical_wall_needed and self.right_vertical_elevation:
            stations.append(self.right_vertical_station)
            elevations.append(self.right_vertical_elevation)
            if self.mannings_n and len(self.mannings_n) > 0:
                mannings_n.append(self.mannings_n[-1])

        # Check if we are starting underwater or with ineffective flow
        if elevations[0] < wse_1:
            flow_stations.append(stations[0])
            flow_elevations.append(elevations[0])
        if inef_index is not None and stations[0] == self.ineffective_flow_stations[inef_index][0]:
            have_ineffective_flow = True

        i = 1

        # Do analysis on all points
        while i < len(stations):
            # Update the wse elevation, if using super elevation
            if super_wse_interp is not None:
                wse_1 = super_wse_interp.interpolate_y(stations[i - 1])[0]
                wse_2 = super_wse_interp.interpolate_y(stations[i])[0]

            both_dry = False

            # both points below water
            if (elevations[i - 1] < wse_1) and (elevations[i] < wse_2):
                trap_area, trap_perimeter = self._compute_trapezoidal_flow_area(
                    stations[i - 1], stations[i], elevations[i - 1], elevations[i], wse_1,
                    wse_2)

                flow_area += trap_area
                perimeter += trap_perimeter
                # top_width += trap_top_width
                if stations[i] in thalwegs:
                    have_thalweg = True

                # Update polygon fill geometry
                flow_stations.append(stations[i])
                flow_elevations.append(elevations[i])

                # Manning n computations
                if calc_mannings_n and trap_perimeter > 0.0:
                    mannings_n_horton += trap_perimeter * mannings_n[i - 1]**1.5
                    mannings_n_pavlovskii += trap_perimeter * mannings_n[i - 1]**2.0
                    horton_perimeter.append(trap_perimeter)
                    horton_area.append(trap_area)
                    horton_mannings_n.append(mannings_n[i - 1])

            # one point dry, one point wet -or- one point at water level and other point is wet
            elif (elevations[i - 1] < wse_1) or (elevations[i] < wse_2) or \
                    (elevations[i - 1] == wse_1) and (elevations[i] < wse_2):
                tri_area, tri_perimeter, left, x_int, y_int = self._compute_triangular_flow_area(
                    stations[i - 1], stations[i], elevations[i - 1], elevations[i], wse_1, wse_2)

                # Update current polygon area, perimeter, and top width
                flow_area += tri_area
                perimeter += tri_perimeter
                # top_width += tri_top_width

                if calc_mannings_n and tri_perimeter > 0.0:
                    mannings_n_horton += tri_perimeter * mannings_n[i - 1]**1.5
                    mannings_n_pavlovskii += tri_perimeter * mannings_n[i - 1]**2.0
                    horton_perimeter.append(tri_perimeter)
                    horton_area.append(tri_area)
                    horton_mannings_n.append(mannings_n[i - 1])

                if left:
                    if stations[i] in thalwegs:
                        have_thalweg = True

                    # Add the intersection point
                    flow_stations.append(x_int)
                    flow_elevations.append(y_int)
                    left_int_x = x_int
                    left_int_y = y_int
                    left_int_wse = None

                    # Add the current point
                    flow_stations.append(stations[i])
                    flow_elevations.append(elevations[i])

                else:  # Right
                    # End Polygon (add data and reset current polygon datat)
                    flow_stations, flow_elevations, flow_area, top_width, perimeter, have_thalweg, left_int_x, \
                        left_int_y, left_int_wse, mannings_n_horton, mannings_n_pavlovskii, horton_perimeter, \
                        horton_area, horton_mannings_n = self._end_current_polygon(
                            stations, elevations, i, flow_stations, flow_elevations, flow_stations_by_poly,
                            flow_elevations_by_poly, flow_area_by_poly, top_width_by_poly, perimeter_by_poly,
                            thalweg_in_poly, poly_has_ineffective_flow, mannings_n_horton_by_poly,
                            mannings_n_pavlovskii_by_poly, horton_perimeter_by_poly, horton_area_by_poly,
                            horton_mannings_n_by_poly, wse_stations_in_poly, wse_elevations_in_poly, flow_area,
                            perimeter, top_width, have_thalweg, have_ineffective_flow, mannings_n_horton,
                            mannings_n_pavlovskii, horton_perimeter, horton_area, horton_mannings_n, x_int, y_int,
                            left_int_x, left_int_y, left_int_wse)

            # both points dry
            else:
                both_dry = True
                if inef_index is not None:
                    if stations[i] == self.ineffective_flow_stations[inef_index][0] or \
                            stations[i] == self.ineffective_flow_stations[inef_index][1]:
                        # Check if we enter/leave ineffective flow area, despite being dry (no poly to add)
                        have_ineffective_flow = not have_ineffective_flow
                        if not have_ineffective_flow:  # Check if need to increment index or skip future ineff checks
                            if inef_index < len(self.ineffective_flow_stations) - 1:
                                inef_index += 1
                            else:
                                inef_index = None

            if not both_dry and inef_index is not None:
                # Check for ineffective flow
                add_poly = False
                if not have_ineffective_flow:
                    # Check if we enter ineffective flow area
                    if stations[i] == self.ineffective_flow_stations[inef_index][0]:
                        add_poly = True
                else:
                    # Check if we leave ineffective flow area
                    if stations[i] == self.ineffective_flow_stations[inef_index][1]:
                        add_poly = True
                        if inef_index < len(self.ineffective_flow_stations) - 1:
                            inef_index += 1
                        else:
                            inef_index = None
                if add_poly:
                    # End Polygon (add data and reset current polygon data)
                    flow_stations, flow_elevations, flow_area, top_width, perimeter, have_thalweg, left_int_x, \
                        left_int_y, left_int_wse, mannings_n_horton, mannings_n_pavlovskii, horton_perimeter, \
                        horton_area, horton_mannings_n = self._end_current_polygon(
                            stations, elevations, i, flow_stations, flow_elevations, flow_stations_by_poly,
                            flow_elevations_by_poly, flow_area_by_poly, top_width_by_poly, perimeter_by_poly,
                            thalweg_in_poly, poly_has_ineffective_flow, mannings_n_horton_by_poly,
                            mannings_n_pavlovskii_by_poly, horton_perimeter_by_poly, horton_area_by_poly,
                            horton_mannings_n_by_poly, wse_stations_in_poly, wse_elevations_in_poly, flow_area,
                            perimeter, top_width, have_thalweg, have_ineffective_flow, mannings_n_horton,
                            mannings_n_pavlovskii, horton_perimeter, horton_area, horton_mannings_n, None, None,
                            left_int_x, left_int_y, left_int_wse, wse_2)
                    have_ineffective_flow = not have_ineffective_flow
                    left_int_x = stations[i]
                    left_int_y = elevations[i]
                    left_int_wse = wse_2
                    flow_stations.append(left_int_x)
                    flow_elevations.append(left_int_y)

            i += 1
        # end while

        # End Polygon (add data and reset current polygon data) IF it needs to be added
        add_int_point = left_int_x is not None and left_int_y is not None
        log_data = len(flow_stations_by_poly) == 0 or add_int_point

        flow_stations, flow_elevations, flow_area, top_width, perimeter, have_thalweg, left_int_x, \
            left_int_y, left_int_wse, mannings_n_horton, mannings_n_pavlovskii, horton_perimeter, horton_area, \
            horton_mannings_n = self._end_current_polygon(
                stations, elevations, i, flow_stations, flow_elevations, flow_stations_by_poly,
                flow_elevations_by_poly, flow_area_by_poly, top_width_by_poly, perimeter_by_poly, thalweg_in_poly,
                poly_has_ineffective_flow, mannings_n_horton_by_poly, mannings_n_pavlovskii_by_poly,
                horton_perimeter_by_poly, horton_area_by_poly, horton_mannings_n_by_poly, wse_stations_in_poly,
                wse_elevations_in_poly, flow_area, perimeter, top_width, have_thalweg, have_ineffective_flow,
                mannings_n_horton, mannings_n_pavlovskii, horton_perimeter, horton_area, horton_mannings_n,
                None, None, left_int_x, left_int_y, left_int_wse, add_int_point=add_int_point, log_data=log_data)

        # Sum results from polygons and remove areas outside thalwegs and ineffective flow
        self.cur_flow_stations = []
        self.cur_flow_elevations = []
        self.cur_ineffective_flow_poly_stations = []
        self.cur_ineffective_flow_poly_elevations = []
        self.cur_wse_stations = []
        self.cur_wse_elevations = []
        ineffective_flow_area = 0.0
        ineffective_flow_perimeter = 0.0
        ineffective_top_width = 0.0
        for index in range(len(flow_area_by_poly)):
            if check_thalwegs and thalweg_in_poly[index] or not check_thalwegs:
                # Add stations and elevations
                if len(self.cur_flow_stations) > 0:
                    self.cur_flow_stations.append('nan')
                    self.cur_flow_elevations.append('nan')
                    self.cur_wse_stations.append('nan')
                    self.cur_wse_elevations.append('nan')
                if len(flow_stations_by_poly[index]) <= 0:
                    continue  # Skip empty polygons
                self.cur_flow_stations.extend(flow_stations_by_poly[index])
                self.cur_flow_elevations.extend(flow_elevations_by_poly[index])
                if len(wse_stations_in_poly) > 0:
                    self.cur_wse_stations.extend(wse_stations_in_poly[index])
                    self.cur_wse_elevations.extend(wse_elevations_in_poly[index])

                if not poly_has_ineffective_flow[index]:
                    # Add to the total flow area, perimeter, and top width
                    flow_area += flow_area_by_poly[index]
                    perimeter += perimeter_by_poly[index]
                    top_width += top_width_by_poly[index]

                    mannings_n_horton += mannings_n_horton_by_poly[index]
                    mannings_n_pavlovskii += mannings_n_pavlovskii_by_poly[index]
                    horton_perimeter.extend(horton_perimeter_by_poly[index])
                    horton_area.extend(horton_area_by_poly[index])
                    horton_mannings_n.extend(horton_mannings_n_by_poly[index])

                else:
                    ineffective_flow_area += flow_area_by_poly[index]
                    ineffective_flow_perimeter += perimeter_by_poly[index]
                    ineffective_top_width += top_width_by_poly[index]

                    self.cur_ineffective_flow_poly_stations.extend(flow_stations_by_poly[index])
                    self.cur_ineffective_flow_poly_elevations.extend(flow_elevations_by_poly[index])

        # Correct only at the end!  negative areas are needed for closed shapes
        if flow_area < 0.0:
            flow_area = abs(flow_area)

        # Manning's n calculations
        if calc_mannings_n and perimeter > 0.0:

            # Horton Method
            horton_manning_value = (mannings_n_horton / perimeter)**(2.0 / 3.0)
            if 'Horton n' not in self.results:
                self.results['Horton n'] = []
            self.results['Horton n'].append(horton_manning_value)
            self.input_dict['calc_data']['Composite n']['calculator'].horton_manning_value = horton_manning_value

            # Pavlovskii Method
            pavlovskii_manning_value = mannings_n_pavlovskii**0.5 / (perimeter ** 0.5)
            if 'Pavlovskii n' not in self.results:
                self.results['Pavlovskii n'] = []
            self.results['Pavlovskii n'].append(pavlovskii_manning_value)
            self.input_dict['calc_data']['Composite n']['calculator'].pavlovskii_manning_value = \
                pavlovskii_manning_value

            # Lotter Method
            current_manning_n = 0.0
            current_perimeter = 0.0
            current_area = 0.0
            if len(horton_mannings_n):
                current_manning_n = horton_mannings_n[0]

            lotter_section_mannings_n = []
            lotter_area = []
            lotter_perimeter = []
            horton_n = 0.0

            horton_summation = 0.0

            for (h_manning_n, h_perimeter,
                 h_area) in zip(horton_mannings_n, horton_perimeter,
                                horton_area):
                if h_manning_n == current_manning_n:
                    # Summation
                    current_perimeter += h_perimeter
                    current_area += h_area
                    horton_summation += h_perimeter * h_manning_n**1.5
                else:
                    # Record previous section
                    lotter_perimeter.append(current_perimeter)
                    lotter_area.append(current_area)
                    horton_n = (horton_summation / current_perimeter)**(2.0 / 3.0)
                    lotter_section_mannings_n.append(horton_n)

                    # Start next section
                    current_manning_n = h_manning_n
                    current_perimeter = h_perimeter
                    current_area = h_area
                    # Summation
                    horton_summation = current_perimeter * h_manning_n**1.5

            # record previous section
            lotter_perimeter.append(current_perimeter)
            lotter_area.append(current_area)
            horton_n = (horton_summation / current_perimeter)**(2.0 / 3.0)
            lotter_section_mannings_n.append(horton_n)

            # Lotter Method
            lotter_summation = 0.0
            total_area = 0.0
            total_perimeter = 0.0
            for (l_manning_n, l_perimeter, l_area) in zip(lotter_section_mannings_n, lotter_perimeter, lotter_area):
                lotter_summation += l_perimeter * (abs(l_area) / l_perimeter)**(5.0 / 3.0) / l_manning_n
                total_area += l_area
                total_perimeter += l_perimeter

            lotter_manning_value = 0.0
            if total_perimeter > 0.0 and lotter_summation > 0.0:
                a_over_p = abs(total_area) / total_perimeter
                lotter_manning_value = total_perimeter * (a_over_p)**(5.0 / 3.0) / lotter_summation
            if 'Lotter n' not in self.results:
                self.results['Lotter n'] = []
            self.results['Lotter n'].append(lotter_manning_value)
            self.input_dict['calc_data']['Composite n']['calculator'].lotter_manning_value = lotter_manning_value

        if 'Flow area' not in self.results:
            self.results['Flow area'] = []
        if 'Wetted perimeter' not in self.results:
            self.results['Wetted perimeter'] = []
        if 'Hydraulic radius' not in self.results:
            self.results['Hydraulic radius'] = []
        if 'Top width' not in self.results:
            self.results['Top width'] = []

        if 'Effective flow area' not in self.results:
            self.results['Effective flow area'] = []
            self.results['Effective wetted perimeter'] = []
            self.results['Effective top width'] = []

        if 'Ineffective flow area' not in self.results:
            self.results['Ineffective flow area'] = []
            self.results['Ineffective wetted perimeter'] = []
            self.results['Ineffective top width'] = []

        if 'Vertical wall needed' not in self.results:
            self.results['Vertical wall needed'] = []

        self.results['Flow area'].append(flow_area + ineffective_flow_area)
        wetted_perimeter = perimeter + ineffective_flow_perimeter
        self.results['Wetted perimeter'].append(wetted_perimeter)
        hydraulic_radius = 0.0
        if wetted_perimeter > 0.0:
            hydraulic_radius = (flow_area + ineffective_flow_area) / wetted_perimeter
        self.results['Hydraulic radius'].append(hydraulic_radius)
        self.results['Top width'].append(top_width + ineffective_top_width)

        self.results['Effective flow area'].append(flow_area)
        self.results['Effective wetted perimeter'].append(perimeter)
        self.results['Effective top width'].append(top_width)

        self.results['Ineffective flow area'].append(ineffective_flow_area)
        self.results['Ineffective wetted perimeter'].append(ineffective_flow_perimeter)
        self.results['Ineffective top width'].append(ineffective_top_width)

        self.results['Vertical wall needed'].append(vertical_wall_needed)

        return flow_area, perimeter, top_width

    def _end_current_polygon(self, stations, elevations, index, flow_stations, flow_elevations,
                             flow_stations_by_poly, flow_elevations_by_poly, flow_area_by_poly,
                             top_width_by_poly, perimeter_by_poly, thalweg_in_poly, poly_has_ineffective_flow,
                             mannings_n_horton_by_poly, mannings_n_pavlovskii_by_poly,
                             horton_perimeter_by_poly, horton_area_by_poly, horton_mannings_n_by_poly,
                             wse_stations_in_poly, wses_in_poly, flow_area, perimeter, top_width,
                             have_thalweg, have_ineffective_flow, mannings_n_horton,
                             mannings_n_pavlovskii, horton_perimeter, horton_area, horton_mannings_n,
                             x_int, y_int, left_int_x, left_int_y, left_int_wse, wse_2=None, add_int_point=True,
                             log_data=True):
        """Ends the current polygon and prepares for the next one.

        Args:
            stations (list): List of station positions.
            elevations (list): List of elevations at the stations.
            index (int): Current index in the stations and elevations lists.
            flow_stations (list): List of stations for the current polygon.
            flow_elevations (list): List of elevations for the current polygon.
            flow_stations_by_poly (list): List of lists of stations for each polygon.
            flow_elevations_by_poly (list): List of lists of elevations for each polygon.
            flow_area_by_poly (list): List of areas for each polygon.
            top_width_by_poly (list): List of top widths for each polygon.
            perimeter_by_poly (list): List of perimeters for each polygon.
            thalweg_in_poly (list): List indicating if a thalweg is present in each polygon.
            poly_has_ineffective_flow (list): List indicating if ineffective flow is present in each polygon.
            mannings_n_horton_by_poly (list): List of Manning's n values using Horton's method for each polygon.
            mannings_n_pavlovskii_by_poly (list): List of Manning's n values using Pavlovskii's method for each polygon.
            horton_perimeter_by_poly (list): List of wetted perimeters using Horton's method for each polygon.
            horton_area_by_poly (list): List of areas using Horton's method for each polygon.
            horton_mannings_n_by_poly (list): List of Manning's n values using Horton's method for each polygon.
            wse_stations_in_poly (list): List of lists of stations where the water surface elevation intersects the
                channel.
            wses_in_poly (list): List of lists of elevations where the water surface elevation intersects the
                channel.
            flow_area (float): Area of the current polygon.
            perimeter (float): Perimeter of the current polygon.
            top_width (float): Top width of the current polygon.
            have_thalweg (bool): Indicates if a thalweg is present in the current polygon.
            have_ineffective_flow (bool): Indicates if ineffective flow is present in the current polygon.
            mannings_n_horton (float): Manning's n value using Horton's method for the current polygon.
            mannings_n_pavlovskii (float): Manning's n value using Pavlovskii's method for the current polygon.
            horton_perimeter (list): Wetted perimeter using Horton's method for the current polygon
            horton_area (list): Area using Horton's method for the current polygon.
            horton_mannings_n (list): Manning's n values using Horton's method for the current polygon.
            x_int (float): X coordinate of the intersection point.
            y_int (float): Y coordinate of the intersection point.
            left_int_x (float): X coordinate of the left intersection point.
            left_int_y (float): Y coordinate of the left intersection point.
            left_int_wse (float): Water surface elevation at the left intersection point.
            add_int_point (bool): Whether to add the intersection point to the polygon.
            log_data (bool): Whether to log the data for the polygon.

        Returns:
            (list, list, float, float, float, bool, float, float, float, float, list, list, list):
                Updated flow stations and elevations, area, perimeter, top width,
                thalweg presence flag, left intersection x, y, and wse  coordinates,
                left intersection water surface elevation, Manning's n values,
                and lists for Horton method data.
        """
        if add_int_point:

            left_x = None
            left_wse = None
            right_x = None
            right_wse = None

            if left_int_x is None:
                left_int_x = stations[index - 1]
                left_int_y = elevations[index - 1]

            if wse_2 is not None:
                right_x = stations[index]
                right_wse = wse_2
                flow_stations.append(right_x)
                flow_elevations.append(right_wse)

            # Add the intersection point (location between stations where wse intersects channel)
            # Only applies to Triangular flow areas
            if x_int is not None and y_int is not None and (x_int != left_int_x or y_int != left_int_y):
                right_x = x_int
                right_wse = y_int
                flow_stations.append(right_x)
                flow_elevations.append(right_wse)

            if left_int_wse is not None and abs(left_int_wse - left_int_y) > self.zero_tol:
                left_x = left_int_x
                left_wse = left_int_wse
                flow_stations.append(left_x)
                flow_elevations.append(left_wse)

            flow_stations.append(left_int_x)
            flow_elevations.append(left_int_y)
            if left_wse is None:
                left_x = left_int_x
                left_wse = left_int_y

            if right_x is None:
                if index < len(stations):
                    right_x = stations[index]
                else:
                    right_x = stations[-1]
            if right_wse is None:
                if index < len(elevations):
                    right_wse = elevations[index]
                else:
                    right_wse = elevations[-1]

            # top_width = abs(x_int - left_int_x)
            top_width = ((right_x - left_x)**2 + (right_wse - left_wse)**2)**0.5

            if abs(right_x - left_x) > self.zero_tol:
                wse_stations_in_poly.append([left_x, right_x])
                wses_in_poly.append([left_wse, right_wse])
            else:
                wse_stations_in_poly.append([])
                wses_in_poly.append([])

        # Log data by polygon
        if log_data:
            flow_stations_by_poly.append(flow_stations)
            flow_elevations_by_poly.append(flow_elevations)
            flow_area_by_poly.append(flow_area)
            top_width_by_poly.append(top_width)
            perimeter_by_poly.append(perimeter)
            thalweg_in_poly.append(have_thalweg)
            poly_has_ineffective_flow.append(have_ineffective_flow)

            mannings_n_horton_by_poly.append(mannings_n_horton)
            mannings_n_pavlovskii_by_poly.append(mannings_n_pavlovskii)
            horton_perimeter_by_poly.append(horton_perimeter)
            horton_area_by_poly.append(horton_area)
            horton_mannings_n_by_poly.append(horton_mannings_n)

        return [], [], 0.0, 0.0, 0.0, False, None, None, None, 0.0, 0.0, [], [], []

    def _compute_triangular_flow_area(self, station_1, station_2, elev_1, elev_2, wse_1, wse_2):
        """Computes the triangular flow area for a channel segment.

        Args:
            station_1 (float): The station at the first point.
            station_2 (float): The station at the second point.
            elev_1 (float): The elevation at the first point.
            elev_2 (float): The elevation at the second point.
            wse_1 (float): The water surface elevation at the first point.
            wse_2 (float): The water surface elevation at the second point.

        Returns:
            (float, float, bool, float, float): The area, perimeter, left side flag, x intersection, and y intersection.
        """
        # Calculate slopes
        dx = station_1 - station_2

        # Handle vertical line case
        if dx == 0:
            # If vertical, intersection is at the shared x coordinate
            # wse_1 and wse_2 should be the same in this case
            x_int = station_2
            y_int = wse_2

        elif elev_1 == wse_1:
            # If the channel elevation is equal to the water surface elevation at point 1,
            # then the intersection is at point 1
            x_int = station_1
            y_int = elev_1

        elif elev_2 == wse_2:
            # If the channel elevation is equal to the water surface elevation at point 2,
            # then the intersection is at point 2
            x_int = station_2
            y_int = elev_2

        else:
            slope_channel = (elev_1 - elev_2) / dx
            slope_wse = (wse_1 - wse_2) / dx

            # Calculate y-intercepts (when x = 0)
            b_channel = elev_2 - slope_channel * station_2
            b_wse = wse_2 - slope_wse * station_2

            # Find intersection: slope_1 * x + b_1 = slope_2 * x + b_2
            # Solve for x: (slope_1 - slope_2) * x = b_2 - b_1
            slope_diff = slope_channel - slope_wse

            x_int = 0.0
            if slope_diff != 0.0:
                x_int = (b_wse - b_channel) / slope_diff
            y_int = slope_channel * x_int + b_channel

        if elev_2 < wse_2:
            # "Left" side
            left = True
            h_tri = y_int - elev_2
            b_tri = station_2 - x_int
        else:
            # "Right" side
            left = False
            h_tri = y_int - elev_1
            b_tri = x_int - station_1

        # Computation of x position at water level using similar triangles
        flow_area = 0.5 * h_tri * b_tri
        wetted_perimeter = (b_tri**2 + h_tri**2)**(0.5)
        # top_width = dx
        # top_width = (b_tri**2 + wse_diff**2)**(0.5)

        return flow_area, wetted_perimeter, left, x_int, y_int

    def _compute_trapezoidal_flow_area(self, station_1, station_2, elev_1, elev_2, wse_1, wse_2):
        """Computes the trapezoidal flow area for a channel segment.

        Args:
            station_1 (float): The station at the first point.
            station_2 (float): The station at the second point.
            elev_1 (float): The elevation at the first point.
            elev_2 (float): The elevation at the second point.
            wse_1 (float): The water surface elevation at the first point.
            wse_2 (float): The water surface elevation at the second point.

        Returns:
            (float, float, float): The area, perimeter, and x_left value.
        """
        dx = station_2 - station_1
        h1 = abs(wse_2 - elev_1)
        h2 = abs(wse_1 - elev_2)

        flow_area = (((h1 + h2) * 0.5) * dx)
        wetted_perimeter = (dx**2 + (h1 - h2)**2)**0.5
        # top_width = (x_dist**2 + (wse_1 - wse_2)**2)**0.5

        return flow_area, wetted_perimeter

# # Following code comes from Hydraulic Toolbox, but is not currently used; it is efficient for determining
# # flow for defined shapes (e.g., rectangular, trapezoidal, triangular) where a derivative can be calculated.
# # It is left here for the day that this code is expanded to include defined shapes.
#     @staticmethod
#     def _newton_raphson(func_ptr, x1_lower_range, x2_upper_range, x_precision,
#                         target):
#         """This function is used in Hydraulic Toolbox for defined shapes (that can have a derivative).

#         I'm leaving this function for the day that this code is expanded to that (it should find a solution faster
#         than simple interpolation).

#         Args:
#             func_ptr (callable): Method to calculate flow and flow eq derivative. Should take a float argument and
#                 return a tuple of two floats.
#             x1_lower_range (float): Minimum allowable result
#             x2_upper_range (float): Maximum allowable result
#             x_precision (float): The allowable difference of result between iterations
#             target (float): The target result
#         """
#         # flow_eq_derivative = 0.0
#         # dx = 0.0
#         # dx_old = 0.0
#         # f = 0.0
#         # flow_higher = 0.0
#         # flow_lower = 0.0
#         #
#         # temp = 0.0
#         # x_higher = 0.0
#         # x_lower = 0.0
#         # rts = 0.0

#         # This equation uses the function divided by the function derived
#         flow_lower, flow_eq_derivative = func_ptr(x1_lower_range)
#         flow_higher, flow_eq_derivative = func_ptr(x2_upper_range)

#         if ((flow_lower > target) and (flow_higher > target)) or ((flow_lower < target) and (flow_higher < target)):
#             raise Exception("Root must be bracketed in NewtonRaphson")
#         if flow_higher == target:
#             return x2_upper_range
#         if flow_lower < target:
#             x_lower = x1_lower_range
#             x_higher = x2_upper_range
#         else:
#             x_higher = x1_lower_range
#             x_lower = x2_upper_range
#         rts = 0.25 * (x1_lower_range + x2_upper_range)
#         dx_old = abs(x2_upper_range - x1_lower_range)
#         dx = dx_old
#         f, flow_eq_derivative = func_ptr(rts)

#         for _ in range(1, 100):
#             if ((rts - x_higher) * flow_eq_derivative - f) * ((rts - x_lower) * flow_eq_derivative - f) > 0.0 or \
#                     abs(2.0 * f) > abs(dx_old * flow_eq_derivative):
#                 dx_old = dx
#                 dx = 0.5 * (x_higher - x_lower)
#                 rts = x_lower + dx
#                 if x_lower == rts:
#                     return rts
#             else:
#                 dx_old = dx
#                 dx = f / flow_eq_derivative
#                 temp = rts
#                 rts -= dx
#                 if temp == rts:
#                     return rts
#             if abs(dx) < x_precision:
#                 return rts
#             f, flow_eq_derivative = func_ptr(rts)
#             if f < target:
#                 x_lower = rts
#             else:
#                 x_higher = rts
#         raise Exception("Maximum number of iterations exceeded")
