"""CalcData for performing Channel geometry operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys
from uuid import uuid4

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_calc import ChannelGeometryCalc
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.cross_section import CrossSectionTable
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.embedment_data import EmbedmentData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.composite_manning_n_data import CompositeManningNData


open_gutter_shapes = ['curb and gutter', ]
open_channel_shapes = [
    'rectangle', 'round-cornered rectangle', 'trapezoidal', 'triangle',
    'round-bottomed triangle', 'parabola', 'cross-section',
]
closed_shapes = [
    'box', 'circle', 'elliptical', 'pipe arch', 'horseshoe',  # 'oblong',
    'cross-section',
]

open_shapes = []
open_shapes.extend(open_gutter_shapes)
open_shapes.extend(open_channel_shapes)

open_and_closed_channels_shapes = []
open_and_closed_channels_shapes.extend(open_channel_shapes)
open_and_closed_channels_shapes.extend(closed_shapes)

all_shapes = []
all_shapes.extend(open_gutter_shapes)
all_shapes.extend(open_channel_shapes)
all_shapes.extend(closed_shapes)

grate_types = [
    'grate',
    'curb opening',
    'slotted drain',
    'sweeper combo',
    'equal-length combo',
]


class ChannelGeometryData(CalcData):
    """A class that defines a channel and performs Channel geometry computations."""

    def __init__(self, stand_alone_calc: bool = True, allowed_shapes: str = 'all', default_shape: bool = None,
                 hide_depth: bool = False, app_data: AppData = None, model_name: str = None,
                 project_uuid: uuid4 = None):
        """Initializes the Channel geometry calculator.

        Args:
            stand_alone_calc (bool): Is the calculator a standalone calc or part of another class?
            allowed_shapes (string): sets which channel shapes the user can select: 'open', 'closed', 'both'
            default_shape (bool): which shape should be the default shape
            hide_depth (bool): Is this class being used for tailwater calculations?
            app_data (AppData): application data (settings, etc.).
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Channel Geometry Calculator'
        self.type = 'ChannelGeometryCalc'
        self.class_name = 'Channel Geometry Calculator'

        self.hide_depth = hide_depth

        _, self.zero_tol = self.get_setting('Zero tolerance')

        # Input
        self.open_gutter_shapes = open_gutter_shapes
        self.open_channel_shapes = open_channel_shapes
        self.closed_shapes = closed_shapes
        self.open_shapes = open_shapes
        self.open_and_closed_channels_shapes = open_and_closed_channels_shapes
        self.all_shapes = all_shapes
        d_shape = 'trapezoidal'

        shapes = self.all_shapes
        if allowed_shapes == 'open':
            shapes = self.open_shapes
        elif allowed_shapes == 'open gutter':
            shapes = self.open_gutter_shapes
        elif allowed_shapes == 'open channels':
            shapes = self.open_channel_shapes
        elif allowed_shapes == 'open and closed channels':
            shapes = self.open_and_closed_channels_shapes
        elif allowed_shapes == 'closed':
            shapes = self.closed_shapes
            d_shape = 'circle'
        if default_shape is not None:
            d_shape = default_shape

        self.calculator = ChannelGeometryCalc(
            stand_alone_calc=stand_alone_calc, allowed_shapes=allowed_shapes, hide_depth=hide_depth,
            zero_tol=self.zero_tol, open_shapes=self.open_shapes, closed_shapes=self.closed_shapes,
            all_shapes=self.all_shapes, shapes=shapes)

        self.input['Shape'] = Variable('Shape', "list", 0, shapes)
        self.input['Shape'].set_val(d_shape)
        self.input['Head'] = Variable('Head', "list", 0, ['Depth', 'Elevation'], complexity=2)

        self.input['Compute cross-section'] = Variable('Compute cross-section', "bool", True)

        # Cross-section data
        # soil_color = (102, 51, 0)
        # soil_fill_color = (215, 211, 199)
        # if self.theme is not None:
        #     soil_color = self.theme['Plot soil color']
        #     soil_fill_color = self.theme['Plot soil fill color']

        self.station_var = Variable('Station', 'float_list', 0, [0.0], precision=2,
                                    limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                    native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                      limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                      native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.manning_var = Variable("Manning's n", 'float_list', 0, [0.0], limits=[0.0, 1.0], precision=4,
                                    unit_type=['coefficient'], native_unit='coefficient', us_units=[['coefficient']],
                                    si_units=[['coefficient']])

        # TODO: Consider adding something similar to 'Cross-section' of Structure where the user can add
        # objects that would block flow and add to the Manning's n value

        name = 'Cross-section'
        cross_section = {
            'Station': self.station_var,
            'Elevation': self.elevation_var,
            "Manning's n": self.manning_var
        }

        self.input['Cross-section'] = Variable(name, 'table', CrossSectionTable(
            self.theme, name=name, plot_names=[name], input=cross_section, min_items=2, app_data=app_data,
            model_name=model_name, project_uuid=project_uuid))

        self.input["Manning's n by station"] = Variable("Manning's n by station", 'table', CrossSectionTable(
            self.theme, name=name, plot_names=[name], input=cross_section, min_items=2, app_data=app_data,
            model_name=model_name, project_uuid=project_uuid, define_cross_section=False))

        self.input['Left bank'] = Variable(
            'Left bank', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-self.max_value, self.max_value))

        self.input['Right bank'] = Variable(
            'Right bank', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-self.max_value, self.max_value))

        # Needed for circle
        self.input['Diameter'] = Variable(
            'Diameter', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Needed for rectangle and Trapezoid
        self.input['Bottom width'] = Variable(
            'Bottom width', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Needed for Trapezoid and Triangle
        self.input['Side slope 1'] = Variable(
            'Side slope 1', 'float', 0.0, [], precision=2, unit_type=['angle'], native_unit='H:1V',
            us_units=[['H:1V', '°']], si_units=[['H:1V', '°']], limits=(0.0, self.max_value))
        # Needed for Trapezoid and Triangle
        self.input['Side slope 2'] = Variable(
            'Side slope 2', 'float', 0.0, [], precision=2, unit_type=['angle'], native_unit='H:1V', us_units=[['H:1V']],
            si_units=[['H:1V']], limits=(0.0, self.max_value))
        # Needed for round-bottom trapezoid
        self.input['Side slope'] = Variable(
            'Side slope', 'float', 0.0, [], precision=2, unit_type=['slope'], native_unit='H:1V', us_units=[['H:1V']],
            si_units=[['H:1V']], limits=(0.0, self.max_value))
        # Needed for Round-cornered trapezoid and round-bottom trapezoid
        # Cannot be less than channel depth
        self.input['Corner radius'] = Variable(
            'Corner radius', 'float', 0, [], precision=3, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Needed for parabolic
        self.input['Top width'] = Variable(
            'Top width', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Needed for Curb
        self.input['Curb height'] = Variable(
            'Curb height', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Cross-slope pavement'] = Variable(
            'Cross-slope of pavement', 'float', 0, [], precision=4, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Define gutter cross-slope'] = Variable(
            'Define gutter cross-slope', 'bool', False)
        self.input['Cross-slope gutter'] = Variable(
            'Cross-slope of gutter', 'float', 0, [], precision=4, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Gutter width'] = Variable(
            'Width of gutter', 'float', 2.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Road width'] = Variable(
            'Distance from roadway crown to gutter', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
            note='Typically half width of road (from centerline to edge of road)')
        # TODO: Set this variable from the Grate Inlet Tool; modifies the flow ratios computed
        self.input['Grate width'] = Variable(
            'Width of grate', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Grate type'] = Variable('Grate type', 'list', 0, grate_types, )
        self.input["Use Manning's equation for flow computations"] = Variable(
            "Use Manning's equation for flow computations", "bool", False, complexity=2)

        self.input['Span'] = Variable(
            'Span', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Rise'] = Variable(
            'Rise', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Top radius'] = Variable(
            'Top radius', 'float', 0.0, [], precision=3, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Bottom radius'] = Variable(
            'Bottom radius', 'float', 0.0, [], precision=3, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # self.input['Alignment'] = Variable(
        #     'Alignment', "list", 0, ['vertical', 'horizontal'])
        # self.input['Oblong width'] = Variable(
        #     'Oblong width', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Add Elliptical (align either axes)
        # Add Pipe Arch
        self.input['b'] = Variable(
            'b', 'float', 0.0, [], precision=3, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        # Add Horseshoe shape
        # Add Oblong (align either axes)

        # Dimensions of the channel (for plotting and checking if overtops)
        # Needed for all but cross-section, circle
        self.input['Channel depth'] = Variable(
            'Channel depth', 'float', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['Thalweg invert elevation'] = Variable(
            'Thalweg invert elevation', 'float', 0, [], limits=(-self.max_value, self.max_value), complexity=2,
            precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.input['Embedment'] = Variable(
            'Embedment', 'class', EmbedmentData(rise=0.0, open_shape=False, app_data=app_data,
                                                model_name=model_name, project_uuid=project_uuid))

        # TODO: Implement Radius of curvature (change WSE with superelevation and shear stress),
        # thalweg stations (allows the user to specify geometry that would stay dry below the wse, for example behind a
        # dike/levee) and ineffective flow stations (allows the user to specify areas that are not contributing to flow)
        self.input['Curvature direction'] = Variable(
            'Curvature direction', 'list', 1, ['left', 'None', 'right'], complexity=5)

        self.input['Radius of curvature'] = Variable(
            'Radius of curvature', 'float', 0, [], limits=(-self.max_value, self.max_value), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
            complexity=5, note='Specify 0 for a straight channel.')

        self.input['Only fill flow areas with thalweg locations'] = Variable(
            'Only fill flow areas with thalweg locations', 'bool', False, complexity=5,
            note='If true, only fill flow areas that are connected to user-specified thalweg locations will be used.')

        self.input['Thalweg station(s)'] = Variable(
            'Thalweg station(s)', 'float_list', 0, [], limits=(-self.max_value, self.max_value),
            complexity=5, precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='Specify station(s) of thalweg(s) of channels (single channel or braided channels).')

        self.input['Ineffective flow stations'] = Variable(
            'Ineffective flow stations', 'list_of_lists', 0, [], limits=(-self.max_value, self.max_value),
            complexity=5, precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='Specify stations of ineffective flow areas (Each area needs a start and end station).')

        # The next variable isn't exposed to other calc's interface
        self.input['WSE'] = Variable(
            'Water surface elevation(s)', 'UserArray', UserArray(
                2, ['length'], 'ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
                select_name='specify elevation(s) as:', name_append='Elevation'),
            limits=(-self.max_value, self.max_value))
        self.input['Depths'] = Variable(
            'Depth(s)', 'UserArray', UserArray(
                2, ['length'], 'ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
                select_name='specify depth(s) as:', name_append='Depth'))

        self.input['Selected profile'] = Variable(
            'Selected profile', 'int', 0, [],
            note='This selects which flow profile to plot and show in the specified results.')
        self.plot_index_set = False

        # Plot options
        self.input['Plot options'] = {}
        self.add_plot_options()

        self.unknown = None
        self.channel_is_embedded = False
        self.allow_embedment = True

        self.reference_pdfs[
            'HEC-22'] = 'HEC-22. 4th Edition - Urban Drainage Design.pdf'
        self.reference_pdfs['HEC-18'] = (
            'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 50)

        # The next variable is not the full embedment depth!
        # It is the min embedment depth ( at the thalweg of low flow channel)
        self.min_embedment_depth = 0.0

        # Intermediate
        self.theme = self.get_theme()

        self.compute_prep_functions.extend([self.prepare_cross_section])
        self.compute_finalize_functions.extend([self.set_plotting_data])
        self.intermediate_to_copy.extend(
            ['zero_tol', 'open_gutter_shapes', 'open_channel_shapes', 'closed_shapes', 'open_shape',
             'open_and_closed_channels_shapes', 'all_shapes', 'stand_alone_calc',
             'stations', 'elevations', 'mannings_n', 'base_elevation', 'shape_is_closed',
             'rise_to_crown', 'unembedded_rise_to_crown', 'lowest_station', 'highest_station', 'high_elev',
             'low_elev', 'theta', 'radius', 'top_width_of_water', 'acot_z', 'full_flow_area', 'shapes_to_compute_geom',
             'wse_stations', 'wse_elevations', 'channel_stations', 'channel_elevations', 'embedment_stations',
             'embedment_elevations', 'stations', 'elevations', 'flow_stations', 'flow_elevations', 'thalweg_elevation',
             'thalweg_station', 'crown_elevation', 'crown_station', 'left_max_bank', 'left_max_bank_station',
             'right_max_bank', 'right_max_bank_station', 'left_vertical_station', 'left_vertical_elevation',
             'right_vertical_station', 'right_vertical_elevation', 'ineffective_flow_stations',
             'ineffective_flow_poly_stations', 'ineffective_flow_poly_elevations', 'superelev_top_width',
             'superelev_velocity', 'superelev_froude', 'unknown', 'channel_is_embedded', 'allow_embedment',
             'min_embedment_depth'])

        self.shapes_to_compute_geom = [
            'cross-section', 'curb and gutter', 'elliptical', 'pipe arch', 'horseshoe',
            # 'oblong'
        ]

        # Geometry intermediate (defined before the Composite n calc)
        # Without embedment
        self.channel_stations = []
        self.channel_elevations = []
        # only embedment
        self.embedment_stations = []
        self.embedment_elevations = []
        # channel shape (after embedment)
        self.stations = []
        self.elevations = []
        # flow depth polygon
        self.flow_stations = []
        self.flow_elevations = []
        # Ineffective flow polygons
        self.ineffective_flow_poly_stations = []
        self.ineffective_flow_poly_elevations = []

        self.mannings_n = []
        self.base_elevation = 0.0

        # True for closed shapes (culverts): Circular, box, elliptical, etc
        self.shape_is_closed = False
        self.rise_to_crown = 0.0
        self.unembedded_rise_to_crown = 0.0

        self.ineffective_flow_stations = []

        # Super elevation required data:
        self.superelev_top_width = None
        self.superelev_velocity = None
        self.superelev_froude = None

        # This is never shown to the user from here.  It is included for data access
        self.input['Composite n'] = Variable(
            "Manning's n", 'class',
            CompositeManningNData(
                shape='circle', open_shape=False, embedment_present=False, gradation_present=False,
                total_embedment=0.0, stations=self.stations, elevations=self.elevations,
                app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        # Horton and Einstein --EJones: CE EN 433 Digital Notes
        # Assumes each part has same mean velocity
        # self.mannings_n_horton = []
        # Pavlovskii, Muhlhofer, Einstein, and Banks --EJones: CE EN 433 Digital Notes
        # Assumes total resisting force equals summed resisting force
        # self.mannings_n_pavlovskii = []
        # Lotter --EJones: CE EN 433 Digital Notes
        # Assumes total discharge equals summed discharge
        # self.mannings_n_lotter = []

        self.lowest_station = 0.0
        self.highest_station = 0.0

        self.high_elev = None
        self.low_elev = None

        self.theta = 0.0
        self.radius = 0.0
        self.top_width_of_water = 0.0
        self.acot_z = 0.0

        self.full_flow_area = 0.0

        self.wse_stations = []
        self.wse_elevations = []

        self.thalweg_elevation = 0.0
        self.thalweg_station = 0.0
        self.crown_elevation = 0.0
        self.crown_station = 0.0
        self.left_max_bank = 0.0
        self.left_max_bank_station = 0.0
        self.right_max_bank = 0.0
        self.right_max_bank_station = 0.0

        self.left_vertical_station = None
        self.left_vertical_elevation = None
        self.right_vertical_station = None
        self.right_vertical_elevation = None

        # Results
        self.stand_alone_calc = stand_alone_calc
        self.results = {}

        self.results['Wetted perimeter'] = Variable(
            'Wetted perimeter', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Flow area'] = Variable(
            'Flow area', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])

        self.results['Top width'] = Variable(
            'Top width', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Gutter depression'] = Variable(
            'Gutter depression', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Eo'] = Variable(
            'Eo (Gutter flow to total flow)', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        # dimensionless ?
        self.results['Section factor'] = Variable(
            'Section factor', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Hydraulic radius'] = Variable(
            'Hydraulic radius', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Hydraulic depth'] = Variable(
            'Hydraulic depth', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Equivalent depth'] = Variable(
            'Equivalent depth', 'float_list', 0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        # self.results['WSE stations'] = Variable(
        #     'WSE stations', 'float_list', 0, [0.0, 0.0, 0.0], 2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length)

        # self.results['Fill stations'] = Variable(
        #     'Fill stations', 'float_list', 0, [0.0, 0.0, 0.0], 2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length)

        # self.results['Fill elevations'] = Variable(
        #     'Fill elevations', 'float_list', 0, [0.0, 0.0, 0.0], 2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['WSE stations'] = Variable(
            'WSE stations', 'list_of_lists', 0, [0.0, 0.0, 0.0], 2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Fill stations'] = Variable(
            'Fill stations', 'list_of_lists', 0, [0.0, 0.0, 0.0], 2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Fill elevations'] = Variable(
            'Fill elevations', 'list_of_lists', 0, [0.0, 0.0, 0.0], 2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Horton n'] = Variable(
            'composite n (Horton and Einstein Method)', 'float_list', 0.0, [], limits=[0.0, 1.0], precision=4,
            unit_type=['coefficient'], native_unit='coefficient', us_units=[['coefficient']],
            si_units=[['coefficient']])

        self.results['Pavlovskii n'] = Variable(
            'composite n (Pavlovskii, Muhlhofer, Einstein, and Banks Method)', 'float_list', 0.0, [], limits=[0.0, 1.0],
            precision=4, unit_type=['coefficient'], native_unit='coefficient', us_units=[['coefficient']],
            si_units=[['coefficient']])

        self.results['Lotter n'] = Variable(
            'composite n (Lotter Method)', 'float_list', 0.0, [], limits=[0.0, 1.0], precision=4,
            unit_type=['coefficient'], native_unit='coefficient', us_units=[['coefficient']],
            si_units=[['coefficient']])

        # TODO: Implement this calculation
        self.results["Reynold's number"] = Variable(
            "Reynold's number", 'float_list', 0.0, [], limits=[0.0, 1.0], precision=4, complexity=1,
            unit_type=['coefficient'], native_unit='coefficient', us_units=[['coefficient']],
            si_units=[['coefficient']])

        self.original_channel_wse = 0.0

        self.can_compute_shape = False
        self.can_compute = False

        self.warnings = []

        self.reference_pdfs['HEC-18'] = ('HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 137)

        self.plots['Cross-section'] = {}
        self.plots['Cross-section']['Legend'] = 'best'

    def add_plot_options(self):
        """Add plot options for a scenario.
        """
        station_vs_elevation = {}
        station_vs_elevation['Station'] = copy.copy(self.station_var)
        station_vs_elevation['Elevation'] = copy.copy(self.elevation_var)

        self.input_dict = {}
        self.input_dict['Channel'] = station_vs_elevation

        self.min_y = sys.float_info.max

        # Load theme colors
        self.theme = self.get_theme()
        channel_bank_color = self.theme['Plot channel bank line color']  # Bank lines, not channel lines

        structure_color = self.theme['Plot structure color']
        # structure_fill_color = self.theme['Plot structure fill color']

        # embankment_color = self.theme['Plot embankment color']
        # embankment_fill_color = self.theme['Plot embankment fill color']

        soil_color = self.theme['Plot soil color']
        soil_fill_color = self.theme['Plot soil fill color']

        wsp_color = self.theme['Plot WSP color']
        wsp_fill_color = self.theme['Plot WSP fill color']

        # normal_color = self.theme['Plot normal depth color']
        # normal_fill_color = self.theme['Plot normal depth fill color']

        # critical_color = self.theme['Plot critical depth color']
        # critical_fill_color = self.theme['Plot critical depth fill color']

        # hydraulic_jump_color = self.theme['Plot hydraulic jump color']
        # hydraulic_jump_color = self.theme['Plot hydraulic jump fill color']

        # Need plot items for thalweg, point properties (toes, abutment? user-defined?) Auto determine?
        # Need plot items for ineffective flow areas, n values?

        self.input['Plot options']['Cross-section'] = Variable(
            'Plot options', 'class', PlotOptions(
                'Cross-section', input_dict=self.input_dict, show_series=True, app_data=self.app_data,
                model_name=self.model_name, project_uuid=self.project_uuid), complexity=1)

        # Lines
        self.input['Plot options']['Cross-section'].get_val().set_plot_line_options(
            index=0, name='Channel bank',
            line_intercept=None, line_alignment='vertical', line_color=channel_bank_color, linetype='dotted',
            line_width=1.5, text_color=channel_bank_color, text_alignment='top')

        # Series
        self.input['Plot options']['Cross-section'].get_val().set_plot_series_options(
            related_index=0, index=0, x_axis='Station', y_axis='Elevation', name='Flow', line_color=wsp_color,
            linetype=None, line_width=2, fill_below_line=None, fill_color=wsp_fill_color, pattern=None,
            density=None)
        self.input['Plot options']['Cross-section'].get_val().set_plot_series_options(
            related_index=0, index=1, x_axis='Station', y_axis='Elevation', name='Embedment', line_color=soil_color,
            linetype=None, line_width=2, fill_below_line=None, fill_color=soil_fill_color, pattern='earth',
            density=None)
        self.input['Plot options']['Cross-section'].get_val().set_plot_series_options(
            related_index=0, index=2, x_axis='Station', y_axis='Elevation', name='Channel',
            line_color=structure_color, linetype=None, line_width=4, fill_below_line=None, fill_color=None,
            pattern=None, density=None)

    def set_plotting_data(self, index=None):
        """Sets up the plotting data for the cross section.

        Args:
            index (int): Not used.
        """
        if index is None:
            index = self.input['Selected profile'].get_val() - 1  # Index for users is 1-based
        self.calculator.set_plotting_data(index)

    def prepare_cross_section(self):
        """Prepare the cross-section data before computing geometry.
        """
        self.input['Cross-section'].value.compute_data()
        self.input["Manning's n by station"].value.cs_stations = copy.copy(self.input[
            'Cross-section'].value.stations)
        self.input["Manning's n by station"].value.cs_elevations = copy.copy(self.input[
            'Cross-section'].value.elevations)
        self.input["Manning's n by station"].value.cs_mannings_n = copy.copy(self.input[
            'Cross-section'].value.mannings_n_long)
        self.input["Manning's n by station"].value.compute_data()

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        if not self.hide_depth:
            input_vars['Shape'] = self.input['Shape']

        add_channel_depth = False
        if self.input['Shape'].get_val() == 'cross-section':
            input_vars['Cross-section'] = self.input['Cross-section']
            # input_vars['Original stations'] = self.input['Original stations']
            # input_vars['Original elevations'] = self.input['Original elevations']
        elif self.input['Shape'].get_val() == 'circle':
            if unknown != 'Diameter':
                input_vars['Diameter'] = self.input['Diameter']
        elif self.input['Shape'].get_val() == 'triangle':
            if unknown != 'Side slope':
                input_vars['Side slope 1'] = self.input['Side slope 1']
                input_vars['Side slope 2'] = self.input['Side slope 2']
        elif self.input['Shape'].get_val() == 'round-bottomed triangle':
            if unknown != 'Side slope':
                input_vars['Side slope'] = self.input['Side slope']
            if unknown != 'Corner radius':
                input_vars['Corner radius'] = self.input['Corner radius']
        elif self.input['Shape'].get_val() == 'trapezoidal':
            if unknown != 'Side slope':
                input_vars['Side slope 1'] = self.input['Side slope 1']
                input_vars['Side slope 2'] = self.input['Side slope 2']
            if unknown != 'Bottom width':
                input_vars['Bottom width'] = self.input['Bottom width']
        elif self.input['Shape'].get_val() == 'parabola':
            if unknown != 'Top width':
                input_vars['Top width'] = self.input['Top width']
            if unknown != 'Channel depth':
                add_channel_depth = True
        elif self.input['Shape'].get_val() == 'rectangle':
            if unknown != 'Bottom width':
                input_vars['Bottom width'] = self.input['Bottom width']
        elif self.input['Shape'].get_val() == 'round-cornered rectangle':
            if unknown != 'Bottom width':
                input_vars['Bottom width'] = self.input['Bottom width']
            if unknown != 'Corner radius':
                input_vars['Corner radius'] = self.input['Corner radius']
        elif self.input['Shape'].get_val() == 'curb and gutter':
            input_vars['Curb height'] = self.input['Curb height']
            input_vars['Cross-slope pavement'] = self.input['Cross-slope pavement']
            input_vars['Define gutter cross-slope'] = self.input['Define gutter cross-slope']
            if input_vars['Define gutter cross-slope'].get_val():
                input_vars['Cross-slope gutter'] = self.input['Cross-slope gutter']
            input_vars['Gutter width'] = self.input['Gutter width']
            input_vars['Road width'] = self.input['Road width']
            # input_vars['Width of spread'] = self.input['Width of spread']
            input_vars["Use Manning's equation for flow computations"] = self.input[
                "Use Manning's equation for flow computations"]
        elif self.input['Shape'].get_val() in ['box', 'elliptical']:
            input_vars['Span'] = self.input['Span']
            input_vars['Rise'] = self.input['Rise']
        elif self.input['Shape'].get_val() == 'pipe arch':
            input_vars['Span'] = self.input['Span']
            input_vars['Rise'] = self.input['Rise']
            input_vars['Bottom radius'] = self.input['Bottom radius']
            input_vars['Top radius'] = self.input['Top radius']
            input_vars['Corner radius'] = self.input['Corner radius']
            input_vars['b'] = self.input['b']
            pass
        elif self.input['Shape'].get_val() == 'horseshoe':
            input_vars['Rise'] = self.input['Rise']
        # elif self.input['Shape'].get_val() == 'oblong':
        #     input_vars['Oblong width'] = self.input['Oblong width']
        #     input_vars['Corner radius'] = self.input['Corner radius']
        #     input_vars['Alignment'] = self.input['Alignment']
        # self.input['Head']
        # self.input['Compute cross-section']

        if self.input['Shape'].get_val() in [
                'triangle', 'round-bottomed triangle', 'trapezoidal',
                'parabola', 'rectangle', 'round-cornered rectangle'
        ]:
            add_channel_depth = True
        if not self.hide_depth and add_channel_depth:
            input_vars['Channel depth'] = self.input['Channel depth']

        if self.allow_embedment and self.input['Shape'].get_val() != 'curb and gutter':
            input_vars['Embedment'] = self.input['Embedment']
            if self.shape_is_closed and self.unembedded_rise_to_crown > 0.0:
                input_vars['Embedment'].get_val().input['Embedment depth'].limits = (0.0, self.unembedded_rise_to_crown)

        if self.input['Shape'].get_val() not in 'cross-section':
            input_vars['Thalweg invert elevation'] = self.input['Thalweg invert elevation']

        if self.stand_alone_calc:
            input_vars['Head'] = self.input['Head']
            if input_vars['Head'].get_val() == 'Depth':
                input_vars['Depths'] = self.input['Depths']
                if self.rise_to_crown < 0.0:
                    self.determine_if_shape_is_closed()
                if self.shape_is_closed and self.rise_to_crown > 0.0:
                    embed = self.input['Embedment'].get_val().input['Embedment depth'].get_val()
                    input_vars['Depths'].get_val().limits = (0.0, self.unembedded_rise_to_crown - embed)
                else:
                    input_vars['Depths'].get_val().limits = (0.0, sys.float_info.max)
            else:
                input_vars['WSE'] = self.input['WSE']

        input_vars['Curvature direction'] = self.input['Curvature direction']
        if self.input['Curvature direction'].get_val() != 'None':
            input_vars['Radius of curvature'] = self.input['Radius of curvature']
        input_vars['Only fill flow areas with thalweg locations'] = self.input[
            'Only fill flow areas with thalweg locations']
        if self.input['Only fill flow areas with thalweg locations'].get_val():
            input_vars['Thalweg station(s)'] = self.input['Thalweg station(s)']
        input_vars['Ineffective flow stations'] = self.input['Ineffective flow stations']

        num_calcs = 0
        # if 'Flows' in input_vars:
        #     num_calcs = len(input_vars['Flows'].get_val().get_result())
        if 'Depths' in input_vars:
            num_calcs = len(input_vars['Depths'].get_val().get_result())
        elif 'WSE' in input_vars:
            num_calcs = len(input_vars['WSE'].get_val().get_result())
        # elif 'Width of spread' in input_vars:
        #     num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            self.input['Selected profile'].limits = (1, num_calcs)
            input_vars['Selected profile'] = self.input['Selected profile']
            if not self.plot_index_set:
                self.input['Selected profile'].set_val(num_calcs)  # Default to last index
                self.plot_index_set = True
        else:
            self.input['Selected profile'].limits = (0, num_calcs)
            self.input['Selected profile'].set_val(0)
            self.plot_index_set = False

        return input_vars

    def get_results_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        result_vars = {}

        if not self.get_can_compute():
            return result_vars

        result_vars['Wetted perimeter'] = self.results['Wetted perimeter']
        result_vars['Flow area'] = self.results['Flow area']
        result_vars['Top width'] = self.results['Top width']
        result_vars['Section factor'] = self.results['Section factor']
        result_vars['Hydraulic radius'] = self.results['Hydraulic radius']
        result_vars['Hydraulic depth'] = self.results['Hydraulic depth']
        # result_vars['Composite n computation method'] = \
        #     self.input['Composite n'].results['Composite n computation method']
        # result_vars['Composite n value'] = self.input['Composite n'].results['Composite n value']
        if len(self.results['Equivalent depth'].get_val()):
            result_vars['Equivalent depth'] = self.results['Equivalent depth']

        # result_vars['WSE stations'] = self.results['WSE stations']
        if len(self.results['Gutter depression'].get_val()):
            result_vars['Gutter depression'] = self.results[
                'Gutter depression']
        if len(self.results['Eo'].get_val()):
            result_vars['Eo'] = self.results['Eo']
        if len(self.results['Horton n'].get_val()):
            result_vars['Horton n'] = self.results['Horton n']
        if len(self.results['Pavlovskii n'].get_val()):
            result_vars['Pavlovskii n'] = self.results['Pavlovskii n']
        if len(self.results['Lotter n'].get_val()):
            result_vars['Lotter n'] = self.results['Lotter n']

        return result_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
                result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        input_vars = self.get_input_group(unknown)
        # if 'Flows' in input_vars:
        #     num_calcs = len(input_vars['Flows'].get_val().get_result())
        if 'Depths' in input_vars:
            num_calcs = len(input_vars['Depths'].get_val().get_result())
        elif 'WSE' in input_vars:
            num_calcs = len(input_vars['WSE'].get_val().get_result())
        # elif 'Width of spread' in input_vars:
        #     num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            profile_index = self.input['Selected profile'].get_val() - 1  # Index for users is 1-based
            results_group = self.get_results_group(unknown)
            results_vars['Selected profile results'] = {}
            for item in results_group:
                results_vars['Selected profile results'][item] = copy.copy(results_group[item])
                if len(results_vars['Selected profile results'][item].value_options) > profile_index:
                    results_vars['Selected profile results'][item].set_val(
                        results_vars['Selected profile results'][item].get_val()[profile_index])

            results_vars['Full results'] = self.get_results_group(unknown)
        else:
            results_vars['Results'] = self.get_results_group(unknown)

        return results_vars

    def check_warnings(self, wses=None, critical_wses=None):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        warning = None
        if wses and critical_wses:
            warning = self.check_for_vertical_wall_warning(wses, critical_wses)
        if warning:
            self.warnings.append(warning)
        return self.warnings

    def _set_current_value(self, value, unknown):
        """Sets the current value of a specific variable (used to update when determining the unknown value.

        Args:
            value (float): value to set the variable to
            unknown (string): variable to set

        Returns:
            bool: True if successful
        """
        for variable in self.input.values():
            if unknown == variable.name:
                variable.set_val(value)
                return True
        return False

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.results['Wetted perimeter'].set_val([])
        self.results['Flow area'].set_val([])
        self.results['Top width'].set_val([])
        self.results['Gutter depression'].set_val([])
        self.results['Eo'].set_val([])
        self.results['Section factor'].set_val([])
        self.results['Hydraulic radius'].set_val([])
        self.results['Hydraulic depth'].set_val([])
        self.results['Equivalent depth'].set_val([])
        # self.results['WSE stations'].set_val([])
        # self.results['Fill stations'].set_val([])
        # self.results['Fill elevations'].set_val([])
        self.wse_stations = []
        self.wse_elevations = []
        self.flow_stations = []
        self.flow_elevations = []
        self.results['Horton n'].set_val([])
        self.results['Pavlovskii n'].set_val([])
        self.results['Lotter n'].set_val([])
        if 'Composite n computation method' in self.results:
            self.results['Composite n computation method'].set_val([])
        if 'Composite n value' in self.results:
            self.results['Composite n value'].set_val([])
