"""EmbedmentData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules


class EmbedmentCalc(Calculator):
    """Provides a class that will define the site data of a culvert barrel."""

    def __init__(self, rise, open_shape):
        """Initializes theEmbedment Data class.

        Args:
            rise (float): rise of the channel
            open_shape (bool): whether the channel is open
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__()

        # Input
        self.rise = rise
        self.open_shape = open_shape

    def build_embedment_method_list(self):
        """Build the list of embedment methods."""
        # build list for the ways to specify manning's n value
        self.embedment_method_list = []
        if not self.open_shape:
            self.embedment_method_list.append('embed by percentage of rise')
        self.embedment_method_list.append('embed by depth')
        if self.open_shape:
            self.embedment_method_list.append('embed evenly across channel')

    def get_embedment_depth(self):
        """Get the embedment depth.

        Returns:
            float: embedment depth
        """
        self._compute_data()
        return self.input_dict['calc_data']['Embedment depth']

    def set_no_embedment_depth(self):
        """Set the embedment depth to zero."""
        self.input_dict['calc_data']['Embedment depth'] = 0.0
        self.input_dict['calc_data']['Embedment percent'] = 0.0

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        if self.input_dict['calc_data']['Embedment entry'] in ['embed by percentage of rise']:
            percent_of_depth = self.input_dict['calc_data']['Embedment percent']
            embedment_depth = self.rise * percent_of_depth / 100.0
            self.input_dict['calc_data']['Embedment depth'] = embedment_depth
        else:
            embedment_depth = self.input_dict['calc_data']['Embedment depth']
            if self.rise > 0.0:
                percent_of_depth = embedment_depth / self.rise * 100.0
            else:
                percent_of_depth = 0.0
            self.input_dict['calc_data']['Embedment percent'] = percent_of_depth
        return True
