"""Provides a class that will Handle user-selected for boundary conditions."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class ShapeDatabase():
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """

    shape_database_filename = Path(__file__).resolve().parent / 'shape_db.dat'
    shape_database = None

    culv_params_to_chan_params = {
        'Br': 'Bottom radius',
        'Tr': 'Top radius',
        'Cr': 'Corner radius',
        'B': 'b',
    }

    inlet_configuration_list = [
        'beveled edge', 'mitered to conform to slope', 'square edge with headwall', 'thin edge projecting'
    ]

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Shape Database Class.

        Args:
            app_data (AppData): the application data
            model_name (str): the model name
            project_uuid (str): the project unique identifier
        """
        if app_data is not None and ShapeDatabase.shape_database is None:
            ShapeDatabase.shape_database = app_data.model_dict['Hydraulic Toolbox'].read_manager.hdf5_to_dict(
                ShapeDatabase.shape_database_filename)

        self.app_data = app_data
        self.model_name = model_name
        self.project_uuid = project_uuid

        self.cat_name = None
        self.cats = []
        self.sel_cat = None
        self.sub_cat_name = None
        self.sel_sub_cat = None
        self.sub_cats = []
        self.sel_items = []
        self.sel_item = None
        self.geometries = []
        self.geometry = None

        self.prev_sel_cat = -1
        self.prev_sel_sub_cat = -1

        # self.x = []
        # self.y_top = []
        # self.y_bottom = []

        # self.cs_x = []
        # self.cs_y = []

        # self.channel_n = None

        self.prev_shape = None
        # self.dim_list = []
        # self.show_select = False
        self.prev_embed = None

        # self.culvert_icon_index = None

        self.unit_converter = ConversionCalc(self.app_data)

        # self.update_lists()

    # def update_lists(self, inlet_control=None, geometry=None, cur_shape=None, cur_mat=None):
    def update_lists(self, inlet_control):
        """Update lists based on other variable values."""
        if ShapeDatabase.shape_database is None:
            self.warnings['Shapes database not loaded'] = 'Shapes database not loaded; only manual polynomial '
            'calculations available'
            return

        manning_list = ['Mannings', 'Mannings n', 'Manning n']

        # # Get shortcuts
        # # Note that mat and configuration are intentionally variables. value_options will be updated
        # if inlet_control is None:
        #     if cur_shape is None or cur_mat is None or geometry is None:
        #         return geometry
        #     geometry = self.update_geometry(cur_shape, cur_mat, manning_list, geometry)
        #     return geometry

        if len(inlet_control.input['Culvert shape'].value_options) == 0:
            inlet_control.input['Culvert shape'].value_options = ShapeDatabase.shape_database['Shape Names']

        geometry = inlet_control.input['Geometry'].value
        composite_n = inlet_control.input['Composite n'].get_val()
        cur_shape = inlet_control.input['Culvert shape'].get_val()
        mat = inlet_control.input['Culvert material']
        configuration = inlet_control.input['Inlet configuration']
        inlet_type = inlet_control.input['Inlet type'].get_val()
        compute_geometry = inlet_control.input['Compute geometry'].get_val()
        specify_dimensions = inlet_control.input['Specify dimensions'].get_val()

        use_interp_inlet_configs = False
        inlet_control.show_select = False

        # Update the culvert shapes
        # if len(inlet_control.input['Culvert shape'].value_options) == 0:
        #     inlet_control.input['Culvert shape'].value_options = ShapeDatabase.shape_database['Shape Names']
        # cur_shape = self.input['Culvert shape'].get_val()
        # compute_geometry = self.input['Specify dimensions'].get_val()
        # specify_dimensions = self.input['Compute geometry'].get_val()
        geometry.input['Shape'].set_val(inlet_control.calculator.get_channel_shape_from_culvert_shape(
            cur_shape, compute_geometry, specify_dimensions))
        inlet_control.culvert_icon_index = ShapeDatabase.shape_database[cur_shape]['Icon']
        if cur_shape == 'User Defined':
            use_interp_inlet_configs = True

        # Update Embedment
        embed_percent = geometry.input['Embedment'].value.get_embedment_percent()
        composite_n.embedment_present = embed_percent > 0.0
        composite_n.update_lists()

        if self.prev_embed != embed_percent:
            if composite_n.input['n entry'].get_val() != 'specify composite n value':
                composite_n.input['n entry'].set_val('specify channel and embedment n values')
                composite_n.input['Embedment n'].set_val(0.035)

        self.prev_embed = embed_percent

        # Update material
        mat_names = ShapeDatabase.shape_database[cur_shape]['Material Names']
        if mat.value_options != mat_names:
            mat.value_options = mat_names
            mat.value = 0
        cur_mat = mat.get_val()
        if cur_mat in composite_n.input['Material'].value_options:
            composite_n.input['Material'].set_val(cur_mat)
        if 'Mannings' in ShapeDatabase.shape_database[cur_shape][cur_mat]:
            inlet_control.channel_n = ShapeDatabase.shape_database[cur_shape][cur_mat]['Mannings']
            if isinstance(inlet_control.channel_n, list):
                inlet_control.channel_n = inlet_control.channel_n[0]

        # Culvert type is irrelevant if we ignore tapered culverts
        # type_names = ShapeDatabase.shape_database[cur_shape]['Material Data'][cur_mat]['Type Names']
        # # We are not currently planning to keep tapered culverts, so remove those types
        # # This is prefered to removing them from the file, in case we want to implement them later
        # for type in type_names[:]:
        #     if 'tapered' in type.lower():
        #         type_names.remove(type)
        # type_names.append('single broken-back')
        # type_names.append('double broken-back')
        # if mat.value_options != type_names:
        #     self.input['Culvert type'].value_options = type_names
        #     self.input['Culvert type'].value = 0
        # cur_type = self.input['Culvert type'].get_val()

        # Update inlet configuration
        embed_polys = {}
        if inlet_type == 'best available' and not use_interp_inlet_configs:
            if embed_percent > 0.0:
                if 'Embedment Depth' in ShapeDatabase.shape_database[cur_shape] and 'Dimension Coefficients' in \
                        ShapeDatabase.shape_database[cur_shape]['Embedment Depth']:
                    percents = ShapeDatabase.shape_database[cur_shape]['Embedment Depth']['Dimension Coefficients'][
                        'Percent']
                    for long_percent in percents:
                        percent = f'{round(long_percent, 6)}'
                        inlet_config_names = ShapeDatabase.shape_database[cur_shape]['Embedment Depth'][
                            'Dimension Coefficients'][percent]['Inlet Names']
                        if configuration.value_options != inlet_config_names:
                            configuration.value_options = inlet_config_names
                            configuration.value = 0
                        for inlet_name in inlet_config_names:
                            if inlet_name not in embed_polys:
                                embed_polys[inlet_name] = {}
                            if percent not in embed_polys[inlet_name]:
                                embed_polys[inlet_name][percent] = {}
                            poly_names = ShapeDatabase.shape_database[cur_shape]['Embedment Depth'][
                                'Dimension Coefficients'][percent][inlet_name]['Polynomial Coefficients']['Names']
                            poly_values = ShapeDatabase.shape_database[cur_shape]['Embedment Depth'][
                                'Dimension Coefficients'][percent][inlet_name]['Polynomial Coefficients']['Values']
                            for poly_name, poly_value in zip(poly_names, poly_values):
                                embed_polys[inlet_name][percent][poly_name] = poly_value
            else:
                inlet_config_names = ShapeDatabase.shape_database[cur_shape][cur_mat]['Entrance Types']['Straight'][
                    'Inlet Names']
                ShapeDatabase.shape_database['Circular']['Concrete']['Entrance Types']['Straight']['Inlet Names']
                if configuration.value_options != inlet_config_names:
                    configuration.value_options = inlet_config_names
                    configuration.value = 0
        elif inlet_type in ['interpolation: circular or elliptical', 'interpolation: arch or embedded'] or \
                use_interp_inlet_configs:
            if configuration.value_options != ShapeDatabase.inlet_configuration_list:
                configuration.value_options = ShapeDatabase.inlet_configuration_list
                configuration.value = 0
        cur_inlet_config = configuration.get_val()

        # Update polynomial coefficients
        inlet_control.polynomial_available = False
        if embed_percent > 0.0:
            if len(embed_polys) > 0 and cur_inlet_config in embed_polys:
                inlet_control.polynomial_available = True
                if len(embed_polys[cur_inlet_config]) == 1:
                    # Single embedment percent available
                    [percent_key] = embed_polys[cur_inlet_config]
                    for poly_name, poly_value in embed_polys[cur_inlet_config][percent_key].items():
                        self.set_poly_value(inlet_control, poly_name, poly_value)
                else:
                    # Multiple embedment percents available
                    percent_keys = list(embed_polys[cur_inlet_config].keys())
                    percent_keys.sort(key=lambda x: float(x))
                    poly_name_values = {}
                    for percent_key in percent_keys:
                        for poly_name, poly_value in embed_polys[cur_inlet_config][percent_key].items():
                            if poly_name not in poly_name_values:
                                poly_name_values[poly_name] = []
                            poly_name_values[poly_name].append(poly_value)
                    _, null_data = self.app_data.get_setting('Null data', model_name=self.model_name,
                                                             project_uuid=self.project_uuid)
                    _, zero_tol = self.app_data.get_setting('Zero tolerance', model_name=self.model_name,
                                                            project_uuid=self.project_uuid)
                    for poly_name, poly_values in poly_name_values.items():
                        interp = Interpolation([float(k) for k in percent_keys], poly_values, null_data=null_data,
                                               zero_tol=zero_tol)
                        interp_value, _ = interp.interpolate_y(embed_percent / 100.0, extrapolate=False)
                        self.set_poly_value(inlet_control, poly_name, interp_value)
        elif inlet_type == 'best available' and cur_inlet_config in \
                ShapeDatabase.shape_database[cur_shape][cur_mat]['Entrance Types']['Straight'] and \
                'Polynomial Coefficients' in \
                ShapeDatabase.shape_database[cur_shape][cur_mat]['Entrance Types']['Straight'][cur_inlet_config]:
            inlet_control.polynomial_available = True
            poly_names = ShapeDatabase.shape_database[cur_shape][cur_mat]['Entrance Types']['Straight'][
                cur_inlet_config]['Polynomial Coefficients']['Names']
            poly_values = ShapeDatabase.shape_database[cur_shape][cur_mat]['Entrance Types']['Straight'][
                cur_inlet_config]['Polynomial Coefficients']['Values']
            for name, value in zip(poly_names, poly_values):
                self.set_poly_value(inlet_control, name, value)

        # Update which interpolation numbers to use
        inlet_control.use_cir_ell_interp = False
        if cur_shape in ['circular', 'elliptical', 'pipe arch', 'horseshoe']:
            inlet_control.use_cir_ell_interp = True

        # Update variable for specific way to apply polynomial equation
        inlet_control.use_qad_poly = False
        if cur_shape in ['Concrete Open-Bottom Arch']:
            inlet_control.use_qad_poly = True

        # Update the geometry
        geometry = self.update_geometry(cur_shape, cur_mat, manning_list, geometry, inlet_control)

        # Update the dimensions (diameter, rise, span, etc)
        inlet_control.dim_list = {}
        for dim_name in ShapeDatabase.shape_database[cur_shape]['Dimension Names']:
            if dim_name == 'Embedment depth':
                continue
            if dim_name in geometry.input:
                inlet_control.dim_list[dim_name] = copy.deepcopy(geometry.input[dim_name])
        if cur_shape != self.prev_shape and self.prev_shape is not None:
            for dim_name in inlet_control.dim_list:
                english_unit = ShapeDatabase.shape_database[cur_shape][dim_name]['English Units']
                metric_unit = ShapeDatabase.shape_database[cur_shape][dim_name]['Metric Units']
                value = None
                if 'Default Value' in ShapeDatabase.shape_database[cur_shape][dim_name]:
                    value = ShapeDatabase.shape_database[cur_shape][dim_name]['Default Value']
                elif 'Value' in ShapeDatabase.shape_database[cur_shape][dim_name]:
                    value = ShapeDatabase.shape_database[cur_shape][dim_name]['Value']
                if english_unit in inlet_control.dim_list[dim_name].available_us_units[0]:
                    inlet_control.dim_list[dim_name].selected_us_unit = english_unit
                if metric_unit in inlet_control.dim_list[dim_name].available_si_units[0]:
                    inlet_control.dim_list[dim_name].selected_si_unit = metric_unit
                if value == -1.0:
                    inlet_control.dim_list[dim_name].read_only = True
                    # self.show_select = True

        if self.prev_shape != cur_shape:
            inlet_control.input['Composite n'].value.input['Composite n'].set_val(inlet_control.channel_n)
        inlet_control.input['Composite n'].value.input['Channel n'].set_val(inlet_control.channel_n)

        self.prev_shape = cur_shape

        return

    def set_poly_value(self, inlet_control, name: str, value: float):
        """Set the polynomial coefficient value in the inlet control.

        Args:
            inlet_control (InletControlData): the inlet control data
            name (str): name of the polynomial coefficient
            value (float): value of the polynomial coefficient
        """
        if name == 'KE':
            inlet_control.input['KE'].set_val(value)
        elif name == 'SR':
            inlet_control.input['SR'].set_val(value)
        elif name == 'A':
            inlet_control.input['A'].set_val(value)
        elif name == 'BS':
            inlet_control.input['BS'].set_val(value)
        elif name == 'C':
            inlet_control.input['C'].set_val(value)
        elif name == 'DIP':
            inlet_control.input['DIP'].set_val(value)
        elif name == 'EE':
            inlet_control.input['EE'].set_val(value)
        elif name == 'F':
            inlet_control.input['F'].set_val(value)

    def update_geometry(self, cur_shape: str, cur_mat: str, manning_list: list, geometry, inlet_control):
        """Update the geometry based on other variable values.

        Args:
            cur_shape (str): current shape
            cur_mat (str): current material
            manning_list (list): list of manning n parameter names
            geometry (InputGroup): geometry input group
            inlet_control (InletControlData): the inlet control data
        """
        cat_key = None
        sub_cat_key = None
        if 'Categories' in ShapeDatabase.shape_database[cur_shape][cur_mat]:
            inlet_control.show_select = True
            if cur_shape == 'User Defined':
                inlet_control.show_select = False
            _, units_system = self.app_data.get_setting('Selected unit system')
            if units_system == 'U.S. Customary Units' or units_system == 'Both':
                unit_type = 'English'
            else:
                unit_type = 'Metric'
            # Categories
            cat_index = 1
            cat_name = None
            cats = []
            while f'Category {cat_index}' in ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories']:
                if 'Group Name' in ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][
                        f'Category {cat_index}']:
                    cat_name = ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][
                        f'Category {cat_index}']['Group Name']
                    if isinstance(cat_name, list):
                        cat_name = cat_name[0]
                if 'Name' in ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][
                        f'Category {cat_index}']:
                    name = ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][
                        f'Category {cat_index}']['Name']
                    if isinstance(name, list):
                        name = name[0]
                    cats.append(name)
                cat_index += 1
            if self.cats != cats or self.sel_cat is None:
                self.cats = cats
                self.cat_name = cat_name
                if self.cat_name is None:
                    self.cat_name = 'Category 1'
                self.sel_cat = 0
                self.sel_sub_cat = 0
                self.sel_item = 0

            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Category'].name = self.cat_name
            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Category'].value_options = self.cats
            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Category'].set_val(self.sel_cat)

            # Sub categories
            index = 0
            sub_cat_index = 1
            sub_cat_name = None
            sub_cats = []
            index_to_sub_cat = {}
            cat_key = f'Category {self.sel_cat + 1}'
            while f'Sub Category {sub_cat_index}' in ShapeDatabase.shape_database[cur_shape][cur_mat][
                    'Categories'][cat_key]:
                if 'Group Name' in ShapeDatabase.shape_database[cur_shape][cur_mat][
                        'Categories'][cat_key][f'Sub Category {sub_cat_index}']:
                    sub_cat_name = ShapeDatabase.shape_database[cur_shape][cur_mat][
                        'Categories'][cat_key][f'Sub Category {sub_cat_index}']['Group Name']
                    if isinstance(sub_cat_name, list):
                        sub_cat_name = sub_cat_name[0]
                if 'Name' in ShapeDatabase.shape_database[cur_shape][cur_mat][
                        'Categories'][cat_key][f'Sub Category {sub_cat_index}']:
                    units = ShapeDatabase.shape_database[cur_shape][cur_mat][
                        'Categories'][cat_key][f'Sub Category {sub_cat_index}']['Units']
                    if units == unit_type or unit_type in units:
                        index_to_sub_cat[index] = sub_cat_index
                        sub_cat = ShapeDatabase.shape_database[cur_shape][cur_mat][
                            'Categories'][cat_key][f'Sub Category {sub_cat_index}']['Name']
                        if isinstance(sub_cat, list):
                            sub_cat = sub_cat[0]
                        sub_cats.append(sub_cat)
                        index += 1
                else:
                    index_to_sub_cat[index] = sub_cat_index
                    sub_cats.append(f'Sub Category {sub_cat_index}')
                    index += 1
                sub_cat_index += 1
            if self.sub_cats != sub_cats or self.sel_sub_cat is None:
                self.sub_cats = sub_cats
                self.sub_cat_name = sub_cat_name
                if self.sub_cat_name is None:
                    self.sub_cat_name = 'Sub Category 1'
                self.sel_sub_cat = 0
                self.sel_item = 0

            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Sub Category'].name = self.sub_cat_name
            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Sub Category'].value_options = self.sub_cats
            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Sub Category'].set_val(self.sel_sub_cat)

            # Items  - Which size is selected
            sel_sub_cat_index = index_to_sub_cat[self.sel_sub_cat]
            if unit_type == 'Metric':
                sel_sub_cat_index -= 1

            sub_cat_key = f'Sub Category {sel_sub_cat_index}'
            if self.sel_item is None:
                self.sel_item = 0

            # Get the geometry (cross-section points)
            if 'Geometry' in ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][cat_key][
                    sub_cat_key] and 'Data' in ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][cat_key][
                    sub_cat_key]['Geometry']:
                self.geometries = ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][cat_key][sub_cat_key][
                    'Geometry']['Data']

                # self.geometry = self.geometries[self.sel_item]

                inlet_control.x = []
                inlet_control.y_top = []
                inlet_control.y_bottom = []
                # First index is point number, second index is x, y_top, or y_bottom, third index list_index
                if len(self.geometries) > 0:
                    if self.sel_item >= len(self.geometries[0][0]):
                        self.sel_item = len(self.geometries[0][0]) - 1
                for i in range(len(self.geometries)):
                    inlet_control.x.append(self.geometries[i][0][self.sel_item])
                    inlet_control.y_top.append(self.geometries[i][1][self.sel_item])
                    inlet_control.y_bottom.append(self.geometries[i][2][self.sel_item])

                inlet_control.cs_x = []
                inlet_control.cs_y = []
                for i in range(len(inlet_control.x)):
                    inlet_control.cs_x.append(inlet_control.x[i])
                    inlet_control.cs_y.append(inlet_control.y_top[i])
                for i in range(len(inlet_control.x) - 1, -1, -1):
                    inlet_control.cs_x.append(inlet_control.x[i])
                    inlet_control.cs_y.append(inlet_control.y_bottom[i])

                geometry.input['Cross-section'].value.input['Table options'].value.input[
                    'Number of items'].set_val(len(inlet_control.cs_x))
                geometry.input['Cross-section'].value.input['Data input'].value.input[
                    'Station'].value_options = inlet_control.cs_x
                geometry.input['Cross-section'].value.input['Data input'].value.input[
                    'Elevation'].value_options = inlet_control.cs_y

        inlet_control.selected_params = {}
        self.parameters = {}
        if 'Parameters' in ShapeDatabase.shape_database[cur_shape][cur_mat]:
            self.parameters = ShapeDatabase.shape_database[cur_shape][cur_mat]['Parameters']
        elif 'Categories' in ShapeDatabase.shape_database[cur_shape][cur_mat] and \
                'Parameters' in ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][cat_key][
                sub_cat_key]:
            self.parameters = ShapeDatabase.shape_database[cur_shape][cur_mat]['Categories'][cat_key][
                sub_cat_key]['Parameters']

        num_items = 0
        test_param = ''
        if 'Parameter Names' in self.parameters:
            test_param = self.parameters['Parameter Names'][0]
        if test_param in self.parameters:
            num_items = len(self.parameters[test_param]['Data'])

        self.items = []
        spans = None
        rises = None
        span_unit = ''
        precision = 2
        if 'Span' in self.parameters and 'Rise' in self.parameters:
            unit_converter = ConversionCalc(self.app_data)
            span_unit = geometry.input['Span'].selected_us_unit
            span_native_unit = self.parameters['Span']['English Units'][0]
            precision = geometry.input['Span'].precision
            # Always have span and rise in the same units - just match span
            # rise_unit = self.geometry.input['Rise'].selected_us_unit
            rise_native_unit = self.parameters['Rise']['English Units'][0]
            spans = self.parameters['Span']['Data']
            rises = self.parameters['Rise']['Data']
            spans = [unit_converter.convert_units(span_native_unit, span_unit, val)[1] for val in spans]
            rises = [unit_converter.convert_units(rise_native_unit, span_unit, val)[1] for val in rises]
        if len(self.parameters) != num_items:
            if spans is not None and rises is not None:
                self.items = [f'Value {i + 1} — {spans[i]:.{precision}f} ({span_unit}) X {rises[i]:.{precision}f} '
                              f'({span_unit})' for i in range(num_items)]
            else:
                self.items = [f'Value {i + 1}' for i in range(num_items)]
            inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
                'Selected row'].value_options = self.items

        inlet_control.input['Select culvert size'].value.input['Table options'].value.input[
            'Selected row'].set_val(self.sel_item)

        if len(self.parameters) > 0:
            parameter_names = self.parameters['Parameter Names']
            for param in parameter_names:
                inlet_control.selected_params[param] = self.parameters[param]['Data'][self.sel_item]
                chan_param = param
                if param in self.culv_params_to_chan_params:
                    chan_param = self.culv_params_to_chan_params[param]
                if chan_param in geometry.input:
                    orig_us_units = geometry.input[chan_param].selected_us_unit
                    orig_si_units = geometry.input[chan_param].selected_si_unit
                    units = self.parameters[param]['English Units']
                    if isinstance(units, list):
                        units = units[0]
                    value = inlet_control.selected_params[param]
                    if units in geometry.input[chan_param].available_us_units[0]:
                        geometry.input[chan_param].selected_us_unit = units
                    if f'{param} 2' in self.parameters:
                        value_2 = self.parameters[f'{param} 2']['Data'][self.sel_item]
                        units_2 = self.parameters[f'{param} 2']['English Units']
                        if isinstance(units_2, list):
                            units_2 = units_2[0]
                        _, value_2_in_1 = self.unit_converter.convert_units(units_2, units, value_2)
                        value += value_2_in_1
                    geometry.input[chan_param].set_val(value, app_data=self.app_data)
                    if self.prev_sel_cat == self.sel_cat and \
                            self.prev_sel_sub_cat == self.sel_sub_cat:
                        geometry.input[chan_param].selected_us_unit = orig_us_units
                        geometry.input[chan_param].selected_si_unit = orig_si_units
                if param in manning_list:
                    inlet_control.channel_n = inlet_control.selected_params[param]

            self.prev_sel_cat = self.sel_cat
            self.prev_sel_sub_cat = self.sel_sub_cat

        return geometry
