"""Weir Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math

# 2. Third party modules
from sortedcontainers import SortedDict

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class WeirCalc(Calculator):
    """A class that defines a weir and performs weir computations."""

    def __init__(self, ):
        """Initializes the weir calculator."""
        super().__init__()

        self.broadCrestedPavedX = [0.0, .15, .35, .45, .65, 1.0, 2.0, 3.0, 4.0]
        self.broadCrestedPavedY = [2.88, 2.95, 3.0, 3.02, 3.03, 3.031, 3.034, 3.037, 3.04]
        self.broadCrestedGravelX = [0.0, .2, .52, .9, 1.45, 2.0, 2.3, 3.12, 4.0]
        self.broadCrestedGravelY = [2.5, 2.6, 2.7, 2.8, 2.9, 2.975, 3.0, 3.035, 3.04]
        self.sharperCrestedPavedX = [.15, .16, .18, .2, .22, .24, .26, .28, .3]
        self.sharperCrestedPavedY = [3.05, 3.055, 3.065, 3.075, 3.08, 3.085, 3.087, 3.09, 3.095]
        self.sharperCrestedGravelX = [.15, .16, .18, .2, .22, .24, .26, .28, .3]
        self.sharperCrestedGravelY = [2.95, 2.96, 2.98, 3.0, 3.025, 3.045, 3.06, 3.08, 3.095]
        self.k1ReductionFactorPavedX = [.815, .845, .87, .915, .948, .962, .979, .988, 1.0]
        self.k1ReductionFactorPavedY = [1.0, .985, .965, .9, .8, .7, .6, .5, 0.0]
        self.k1ReductionFactorGravelX = [.75, .8, .84, .88, .928, .949, .963, .975, 1.0]
        self.k1ReductionFactorGravelY = [1.0, .985, .965, .9, .8, .7, .6, .5, 0.0]

        self.original_results = [
            'Flows', 'WSE', 'Depths', 'WSE stations', 'Tailwater',
            'Tailwater elevations', 'Weir coefficient', 'Weir coefficients', 'Applied coefficients'
        ]

    def _get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Returns:
            bool: True if can compute
        """
        result = True
        _, self.zero_tol = self.get_data('Zero tolerance')

        self.unknown = self.input_dict['calc_data']['Calculate']

        found = False

        weir_type = self.input_dict['calc_data']['Weir type']

        length_of_known = 0
        if self.unknown != 'Head':
            depths = self.input_dict['calc_data']['Depths']
            wses = self.input_dict['calc_data']['WSE']
            if self.input_dict['calc_data']['Head'] == 'Depth':
                length_of_known = len(depths)
                if hasattr(depths, '__len__') and len(depths) < 1:
                    self.warnings['depths'] = "Please enter a depth"
                    result = False
                else:
                    for depth in depths:
                        if depth > 0.0:
                            found = True
                            break
                    if not found:
                        self.warnings['depths2'] = "Please enter a positive, non-zero depth"
                        result = False
            elif self.input_dict['calc_data']['Head'] == 'Elevation':
                crest_base_elevation = self.input_dict['calc_data']['Thalweg invert elevation']
                if weir_type == 'rectangular (sharp-crested)':
                    crest_base_elevation += self.input_dict['calc_data']['Weir height']
                elif weir_type == 'irregular':
                    crest_elevations = self.input_dict['calc_data']['Irregular weir crest geometry']['Crest elevations']
                    crest_base_elevation = min(crest_elevations)
                if hasattr(wses, '__len__') and len(wses) < 1:
                    self.warnings['wse'] = "Please enter a water surface elevation"
                    result = False
                else:
                    length_of_known = len(wses)
                    for wse in wses:
                        if wse > 0.0:
                            if wse < crest_base_elevation:
                                self.warnings['wse3'] = "Please enter a water surface elevation greater than the crest "
                                self.warnings['wse3'] += "base elevation of " + str(crest_base_elevation)
                                result = False
                            else:
                                found = True
                                break
                    if not found:
                        self.warnings['wse2'] = "Please enter a positive, non-zero water surface elevation"
                        result = False

        if self.unknown != 'Flows':
            flows = self.input_dict['calc_data']['Flows']
            if hasattr(flows, '__len__') and len(flows) < 1:
                self.warnings['flows'] = "Please enter a flow"
                result = False
            elif length_of_known > 0 and length_of_known != len(flows):
                self.warnings['flows3'] = "Please enter the same number of flows as depths"
                result = False
            else:
                found = False
                length_of_known = len(flows)
                for flow in flows:
                    if flow > 0.0:
                        found = True
                        break
                if not found:
                    self.warnings['flows2'] = "Please enter a positive, non-zero flow"
                    result = False

        if self.input_dict['calc_data']['Weir orientation'] == 'Parallel (flow can bypass weir)':
            froude = self.input_dict['calc_data']['Froude']
            if hasattr(froude, '__len__') and len(froude) < 1:
                self.warnings['flows'] = "Please enter froude numbers"
                result = False
            elif length_of_known != len(froude):
                self.warnings['froude3'] = "Please enter the same number of Froude numbers as flows or depths"
                result = False
            else:
                found = False
                for fr in froude:
                    if fr <= 0.0:
                        found = True
                        break
                if found:
                    self.warnings['froude'] = "Please enter all Froude numbers as positive, non-zero values"
                    result = False

        if weir_type in self.triangular_shapes:
            var_list = []
            if weir_type == 'V-notch - User-defined':
                select_angle = self.input_dict['calc_data']['Select angle or coef']
                if select_angle == 'v-notch angle':
                    var_list.append('Notch angle')
                else:
                    var_list.append('Weir coefficient')
            if self.unknown in var_list:
                var_list.remove(self.unknown)
            result = self.check_float_vars_to_greater_zero(var_list=var_list, result=result)
        if weir_type in self.rectangular_shapes:
            var_list = ['Weir length',]
            if weir_type == 'rectangular (sharp-crested)':
                var_list.append('Weir height')
            elif weir_type == 'rectangular (broad-crested)':
                var_list.append('Weir coefficient')
            if self.unknown in var_list:
                var_list.remove(self.unknown)
            result = self.check_float_vars_to_greater_zero(var_list=var_list, result=result)
        elif weir_type in self.trapezoidal_shapes:
            var_list = ['Weir length',]
            if weir_type != 'Cipolletti':
                var_list.append('Side slope')
            if self.unknown in var_list:
                var_list.remove(self.unknown)
            result = self.check_float_vars_to_greater_zero(var_list=var_list, result=result)
        elif weir_type in self.irregular_shapes:
            crest_elevations = self.input_dict['calc_data']['Irregular weir crest geometry']['Crest elevations']
            crest_stations = self.input_dict['calc_data']['Irregular weir crest geometry']['Crest stations']
            if len(crest_elevations) < 2 or len(crest_stations) < 2:
                self.warnings['irregular'] = \
                    "Please enter at least two crest elevations and stations for the irregular weir geometry"
                result = False
            elif min(crest_stations) == max(crest_stations):
                self.warnings['irregular2'] = "Please enter irregular weir geometry"
                result = False

        return result

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.warnings = {}

        self.results['Flows'] = []
        self.results['WSE'] = []
        self.results['Depths'] = []
        self.results['WSE stations'] = []
        self.results['Tailwater'] = []
        self.results['Tailwater elevations'] = []
        self.results['Weir coefficient'] = []
        self.results['Weir coefficients'] = []
        self.results['Applied coefficients'] = []
        self.perpendicular_coefficients = []

        # Remove results from other items (self.unknowns)
        for key in list(self.results.keys()):
            if key not in self.original_results:
                del self.results[key]

    def get_low_and_high_value(self, unknown=None, tw_e=None):
        """Returns the low and high value for a given unknown.

        Args:
            unknown (string): unknown variable that we are calculating
            tw_e (float): the tailwater elevation

        Returns:
            low_val (float): lowest value for the unknown variable
            high_val (float): highest value for the unknown variable
        """
        low_val = 0.0
        high_val = 0.0
        if not unknown:
            unknown = self.input_dict['calc_data']['Calculate']

        result, num_divisions = self.get_data('Number of divisions for interpolation curve')
        if not result or num_divisions < 2:
            num_divisions = 10
        unreasonably_large_number = 1e6

        if unknown in ['Depth', 'Head', 'Tailwater', 'Tailwater elevations']:
            thalweg = self.input_dict['calc_data']['Thalweg invert elevation']
            low_val = thalweg
            if unknown in ['Depth', 'Head']:
                low_val = max(thalweg, tw_e if tw_e is not None else 0.0)
            high_val = unreasonably_large_number + thalweg
            dense_start = low_val
            dense_end = low_val + 20.0
            step = (dense_end - dense_start) / (num_divisions - 3)
            list_range = [low_val + i * step for i in range(num_divisions - 3)]
            list_range.append(low_val + 50.0)
            list_range.append(low_val + 200.0)
            list_range.append(high_val)
            return low_val, high_val, list_range
        elif unknown in ['Flows']:
            num_dense = num_divisions - 2
            dense_start = 1.0
            dense_end = 5000.0
            power = 4.0
            high_val = unreasonably_large_number
            dense_range = [
                dense_start + (dense_end - dense_start) * ((i / (num_dense - 1)) ** power)
                for i in range(num_dense)
            ]
            list_range = [low_val] + dense_range + [high_val]
            return low_val, high_val, list_range
        else:
            if unknown in ['Weir coefficient']:
                low_val = self.zero_tol
                high_val = 1.0
                step = (high_val - 0.4) / (num_divisions - 2)
                list_range = [low_val] + [0.4 + i * step for i in range(num_divisions - 2)] + [high_val]
            elif unknown in ['Side slope',]:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [10 ** (math.log10(0.01) + (math.log10(100) - math.log10(0.01))
                                     * (i / (num_divisions - 1))) for i in range(num_divisions)]
            elif unknown in ['Weir length',]:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                # list_range = [low_val] + [0.5 + (2500 - .5) * (i / (num_divisions - 1)) for i in range(num_divisions)
                #                           ] + [high_val]
                num_dense = num_divisions - 2
                dense_start = 0.25
                dense_end = 2500.0
                power = 4.0
                high_val = unreasonably_large_number
                dense_range = [
                    dense_start + (dense_end - dense_start) * ((i / (num_dense - 1)) ** power)
                    for i in range(num_dense)
                ]
                list_range = [low_val] + dense_range + [high_val]
            elif unknown in ['Weir width', 'Weir height',]:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [low_val] \
                    + [0.01 + (10 - 0.01) * (i / (num_divisions - 2)) for i in range(num_divisions - 1)] \
                    + [100] + [high_val]
            elif unknown in ['Notch angle']:
                low_val = self.zero_tol
                high_val = 180.0 - self.zero_tol
                step = (high_val - low_val) / (num_divisions - 1)
                list_range = [low_val + i * step for i in range(num_divisions)]
            else:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [low_val] + [0.5 + (100 - 0.5) * (i / (num_divisions - 1)) for i in range(num_divisions)
                                          ] + [high_val]

        return low_val, high_val, list_range

    def update_weir_coefficient(self, head_above_crest: float, froude: float):
        """Updates the weir coefficient.

        Args:
            head_above_crest (float): The head above the weir crest
            froude (float): The froude number
        """
        self.coefficient = 0.0

        weir_type = self.input_dict['calc_data']['Weir type']
        if weir_type == 'irregular':
            pass  # We will compute a coefficient per segment
        elif weir_type == 'rectangular':
            if self.prior_weir_type != 'rectangular':
                self.input_dict['calc_data']['Weir coefficient'] = 3.1
            self.coefficient = self.input_dict['calc_data']['Weir coefficient']
        elif weir_type == 'rectangular (sharp-crested)':
            # HEC-22 Third editon, page 8-23, pdf page 325, EQ 8-19
            weir_height = self.input_dict['calc_data']['Weir height']
            self.coefficient = 3.27
            if weir_height > 0.0:
                self.coefficient = 3.27 + 0.4 * (head_above_crest / weir_height)
        elif weir_type == 'rectangular (broad-crested)':
            # HEC-22 Third editon, page 8-26, pdf page 328, EQ 8-22
            self.coefficient = self.input_dict['calc_data']['Weir coefficient']
            if self.coefficient < 2.34:
                self.coefficient = 2.34
            if self.coefficient > 3.32:
                self.coefficient = 3.32
        elif weir_type == 'Cipolletti':
            # TODO: Verify the following coefficient is from Hyd TB; think it should be different like at URL:
            # http://irrigation.wsu.edu/Content/Calculators/Water-Measurements/Cipolletti-Weir.php
            self.coefficient = 3.367
        elif weir_type == 'trapezoidal':
            self.coefficient = 3.367
        elif weir_type == 'V-notch - 90 degrees':
            self.coefficient = 2.5
        elif weir_type == 'V-notch - 60 degrees':
            self.coefficient = 1.443
        elif weir_type == 'V-notch - 45 degrees':
            self.coefficient = 1.035
        elif weir_type == 'V-notch - 22.5 degrees':
            self.coefficient = 0.497
        elif weir_type == 'V-notch - User-defined':
            if self.input_dict['calc_data']['Select angle or coef'] == 'v-notch angle':
                notch_angle = self.input_dict['calc_data']['Notch angle']
                self.coefficient = 2.54 * math.tan(
                    math.radians(notch_angle) / 2.0)
            else:
                self.coefficient = self.input_dict['calc_data']['Weir coefficient']

        if self.input_dict['calc_data']['Weir orientation'] == 'Parallel (flow can bypass weir)':
            self.perpendicular_coefficient = copy.copy(self.coefficient)
            one_sqrt_3 = 1 / math.sqrt(3)
            if self.zero_tol < froude < one_sqrt_3:
                self.coefficient *= ((1 - 3 * froude ** 2) / (2 + 3 * froude ** 2)) ** (0.5)
            else:
                self.warnings['froude2'] = "Froude number is outside of currently supported range"
            # # froude = self.input_dict['calc_data']['Froude']

        self.prior_weir_type = weir_type

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.unknown = self.input_dict['calc_data']['Calculate']
        weir_type = self.input_dict['calc_data']['Weir type']

        crest_base_elevation = self.input_dict['calc_data']['Thalweg invert elevation']
        if weir_type == 'rectangular (sharp-crested)':
            crest_base_elevation += self.input_dict['calc_data']['Weir height']
        if weir_type == 'irregular':
            crest_elevations = self.input_dict['calc_data']['Irregular weir crest geometry'][
                'Crest elevations']
            crest_base_elevation = min(crest_elevations)

        # Convert elevations to depths
        if self.input_dict['calc_data']['Head'] == 'Elevation':
            elevations = self.input_dict['calc_data']['WSE']
            depths = []
            for elevation in elevations:
                depth = elevation - crest_base_elevation
                depths.append(depth)
            self.input_dict['calc_data']['Depths'] = depths
            tw_elevations = self.input_dict['calc_data']['Tailwater elevations']
            tws = []
            for tw_elevation in tw_elevations:
                tw = tw_elevation - crest_base_elevation
                tws.append(tw)
            self.input_dict['calc_data']['Tailwater'] = tws
        # Convert depths to elevations
        else:
            depths = self.input_dict['calc_data']['Depths']
            elevations = []
            for depth in depths:
                elevation = depth + crest_base_elevation
                elevations.append(elevation)
            self.input_dict['calc_data']['WSE'] = elevations
            tws = self.input_dict['calc_data']['Tailwater']
            self.input_dict['calc_data']['Tailwater elevations'] = []
            for tw in tws:
                tw_elevation = tw + crest_base_elevation
                self.input_dict['calc_data']['Tailwater elevations'].append(tw_elevation)

        tailwater = self.input_dict['calc_data']['Tailwater elevations']
        min_tw = min(min(tailwater), crest_base_elevation, 0.0) if len(tailwater) > 0 else min(crest_base_elevation,
                                                                                               0.0)
        velocity = self.input_dict['calc_data']['Velocity']
        froudes = self.input_dict['calc_data']['Froude']
        if self.unknown == 'Flows':
            wses = self.input_dict['calc_data']['WSE']

            # Extend tailwater
            if len(tailwater) < len(wses):
                tailwater = tailwater + [min_tw] * (len(wses) - len(tailwater))

            # Extend velocity
            if len(velocity) < len(wses):
                velocity = velocity + [0.0] * (len(wses) - len(velocity))

            for index in range(len(wses)):
                self.cur_wse = wses[index]
                vel = velocity[index]
                tw_e = crest_base_elevation
                froude = 0.0
                if index < len(froudes):
                    froude = froudes[index]
                if len(tailwater) > index:
                    tw_e = tailwater[index]
                cur_flow = self.compute_weir_flow_from_depth(weir_type, self.cur_wse, tw_e, vel, crest_base_elevation,
                                                             froude=froude)

                self.assign_results(wse=self.cur_wse, tw_e=tw_e, flow=cur_flow,
                                    crest_base_elevation=crest_base_elevation, unknown=self.unknown)

            return True
        # if we are computing head (or other unknowns), complete a table with unknowns
        else:
            flows = self.input_dict['calc_data']['Flows']
            # Extend tailwater
            if len(tailwater) < len(flows):
                tailwater = tailwater + [min_tw] * (len(flows) - len(tailwater))

            # Extend velocity
            if len(velocity) < len(flows):
                velocity = velocity + [0.0] * (len(flows) - len(velocity))

            self.compute_flow_from_unknown(self.unknown, weir_type, tailwater, velocity, crest_base_elevation,
                                           froudes=froudes)
            return True

    def compute_flow_from_unknown(self, unknown: str, weir_type: str, tailwater_e: list, velocity: list,
                                  crest_base_elevation: float, froudes: list):
        """Computes the data possible; stores results in self.

        Args:
            unknown (string): unknown variable that we are calculating
            weir_type (string): selected weir type
            tailwater_e (list): the tailwater elevations
            velocity (list): the velocities
            crest_base_elevation (float): the thalweg invert elevation
            froudes (list): the froude numbers
        """
        _, num_iterations = self.get_data('Max number of iterations', 500)
        _, flow_err = self.get_data('Flow error', 0.001)
        _, flow_err_p = self.get_data('Flow % error', 0.005)
        _, null_data = self.get_data('Null data', -999.99)
        count = 0

        flow_interp = Interpolation([], [], null_data=null_data, zero_tol=self.zero_tol)
        if unknown in ['Flows', 'Depth', 'Head', 'Tailwater', 'Tailwater elevations']:
            flow_interp.use_second_interpolation = True

        flows = self.input_dict['calc_data']['Flows']

        index = 0
        for target_flow in flows:
            # Because the user could specify a different tailwater with each flow,
            # we need to handle our unknown table for each flow; kinda a pain, but
            # necessary...
            tw_e = tailwater_e[index]
            vel = velocity[index]

            froude = 0.0
            if index < len(froudes):
                froude = froudes[index]

            _, _, list_range = self.get_low_and_high_value(unknown, tw_e)
            self.unk_flow_sorted = SortedDict()

            if unknown != 'Head':
                self.cur_wse = self.input_dict['calc_data']['WSE'][0]

            for cur_val in list_range:
                self._set_current_value(cur_val, unknown)
                cur_flow = self.compute_weir_flow_from_depth(
                    weir_type, self.cur_wse, tw_e, vel, crest_base_elevation=crest_base_elevation, froude=froude)
                self.unk_flow_sorted[cur_val] = cur_flow

            curr_flow_err = min(target_flow * flow_err_p, flow_err)
            cur_val = -999.0
            cur_flow = 0.0
            count = 0
            if target_flow > 0.0:
                while abs(cur_flow - target_flow) > curr_flow_err and count < num_iterations:
                    flow_list = list(self.unk_flow_sorted.values())
                    unk_list = list(self.unk_flow_sorted.keys())
                    flow_interp.x = flow_list
                    flow_interp.y = unk_list
                    cur_val, _ = flow_interp.interpolate_y(target_flow)
                    self._set_current_value(cur_val, unknown)
                    cur_flow = self.compute_weir_flow_from_depth(
                        weir_type, self.cur_wse, tw_e, vel, crest_base_elevation=crest_base_elevation, froude=froude)
                    self.unk_flow_sorted[cur_val] = cur_flow
                    count += 1
                    if count == 5:
                        flow_interp.use_second_interpolation = False
                if count >= num_iterations:
                    warning = "Calculations were unable to converge on a solution when trying to find a flow of "
                    warning += str(target_flow) + " cfs. "
                    warning += str(count) + " attempts were made."
                    self.warnings['Convergence'] = warning

            self.assign_results(wse=self.cur_wse, flow=cur_flow, tw_e=tw_e, crest_base_elevation=crest_base_elevation,
                                unknown=unknown)
            index += 1

    def assign_results(self, wse: float, tw_e: float, flow: float, crest_base_elevation: float, unknown: str):
        """Assigns the current value to the results based on the unknown.

        Args:
            wse (float): The current water surface elevation
            tw_e (float): The current tailwater elevation
            flow (float): The current flow
            crest_base_elevation (float): The crest base elevation
            unknown (string): The unknown variable that we are calculating
        """
        if unknown not in ['Flows', 'Head', 'Depth', 'WSE',]:
            if unknown not in self.results:
                self.results[unknown] = []
            self.results[unknown].append(self.input_dict['calc_data'][unknown])
        self.results['Depths'].append(wse - crest_base_elevation)
        self.results['WSE'].append(wse)
        self.results['Tailwater elevations'].append(tw_e)
        self.results['Tailwater'].append(tw_e - crest_base_elevation)
        self.results['Flows'].append(flow)
        self.results['Weir coefficient'].append(self.coefficient)
        self.results['Weir coefficients'].append(self.weir_coefficients)
        self.results['Applied coefficients'].append(self.applied_coefficients)
        if wse < crest_base_elevation:
            self.warnings['Velocity head'] = "The given velocity head exceeds the head needed for the given discharge"
            self.warnings['Velocity head'] += " without any head from depth."
        self.perpendicular_coefficients.append(self.perpendicular_coefficient)

    def compute_weir_flow_from_depth(self, weir_type: str, wse: float, tw_elevation: float, vel: float,
                                     crest_base_elevation: float, froude: float) -> float:
        """Computes the data possible; stores results in self.

        Args:
            weir_type (string): selected weir type
            wse (float): The water surface elevation
            tw_elevation (float): The tailwater elevation
            vel (float): The velocity
            crest_base_elevation (float): The crest base elevation

        Returns:
            total_flow (float): flow across the weir
        """
        _, g = self.get_data('Gravity', 32.2)
        velocity_head = 0.0
        if self.input_dict['calc_data']['Weir orientation'] == 'Perpendicular (intercepts all flow)':
            velocity_head = vel**2 / (2.0 * g)
        head_above_crest = wse - crest_base_elevation + velocity_head
        tailwater_above_crest = tw_elevation - crest_base_elevation

        flow = 0.0

        self.update_weir_coefficient(head_above_crest, froude)

        if weir_type in self.triangular_shapes:
            flow = self.coefficient * head_above_crest**(2.5)
        elif weir_type in self.trapezoidal_shapes:
            length = self.input_dict['calc_data']['Weir length']
            side_slope = self.input_dict['calc_data']['Side slope']
            # coefficient is typically sharp-crested (3.367 *
            flow = 4.0 / 5.0 * self.coefficient * side_slope * head_above_crest ** 2.5 + \
                self.coefficient * length * head_above_crest ** (1.5)
        elif weir_type in self.rectangular_shapes:
            length = self.input_dict['calc_data']['Weir length']
            flow = self.coefficient * length * head_above_crest**(1.5)
        elif weir_type in self.irregular_shapes:
            crest_elevations = self.input_dict['calc_data']['Irregular weir crest geometry']['Crest elevations']
            crest_stations = self.input_dict['calc_data']['Irregular weir crest geometry']['Crest stations']

            wse = head_above_crest + min(crest_elevations)

            self.weir_coefficients = []
            self.applied_coefficients = []
            for index in range(len(crest_stations) - 1):
                station_left = crest_stations[index]
                station_right = crest_stations[index + 1]
                elevation_left = crest_elevations[index]
                elevation_right = crest_elevations[index + 1]
                if elevation_left > wse and elevation_right > wse:
                    pass  # Both points are dry
                elif elevation_left < wse and elevation_right < wse:
                    pass  # Both points are wet
                elif elevation_left >= wse > elevation_right:
                    # Left is dry, right is wet; find intersection of WSE and segment
                    station_left = (wse - elevation_left) / (elevation_right - elevation_left) * station_right + \
                                   (wse - elevation_right) / (elevation_left - elevation_right) * station_left
                    elevation_left = wse
                elif elevation_right >= wse > elevation_left:
                    # Right is dry, left is wet; find intersection of WSE and segment
                    station_right = (wse - elevation_right) / (elevation_left - elevation_right) * station_left + \
                        (wse - elevation_left) / (elevation_right - elevation_left) * station_right
                    elevation_right = wse
                segment_flow = self.compute_irregular_weir_segment(
                    station_left, station_right, elevation_left,
                    elevation_right, wse, tw_elevation)
                self.weir_coefficients.append(self.computed_coefficient)
                self.applied_coefficients.append(self.submerged_coefficient)
                flow += segment_flow

        # submergence
        # HEC-22, third edition, page 8-24, pdf page 326; EQ 8-21
        total_flow = flow
        if tailwater_above_crest > head_above_crest:
            total_flow = 0.0
            submerged_flow = 0.0
            self.warnings['Reverse flow'] = 'The tailwater elevation is higher than driving upstream head.'
        elif tailwater_above_crest > 0.0 and head_above_crest > 0.0:
            submerged_flow = flow * (
                1 - (tailwater_above_crest / head_above_crest)**1.5)**0.385
            if weir_type not in self.irregular_shapes:  # Irregular shapes takes care of submergence
                total_flow = submerged_flow
        return total_flow

    def compute_irregular_weir_segment(self, station_left, station_right,
                                       elevation_left, elevation_right, wse,
                                       tailwater_e):
        """Computes the weir flow across one segment; stores results in self.

        Args:
            station_left (float): The station of the left side of the segment
            station_right (float): The station of the right side of the segment
            elevation_left (float): The elevation of the left side of the segment
            elevation_right (float): The elevation of the right side of the
            wse (float): The water surface elevation
            tailwater (float): The tailwater elevation over the weir crest

        Returns:
            flow (float): flow of water across weir segment
        """
        delta_x = abs(station_right - station_left)
        delta_y = abs(elevation_right - elevation_left)
        weir_length = math.sqrt(delta_x**2 + delta_y**2)
        roadway_top_elevation = (elevation_left + elevation_right) / 2.0
        weir_head = wse - roadway_top_elevation
        tail_head = tailwater_e - roadway_top_elevation
        if tail_head < 0.0:
            tail_head = 0.0
        self.calculate_coefficient(weir_head, tail_head)
        if weir_head <= 0.0:
            return 0.0
        flow = self.submerged_coefficient * weir_length * weir_head**1.5

        return flow

    def calculate_coefficient(self, weir_head, tail_head):
        """Computes the data possible; stores results in self.

        Args:
            weir_head (float): head of the water over the crest of the weir
            tail_head (float): The tailwater head over the weir crest

        Returns:
            submerged_coefficient (float): Coefficient of the weir segment
        """
        weir_surface = self.input_dict['calc_data']['Weir surface']
        weir_width = self.input_dict['calc_data']['Weir width']
        weir_head_to_crest_width = 0.0
        submergence_ratio = 0.0
        if weir_width > 0.0:
            weir_head_to_crest_width = weir_head / weir_width
        if weir_head > 0.0:
            submergence_ratio = tail_head / (weir_head + .000001)

        _, null_data = self.get_data('Null data')
        _, zero_tol = self.get_data('Zero tolerance')

        if weir_head <= 0.0:
            self.computed_coefficient = null_data
            self.submerged_coefficient = null_data
            return self.submerged_coefficient

        # Determine coefficient without submergence
        coef_r = 0.0
        if weir_surface == 'user-defined':
            coef_r = self.input_dict['calc_data']['Weir coefficient']  # unsubmerged coefficient was input
        else:
            # Broad crested tables
            if weir_head_to_crest_width < 0.15:
                if weir_surface == 'paved':
                    coef_interp = Interpolation(self.broadCrestedPavedX, self.broadCrestedPavedY,
                                                null_data=null_data, zero_tol=zero_tol)
                    coef_r, _ = coef_interp.interpolate_y(weir_head, False)
                elif weir_surface == 'gravel':
                    coef_interp = Interpolation(self.broadCrestedGravelX, self.broadCrestedGravelY,
                                                null_data=null_data, zero_tol=zero_tol)
                    coef_r, _ = coef_interp.interpolate_y(weir_head, False)
            # Sharper crested tables
            else:
                if weir_surface == 'paved':
                    coef_interp = Interpolation(self.sharperCrestedPavedX, self.sharperCrestedPavedY,
                                                null_data=null_data, zero_tol=zero_tol)
                    coef_r, _ = coef_interp.interpolate_y(weir_head, False)
                elif weir_surface == 'gravel':
                    coef_interp = Interpolation(self.sharperCrestedGravelX, self.sharperCrestedGravelY,
                                                null_data=null_data, zero_tol=zero_tol)
                    coef_r, _ = coef_interp.interpolate_y(weir_head, False)
        # KT: Reduction factor for submergence of weir by tailwater
        # Determine KT
        kt = 0.0
        if weir_surface == 'paved':
            coef_interp = Interpolation(self.k1ReductionFactorPavedX, self.k1ReductionFactorPavedY,
                                        null_data=null_data, zero_tol=zero_tol)
            kt, _ = coef_interp.interpolate_y(submergence_ratio, False)
        else:
            coef_interp = Interpolation(self.k1ReductionFactorGravelX, self.k1ReductionFactorGravelY,
                                        null_data=null_data, zero_tol=zero_tol)
            kt, _ = coef_interp.interpolate_y(submergence_ratio, False)

        self.computed_coefficient = coef_r
        self.submerged_coefficient = kt * coef_r
        return self.submerged_coefficient
