"""Model definition."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.model.model_base import ModelBase
from xms.FhwaVariable.interface_adapters.view_model.main.calcdata import CalcDataItem
from xms.FhwaVariableFileIO.read.read_manager import ReadManager
from xms.FhwaVariableFileIO.write.write_manager import WriteManager

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.bridge_scour_data import BridgeScourData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.abutment_scour_data import AbutmentScourData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.contraction_scour_data import ContractionScourData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier_scour_data import PierScourData
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_data import ChannelGeometryData
from xms.HydraulicToolboxCalc.hydraulics.culvert.culvert_barrel_data import CulvertBarrelData
from xms.HydraulicToolboxCalc.hydraulics.culvert.culvert_crossing_data import CulvertCrossingData
from xms.HydraulicToolboxCalc.hydraulics.GVF.gvf_data import GVFData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData
from xms.HydraulicToolboxCalc.hydraulics.weir.weir_data import WeirData
from xms.HydraulicToolboxCalc.model.settings.hydraulic_toolbox_project_settings import HydraulicToolboxProjectSettings


class HydraulicToolbox(ModelBase):
    """A class that stores data and interacts with the GUI (Interactor)."""

    def __init__(self, app_folder=None, frozen_app_path=None, view=None, graphics_view=None):
        """Initializes the model specific data (holds all projects).

        Args:
            app_folder (str): The folder where data for the app is located (version & build date) is found or built.
            frozen_app_path (str): The path where data for the app is located when the code is compiled into an exe.
            view: The view to plug into the AppData.
            graphics_view: The graphics view to plug into the AppData.
        """
        # -- Initialize App/About Data --
        self.app_folder = app_folder
        self.frozen_app_path = frozen_app_path

        cur_file_dir = Path(__file__).resolve().parent
        icon_folder = cur_file_dir.parent / 'resources' / 'icons'

        about_dict = {}

        # Application or model name
        model_name = "Hydraulic Toolbox"
        about_dict['model_name'] = model_name
        version = self.get_version(model_name)
        about_dict['version'] = version
        about_dict['build_message'], about_dict['build_date'] = self.get_build_date(model_name)
        model_icon = icon_folder / 'HydraulicToolbox.ico'
        about_dict['model_icon'] = model_icon
        project_icon = icon_folder / 'root.ico'
        about_dict['project_icon'] = project_icon
        about_dict['model_website'] = 'https://www.fhwa.dot.gov/engineering/hydraulics/software/toolbox404.cfm'
        wiki_url = 'https://www.xmswiki.com/wiki/Main_Page'
        about_dict['wiki_url'] = wiki_url
        description = "The Hydraulic Toolbox is a comprehensive software package that integrates the powerful "
        "functionalities of FHWA's HY-8 and Hydraulic Toolbox programs into a single, unified platform. Designed "
        "for hydrologists, engineers, and environmental scientists, this tool provides advanced capabilities for "
        "analyzing and designing hydraulic structures such as culverts, bridges, and channels. By combining the "
        "strengths of both HY-8 and Hydraulic Toolbox, users can perform detailed hydraulic assessments, manage "
        "water flow, and conduct comprehensive analyses with ease and precision.\n\n"

        "The software features state-of-the-art visualization tools, an intuitive user interface, and a wide range "
        "of modeling capabilities. Users can seamlessly transition between different hydraulic modeling tasks, "
        "reducing data redundancy and enhancing productivity. With the Hydraulic Toolbox, professionals can assess "
        "flood risks, design efficient drainage systems, and ensure the safety and functionality of hydraulic "
        "structures. This integrated environment empowers users with the insights and tools needed for informed "
        "decision-making and sustainable water management."
        about_dict['description'] = description

        # Agency that owns/sponsors the model
        agency = "Federal Highway Administration (FHWA)"
        about_dict['agency'] = agency
        about_dict['agency_icon'] = icon_folder / 'FHWA.png'
        about_dict['agency_website'] = 'https://highways.dot.gov/'
        about_dict['agency_description'] = "The Federal Highway Administration (FHWA) is a division of the United "
        "States Department of Transportation (USDOT) responsible for overseeing the construction, maintenance, and "
        "preservation of the nation's highways, bridges, and tunnels. Its mission is to ensure safe, efficient, and "
        "sustainable transportation infrastructure, supporting economic growth and improving mobility for all users. "
        "The FHWA provides funding, technical expertise, and policy guidance to state and local governments for "
        "highway-related projects."

        # Organization that developed the model
        developed_by = "Aquaveo"
        about_dict['developed_by'] = developed_by
        about_dict['developed_by_icon'] = icon_folder / 'aquaveo_logo.JPG'
        about_dict['developed_by_website'] = 'https://aquaveo.com/'
        about_dict['developed_by_description'] = 'Aquaveo is a leading provider of innovative software solutions for '
        'water resources engineering and environmental modeling. Known for its expertise in hydrology, hydraulics, and '
        'groundwater modeling, Aquaveo develops advanced tools that empower engineers and scientists to analyze, '
        'design, and manage water systems effectively. With a focus on user-friendly interfaces and cutting-edge '
        'technology, Aquaveo supports sustainable water management and infrastructure development worldwide.'

        # Organization that developed the interface to the model
        interface_developed_by = developed_by
        about_dict['interface_developed_by'] = interface_developed_by
        about_dict['interface_developed_by_icon'] = about_dict['developed_by_icon']
        about_dict['interface_developed_by_website'] = about_dict['developed_by_website']
        about_dict['interface_developed_by_description'] = ''

        # -- Plug-ins --
        # View and Graphics View is passed into this init, from the main function

        # Plug-in the file I/O managers with external modules
        model_file_types = ["Hydraulic Toolbox Files (*.htb)", "All Files (*.*)"]
        write_manager = WriteManager(model_name, version)
        read_manager = ReadManager(model_name, version)

        # Plug-in the project settings
        project_settings = HydraulicToolboxProjectSettings(model_name, version, agency,
                                                           developed_by)

        # -- Model specific app data --
        # Set the document folder
        doc_folder = Path(__file__).resolve().parents[3] / 'Docs'

        # Initialize AppData (if not already initialized)
        app_data = AppData(app_name=model_name, version=version, agency=agency,
                           developed_by=developed_by, doc_folder=doc_folder,
                           graphics_view=graphics_view, view=view,)

        # Enable/Disable Geometry and Coverages:
        AppData.geometry_enabled = False
        AppData.coverages_enabled = False
        AppData.image_folder_name_plural = 'Images'
        AppData.image_folder_name_singular = 'image'

        # -- Base class --
        # Initialize the base class (this will also save this data in this class)
        super().__init__(model_name=model_name, version=version, model_icon=model_icon,
                         project_icon=project_icon, agency=agency, developed_by=developed_by,
                         interface_developed_by=interface_developed_by, about_dict=about_dict,
                         project_settings=project_settings, wiki_url=wiki_url, model_file_types=model_file_types,
                         write_manager=write_manager, read_manager=read_manager,
                         app_folder=self.app_folder, frozen_app_path=self.frozen_app_path, app_data=app_data)

        # -- Define calulators --
        # Define the available calculators
        calc_name = 'Open-Channel Flow Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=ManningNData,
            icon=os.path.join(icon_folder, 'Channel.ico'),
            tool_tip='Open-Channel Flow Calculator',
            menu_group='Channel',
            add_toolbar=True,
            complexity=0)

        calc_name = 'Channel Geometry Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=ChannelGeometryData,
            icon=os.path.join(icon_folder, 'ChannelGeometry.ico'),
            tool_tip='Channel Geometry Calculator',
            menu_group='Channel',
            add_toolbar=False,
            complexity=2)

        calc_name = 'Gradually-Varied Flow (GVF) Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=GVFData,
            icon=os.path.join(icon_folder, 'gvf.ico'),
            tool_tip='Gradually-Varied Flow (GVF) Calculator',
            menu_group='Channel',
            add_toolbar=False,
            complexity=2)

        calc_name = 'Culvert Crossing Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=CulvertCrossingData,
            icon=os.path.join(icon_folder, 'hy8.ico'),
            tool_tip='Culvert Crossing Calculator',
            menu_group='Culvert Crossing',
            add_toolbar=True,
            complexity=0)

        calc_name = 'Culvert Barrel Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=CulvertBarrelData,
            icon=os.path.join(icon_folder, 'Culvert-Circular.ico'),
            tool_tip='Culvert Barrel Calculator',
            menu_group='Culvert Crossing',
            add_toolbar=False,
            complexity=2)

        calc_name = 'Weir Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=WeirData,
            icon=os.path.join(icon_folder, 'weir.ico'),
            tool_tip='Weir Calculator',
            menu_group='Weir',
            add_toolbar=True,
            complexity=0)

        calc_name = 'Bridge Scour Project Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=BridgeScourData,
            icon=os.path.join(icon_folder, 'bridgescour.png'),
            tool_tip='Bridge Scour Project Calculator',
            menu_group='Bridge Scour',
            add_toolbar=True,
            complexity=0)

        calc_name = 'Contraction Scour Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=ContractionScourData,
            icon=os.path.join(icon_folder, 'Contraction.png'),
            tool_tip='Contraction Scour Calculator',
            menu_group='Bridge Scour',
            add_toolbar=False,
            complexity=1)

        calc_name = 'Pier Scour Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=PierScourData,
            icon=os.path.join(icon_folder, 'Pier.png'),
            tool_tip='Pier Scour Calculator',
            menu_group='Bridge Scour',
            add_toolbar=False,
            complexity=1)

        calc_name = 'Abutment Scour Calculator'
        self.calcdata_dict[calc_name] = CalcDataItem(
            calc_name, item_class=AbutmentScourData,
            icon=os.path.join(icon_folder, 'Abutment.png'),
            tool_tip='Abutment Scour Calculator',
            menu_group='Bridge Scour',
            add_toolbar=False,
            complexity=1)

        # Initialize the tools with information from this class -- moved to main function
        # self.initialize_tools(about_icon=self.model_icon)

        # Add the model interactor to the AppData interactor
        app_data.add_model_interactor(self)

        # # Reinitialize the project settings (We did not have the default project settings when we initialized the
        # # base class)
        # self.reset_project_settings_to_defaults()

    def close(self):
        """Close the model."""
        # Close the view
        if self.app_data.view:
            self.app_data.view.close()
