"""A Python3 module to smooth xy data."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.HydraulicToolboxCalc.util.distance import xy_distance


class Smoothing:
    """Class for smoothing x, y data."""

    def __init__(self, x_values: list[float], y_values: list[float], distance_factor_to_smooth: float = 1.5,
                 update_self: bool = True):
        """Initializes the Smoothing class.

        Args:
            x_values (list of floats): list of x value data
            y_values (list of floats): list of y value data
            distance_factor_to_smooth (float): distance smoothing factor will go to neighbors to use to smooth current
                point
            update_self (bool): Updates self x, y data (or just pass back the new data)
        """
        self.x = x_values
        self.y = y_values

        self.distance_factor_to_smooth = distance_factor_to_smooth

        self.update_self = update_self

    def smooth_out_discontinuity(self, first_index: int, second_index: int, distance: float, weight: float = 0.25
                                 ) -> tuple[list[float], list[float]]:
        """Smooth out a discontinuity.

        Args:
            first_index (int): first index above discontinuity
            second_index (int): second index below discontinuity
            distance (float): distance to travel for smoothing
            weight (float): the weight to apply to the smoothing

        Returns:
            x (list of floats): the smoothed x values
            y (list of floats): the smoothed y values
        """
        up_index = first_index - 1
        up_distance = 0.0
        num_up_pts = 0
        dn_index = second_index + 1
        dn_distance = 0.0
        num_dn_pts = 0

        distance_to_done = self.distance_factor_to_smooth * distance
        third_distance = distance_to_done * 0.3333

        # Determine the max value is and make sure we do not come within half distance of it
        # so we don't change the span or the slopes near it
        max_x = max(self.x)
        max_index = self.x.index(max_x)
        span_index_1 = max_index - 1
        span_index_2 = max_index + 1

        # Determine the indices of the protected area (corner of culvert)
        max_distance = 0.0
        while span_index_1 > 0 and max_distance < third_distance:
            dist = xy_distance(self.x[span_index_1], self.y[span_index_1],
                               self.x[span_index_1 + 1],
                               self.y[span_index_1 + 1])
            max_distance += dist
            span_index_1 -= 1

        max_distance = 0.0
        while span_index_2 < len(self.x) - 1 and max_distance < third_distance:
            dist = xy_distance(self.x[span_index_2], self.y[span_index_2],
                               self.x[span_index_2 + 1],
                               self.y[span_index_2 + 1])
            max_distance += dist
            span_index_2 += 1

        # Determine indices that we want to smooth
        while up_index > 0 and up_distance < distance_to_done:
            dist = xy_distance(self.x[up_index], self.y[up_index],
                               self.x[up_index + 1], self.y[up_index + 1])

            num_up_pts += 1
            up_distance += dist
            up_index -= 1
            if span_index_1 < up_index < span_index_2 or span_index_1 > up_index > span_index_2:
                break

        while dn_index < len(self.x) and dn_distance < distance_to_done:
            dist = xy_distance(self.x[dn_index], self.y[dn_index],
                               self.x[dn_index - 1], self.y[dn_index - 1])

            num_dn_pts += 1
            dn_distance += dist
            dn_index += 1
            if span_index_1 < dn_index < span_index_2 or span_index_1 > dn_index > span_index_2:
                break

        # Transfer the first points that we won't modify
        x = self.x[0:up_index]
        y = self.y[0:up_index]

        # Run Smooth algorithm twice to reduce wiggles and get a smoother curve
        _, new_x, new_y = self.smooth_poly_line(
            self.x[up_index:dn_index], self.y[up_index:dn_index],
            max(num_up_pts, num_dn_pts), weight)

        # Transfer the smoothed points
        x.extend(new_x)
        y.extend(new_y)

        # Transfer the last points that we won't modify
        x.extend(self.x[dn_index:len(self.x)])
        y.extend(self.y[dn_index:len(self.y)])

        if self.update_self:
            self.x = x
            self.y = y

        return x, y

    @staticmethod
    def smooth_poly_line(x_pts: list[float], y_pts: list[float], num_neighbors: int = 3, weight: float = 0.25
                         ) -> tuple[bool, list[float], list[float]]:
        """Smooths a poly line.

        Args:
            x_pts (list of floats): list of X data for a poly line
            y_pts (list of floats): list of Y data for a poly line
            num_neighbors (int): number of neighbors to consider in smoothing
            weight (float): the weight to apply to the smoothing

        Returns:
            (bool, list of floats, list of floats): whether smoothing was successful, new X data, new Y data
        """
        if len(x_pts) < 3:
            return False, x_pts, y_pts

        if len(x_pts) != len(y_pts):
            return False, x_pts, y_pts

        max_num_neighbors = min(len(x_pts) - 1, num_neighbors)

        num_pts_prior = 1
        num_pts_after = len(x_pts) - 2

        new_x_pts = copy.deepcopy(x_pts)
        new_y_pts = copy.deepcopy(y_pts)

        neighbor_weight = 1.0 - weight
        for index in range(1, len(x_pts) - 1, 1):
            x = 0.0
            y = 0.0
            num_pts_used = 0
            for neighbor_index in range(1, max_num_neighbors, 1):
                if neighbor_index <= num_pts_prior and neighbor_index <= num_pts_after:
                    x += new_x_pts[index - neighbor_index]
                    y += new_y_pts[index - neighbor_index]
                    x += new_x_pts[index + neighbor_index]
                    y += new_y_pts[index + neighbor_index]
                    num_pts_used += 2

            if num_pts_used > 0:
                x_pts[index] = new_x_pts[index] * weight
                y_pts[index] = new_y_pts[index] * weight

                x_pts[index] += x / num_pts_used * neighbor_weight
                y_pts[index] += y / num_pts_used * neighbor_weight

                num_pts_prior += 1
                num_pts_after -= 1
        return True, x_pts, y_pts
