"""AdcircComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil

# 2. Third party modules
from packaging import version

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem, XmsEnvironment as XmEnv
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import Component

# 4. Local modules
# from xms.adcirc.__version__ import version


def get_component_data_object(main_file, comp_uuid, unique_name, name=''):
    """Create a data_object Component to send back to SMS to be built.

    Args:
        main_file (:obj:`str`): Path to the component main file
        comp_uuid: (:obj:`str`): UUID of the component
        unique_name (:obj:`str`): XML component unique name
        name (:obj:`str`): Tree item name of the component

    Returns:
        Component: data_object for the new component

    """
    return Component(
        name=name, comp_uuid=comp_uuid, main_file=main_file, model_name='ADCIRC', unique_name=unique_name, locked=False
    )


class AdcircComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the ADCIRC model."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file.strip('"\''))
        self.data = None
        self.tree_commands = []  # [(menu_text, menu_method)...]
        self.arc_commands = []  # [(menu_text, menu_method)...]
        self.point_commands = []  # [(menu_text, menu_method)...]
        self.uuid = os.path.basename(os.path.dirname(self.main_file))
        self.disp_opts_files = []

    def _get_menu_action(self, command_method, id_files=None, selection=None, main_file=None):
        """Get an ActionRequest for a modal menu item command.

        Args:
            command_method (:obj:`str`): Name of the method to call
            id_files (:obj:`tuple(str)`): Paths to the XMS id and component id files, if applicable
            selection (:obj:`list[int]`): Feature ids of the selected items
            main_file (:obj:`str`): Path to the component main file.

        Returns:
            (:obj:`ActionRequest`): The component menu item ActionRequest
        """
        main_file = main_file if main_file is not None else self.main_file
        action = ActionRequest(
            main_file=main_file,
            modality='MODAL',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name=command_method
        )
        parameters = {}
        if id_files is not None:
            parameters['id_files'] = id_files
        if selection is not None:
            parameters['selection'] = selection
        if parameters:
            action.action_parameters = parameters
        return action

    def _get_menu_item(self, command_text, command_method, id_files=None, selection=None, main_file=None):
        """Get a menu item for a modal command.

        Args:
            command_text (:obj:`str`): The menu text
            command_method (:obj:`str`): Name of the method to call
            id_files (:obj:`tuple(str)`): Paths to the XMS id and component id files, if applicable
            selection (:obj:`list[int]`): Feature ids of the selected items
            main_file (:obj:`str`): Path to the component main file.

        Returns:
            (:obj:`MenuItem`): The component menu item
        """
        action = self._get_menu_action(command_method, id_files=id_files, selection=selection, main_file=main_file)
        return MenuItem(text=command_text, action=action)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.
                - messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        _main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        # We might be duplicating, which puts the new main-file a new location, so we can't just return our main-file.
        main_file = os.path.join(new_path, os.path.basename(self.main_file))
        return main_file, messages, action_requests

    def project_open_event(self, new_path):
        """Called when XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list

        Args:
            new_path (:obj:`str`): Path to the new save location.
        """
        if not self.disp_opts_files or not self.data:  # Component has no display lists
            return

        # If we are not a legacy file from 13.1, we do not need to initialize our display because SMS already did.
        # Will be 0.0 if no currently saved project.
        proj_version = version.parse(XmEnv.xms_environ_project_version())
        if proj_version < version.parse('1.0') or proj_version >= version.parse('13.2'):
            return

        for disp_opts_file in self.disp_opts_files:  # Component has a display lists
            new_disp_opts = os.path.join(new_path, os.path.basename(disp_opts_file))
            if not os.path.isfile(new_disp_opts):
                # Mapped BC components may or may not have their point display option file. They store them if
                # converted from a coverage but display pipe points as lines.
                continue

            if self.cov_uuid:  # drawing on a coverage by id
                self.display_option_list.append(XmsDisplayMessage(file=new_disp_opts, edit_uuid=self.cov_uuid))
            else:  # Free location draw of a mapped component
                self.display_option_list.append(
                    XmsDisplayMessage(file=new_disp_opts, draw_type=DrawType.draw_at_locations)
                )

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list[str]`): A list of the main files of the selected components
                of this type.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`): A list of menus and menu items to be shown.
            Note that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem.
            "None" may be added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list or not self.tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # Add all the project explorer menus
        for command_text, command_method in self.tree_commands:
            menu_list.append(self._get_menu_item(command_text, command_method, main_file=main_file_list[0][0]))
        return menu_list

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (:obj:`bool`): True if the the component is locked for editing. Do not change the files
                if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        if self.tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            actions.append(self._get_menu_action(self.tree_commands[0][1]))

        return messages, actions

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (:obj:`dict`): A dictionary with the key being a string of the feature
                entity type (POINT, ARC, POLYGON). The value of the dictionary is a list of IntegerLiteral ids of the
                selected feature objects.
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`): A list of menus and menu items to be shown.
            Note that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem.
            "None" may be added to the list to indicate a separator.
        """
        menu_list = [None]  # None == spacer
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        if 'ARC' in selection and selection['ARC']:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for command_text, command_method in self.arc_commands:
                menu_list.append(
                    self._get_menu_item(
                        command_text, command_method, id_files=arc_id_files, selection=selection['ARC']
                    )
                )
        if 'POINT' in selection and selection['POINT']:
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
            for command_text, command_method in self.point_commands:
                menu_list.append(
                    self._get_menu_item(
                        command_text, command_method, id_files=point_id_files, selection=selection['POINT']
                    )
                )
        if not menu_list:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return menu_list

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection (:obj:`dict`): A dictionary with the key being a string of the feature
                entity type (POINT, ARC, POLYGON). The value of the dictionary is a list of IntegerLiteral ids of
                the selected feature objects.
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files
                if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)
        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        actions = []
        if 'ARC' in selection:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for _, command_method in self.arc_commands:
                actions.append(self._get_menu_action(command_method, id_files=arc_id_files, selection=selection['ARC']))
                break  # Only expecting one dialog ActionRequest on double-click
        elif 'POINT' in selection:
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
            for _, command_method in self.point_commands:
                actions.append(
                    self._get_menu_action(command_method, id_files=point_id_files, selection=selection['POINT'])
                )
                break  # Only expecting one dialog ActionRequest on double-click

        if not actions:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return [], actions

    def get_display_options_action(self):
        """Get an ActionRequest that will refresh the XMS display list for components with display."""
        return ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='get_initial_display_options',
            comp_uuid=self.uuid
        )

    def copy_external_files(self, new_comp_path):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Args:
            new_comp_path (:obj:`str`): The location of the new component folder in the package.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        return ''

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Args:
            new_main_file (:obj:`str`): The location of the new main file.
            convert_filepaths (:obj:`bool`): False if only the project directory should be updated.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        return ''
