"""BcDataManager class."""

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList

# 4. Local modules
from xms.adcirc.components import bc_component_display as bc_disp
from xms.adcirc.data import bc_data as bcd


class BcDataManager:
    """Helper class for BcData."""
    def __init__(self, bc_comp):
        """Creates a helper class.

        Args:
            bc_comp (:obj:`BcComponent`): The BC component whose data this helper should help

        """
        self.bc_comp = bc_comp

    def handle_merge(self, merge_list):
        # pragma: no cover
        # We will test this if we use it. Sample of merging coverages. Currently, SMS does not provide the component id
        # to feature id mapping. We used to, but it became too much of a problem.
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list[tuple]`): tuple containing:
                main_file (:obj:`str`): The absolute path to the main file of the old component this
                component is being merged from.

                id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                Each value is a tuple that may have two absolute file paths or none. The first
                file is for the ids in XMS on the coverage. The second file contains the ids the
                old component used for those objects. Both id files should be equal in length.
                This dictionary is only applicable if the component derives from
                CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        att_file_idx = 0
        comp_file_idx = 1
        main_file_idx = 0
        id_file_idx = 1

        proj_dir = ''
        copied_disp_opts = False  # Copy display options of the first coverage in the merge list

        tidal_forcing_warning = False  # True if incompatible tidal forcing coverage attributes
        found_periodic_tidal = False  # True as soon as coverage with ocean boundaries and periodic tidal forcing found
        fort19_file = None  # Defined as soon as coverage with ocean boundaries and non-periodic tidal forcing found

        periodic_flow_dset = None  # Defined as soon as coverage with flow boundaries and periodic flow forcing found
        fort20_file = None  # Defined as soon as coverage with flow boundaries and non-periodic flow forcing found
        fort20_hotstart = -1
        flow_ts = 3600
        flow_ts_units = 'seconds'
        flow_forcing_warning = False  # True if incompatible flow forcing coverage attributes

        # Need to build up arrays of new arc/point XMS att ids in the merged coverage that are parallel with arrays
        # of new arc/point component ids in the merged coverage.
        arc_att_ids = []
        arc_comp_ids = []
        point_att_ids = []
        point_comp_ids = []
        for cov in merge_list:
            bc_data = bcd.BcData(cov[main_file_idx])

            # Copy over display options of the first coverage in merge list.
            if not copied_disp_opts:
                copied_disp_opts = True
                old_disp_opts = os.path.join(os.path.dirname(cov[main_file_idx]), bc_disp.BC_JSON)
                new_disp_opts = os.path.join(os.path.dirname(self.bc_comp.main_file), bc_disp.BC_JSON)
                io_util.copyfile(old_disp_opts, new_disp_opts)
                old_disp_opts = os.path.join(os.path.dirname(cov[main_file_idx]), bc_disp.BC_POINT_JSON)
                new_disp_opts = os.path.join(os.path.dirname(self.bc_comp.main_file), bc_disp.BC_POINT_JSON)
                io_util.copyfile(old_disp_opts, new_disp_opts)

            # Mask out old component ids in the data file that no longer exist in XMS.
            if cov[id_file_idx]['ARC']:
                att_ids = read_display_option_ids(cov[id_file_idx]['ARC'][att_file_idx])
                comp_ids = read_display_option_ids(cov[id_file_idx]['ARC'][comp_file_idx])
            else:  # No arcs in this coverage. Might be pipe points though, so keep going.
                att_ids = []
                comp_ids = []
            mask = bc_data.arcs.comp_id.isin(comp_ids)
            bc_data.arcs = bc_data.arcs.where(mask, drop=True)
            # Concatenate the BC arc attributes of each coverage.
            old_to_new_bc_arc_ids = self.bc_comp.data.concat_bc_arcs(bc_data)
            arc_att_ids.extend(att_ids)
            arc_comp_ids.extend([old_to_new_bc_arc_ids[comp_id] for comp_id in comp_ids])

            # Concatenate the pipe point attributes of each coverage.
            if cov[id_file_idx]['POINT']:
                att_ids = read_display_option_ids(cov[id_file_idx]['POINT'][att_file_idx])
                comp_ids = read_display_option_ids(cov[id_file_idx]['POINT'][comp_file_idx])
            else:  # No pipe points in this coverage
                att_ids = []
                comp_ids = []
            mask = bc_data.pipes.comp_id.isin(comp_ids)  # Mask out nonexistent pipe point component ids.
            bc_data.pipes = bc_data.pipes.where(mask, drop=True)
            old_to_new_bc_point_ids = self.bc_comp.data.concat_bc_points(bc_data)
            point_att_ids.extend(att_ids)
            point_comp_ids.extend([old_to_new_bc_point_ids[comp_id] for comp_id in comp_ids])

            # Try to merge tidal forcing options (but only warn once about incompatible merge).
            if not tidal_forcing_warning:
                ocean_arcs = bc_data.arcs.where(bc_data.arcs.type == bcd.OCEAN_INDEX, drop=True)
                if ocean_arcs.sizes['comp_id'] > 0:  # This coverage has ocean arcs.
                    is_periodic = bc_data.info.attrs['periodic_tidal']
                    if is_periodic:
                        found_periodic_tidal = True

                    if not is_periodic and found_periodic_tidal:
                        # Coverage has non-periodic tidal forcing but another one in the merge list uses periodic.
                        tidal_forcing_warning = True
                    elif is_periodic and fort19_file:
                        # Coverage has periodic tidal forcing but another one in the merge list uses non-periodic.
                        tidal_forcing_warning = True
                    elif not is_periodic and fort19_file:
                        # Coverage has non-periodic tidal forcing but another one in the merge list also uses
                        # non-periodic defined on other boundaries (neither fort.19 file is valid for merged coverage).
                        tidal_forcing_warning = True
                    # No other coverage has defined non-periodic forcing yet, so we are OK. For tidal forcing, we only
                    # care if non-periodic is defined on one or more of the coverages being merged. Periodic tidal
                    # forcing is the default and specified with tidal constituent components.
                    elif not is_periodic:
                        proj_dir = bc_data.info.attrs['proj_dir']
                        fort19_file = bc_data.info.attrs['fort.19']

            # Try to merge flow forcing options (but only warn once about incompatible merge).
            if not flow_forcing_warning:
                flow_arcs = bc_data.arcs.where(
                    (bc_data.arcs.type == bcd.RIVER_INDEX) | (bc_data.arcs.type == bcd.FLOW_AND_RADIATION_INDEX),
                    drop=True
                )
                is_periodic = False
                if flow_arcs.sizes['comp_id'] > 0:  # This coverage has flow arcs.
                    is_periodic = bc_data.info.attrs['periodic_flow']

                if not is_periodic and periodic_flow_dset:
                    # Coverage has non-periodic flow forcing but another one in the merge list uses periodic.
                    flow_forcing_warning = True
                elif is_periodic and fort20_file:
                    # Coverage has periodic flow forcing but another one in the merge list uses non-periodic.
                    flow_forcing_warning = True
                elif not is_periodic and fort20_file:
                    # Coverage has non-periodic flow forcing but another one in the merge list also uses
                    # non-periodic defined on other boundaries (neither fort.20 file is valid for merged coverage).
                    flow_forcing_warning = True
                elif is_periodic and periodic_flow_dset:
                    # Coverage has non-periodic tidal forcing but another one in the merge list uses periodic
                    # flow forcing (both fort.19 and periodic constituent datasets are invalid for merged coverage).
                    flow_forcing_warning = True
                # No incompatibility found yet
                elif is_periodic:
                    # First coverage that has defined periodic forcing on its flow boundaries
                    periodic_flow_dset = bc_data.flow_cons
                else:
                    # First coverage that has defined non-periodic forcing on its flow boundaries
                    proj_dir = bc_data.info.attrs['proj_dir']
                    fort20_file = bc_data.info.attrs['fort.20']
                    fort20_hotstart = bc_data.info.attrs['hot_start_flow']
                    flow_ts = bc_data.info.attrs['flow_ts']
                    flow_ts_units = bc_data.info.attrs['flow_ts_units']

        # Set the coverage level forcing options if no incompatibility was found.
        if not tidal_forcing_warning and fort19_file:
            # One of the coverages had non-periodic tidal forcing defined and it did not conflict with any of the
            # other coverages being merged (none of the other coverages had ocean arcs).
            self.bc_comp.data.info.attrs['periodic_tidal'] = 0
            self.bc_comp.data.info.attrs['proj_dir'] = proj_dir
            self.bc_comp.data.info.attrs['fort.19'] = fort19_file
        if not flow_forcing_warning:
            if fort20_file:
                # One of the coverages had non-periodic flow forcing defined and it did not conflict with any of the
                # other coverages being merged (none of the other coverages had river arcs).
                self.bc_comp.data.info.attrs['periodic_flow'] = 0
                self.bc_comp.data.info.attrs['proj_dir'] = proj_dir
                self.bc_comp.data.info.attrs['fort.20'] = fort20_file
                self.bc_comp.data.info.attrs['hot_start_flow'] = fort20_hotstart
                self.bc_comp.data.info.attrs['flow_ts'] = flow_ts
                self.bc_comp.data.info.attrs['flow_ts_units'] = flow_ts_units
            elif periodic_flow_dset:
                # One of the coverages had periodic flow forcing defined and it did not conflict with any of the
                # other coverages being merged (none of the other coverages had river arcs).
                self.bc_comp.data.info.attrs['periodic_flow'] = 1
                self.bc_comp.data.flow_cons = periodic_flow_dset

        # Save arc att ids and comp ids to a temp file so they can be read in the ActionRequest we send back to
        # initialize XMS component coverage ids for the new merged coverage.
        arc_att_file = os.path.join(os.path.dirname(self.bc_comp.main_file), 'tmp_arc_att_ids.txt')
        write_display_option_ids(arc_att_file, arc_att_ids)
        arc_comp_file = os.path.join(os.path.dirname(self.bc_comp.main_file), 'tmp_arc_comp_ids.txt')
        write_display_option_ids(arc_comp_file, arc_comp_ids)
        point_att_file = os.path.join(os.path.dirname(self.bc_comp.main_file), 'tmp_point_att_ids.txt')
        write_display_option_ids(point_att_file, point_att_ids)
        point_comp_file = os.path.join(os.path.dirname(self.bc_comp.main_file), 'tmp_point_comp_ids.txt')
        write_display_option_ids(point_comp_file, point_comp_ids)

        # Commit data before calling update_display_options_file(). That method instantiates a new BcData, updates
        # display UUIDs, and commits. Don't care about the extra commit since all it is going to read is info
        # metadata and the method is called elsewhere.
        self.bc_comp.data.commit()
        self.update_display_options_file(self.bc_comp.main_file, os.path.dirname(self.bc_comp.main_file))

        # Write the merged arc component id display files
        path = os.path.dirname(self.bc_comp.main_file)
        id_lists = [[], [], [], [], [], [], [], [], [], []]  # Indexed by the BC type
        for comp_id in arc_comp_ids:
            type_idx = int(self.bc_comp.data.arcs['type'].loc[dict(comp_id=[comp_id])].data.item())
            id_lists[type_idx].append(comp_id)
        for type_idx, id_list in enumerate(id_lists):
            id_file = bc_disp.BcComponentDisplay.get_display_id_file(type_idx, path)
            write_display_option_ids(id_file, id_list)

        # Write the merged pipe point component id display file.
        write_display_option_ids(os.path.join(path, bc_disp.BC_POINT_ID_FILE), point_comp_ids)

        # Send back ActionRequest to initialize component ids in XMS for the new merged coverage.
        parameters = {
            'ARC_ATT_IDS': arc_att_file,
            'ARC_COMP_IDS': arc_comp_file,
            'POINT_ATT_IDS': point_att_file,
            'POINT_COMP_IDS': point_comp_file,
        }
        action = ActionRequest(
            main_file=self.bc_comp.main_file,
            modality='NO_DIALOG',
            class_name=self.bc_comp.class_name,
            module_name=self.bc_comp.module_name,
            method_name='get_initial_display_options',
            comp_uuid=self.bc_comp.uuid,
            parameters=parameters
        )

        # Send back warning messages
        messages = []
        if tidal_forcing_warning:
            messages.append(
                (
                    'WARNING', 'Incompatible tidal forcing options defined in the coverages being merged. '
                    'Defaulting tidal forcing options in the merged coverage.'
                )
            )
        if flow_forcing_warning:
            messages.append(
                (
                    'WARNING',
                    'Incompatible flow forcing options defined in the coverages being merged. Defaulting flow '
                    'forcing options in the merged coverage.'
                )
            )
        return messages, [action]

    def update_display_options_file(self, new_main_file, new_path, new_data=None):
        """Generate new UUIDs for the component and display lists.

        Will commit data file in this method.

        Args:
            new_main_file (:obj:`str`): Path to the new component's main file
            new_path (:obj:`str`): The new component's directory.
            new_data (:obj:`BcData`): If provided, implies we are not duplicating so we won't wipe the coverage UUID
                or commit the data
        """
        duplicating = False
        if new_data is None:
            new_data = bcd.BcData(new_main_file)
            duplicating = True
        # If duplicating, clear the coverage UUID. Will query XMS for the new coverage's UUID on the create event.
        if duplicating:
            new_data.info.attrs['cov_uuid'] = ''
        # Set component UUID in arc display options file.
        new_comp_uuid = os.path.basename(new_path)
        basename = os.path.basename(self.bc_comp.disp_opts_files[0])
        fname = os.path.join(new_path, basename)
        json_dict = read_display_options_from_json(fname)
        json_dict['uuid'] = str(uuid.uuid4())  # Generate a new arc display list UUID.
        json_dict['comp_uuid'] = new_comp_uuid
        categories = CategoryDisplayOptionList()
        categories.from_dict(json_dict)
        write_display_options_to_json(fname, categories)
        new_data.info.attrs['display_uuid'] = json_dict['uuid']  # Store arc display list UUID in component data.

        if len(self.bc_comp.disp_opts_files) > 1:  # This a BC coverage, not a mapped BC component
            # Set component UUID in point display options file.
            basename = os.path.basename(self.bc_comp.disp_opts_files[1])
            fname = os.path.join(new_path, basename)
            json_dict = read_display_options_from_json(fname)
            json_dict['uuid'] = str(uuid.uuid4())  # Generate a new point display list UUID.
            json_dict['comp_uuid'] = new_comp_uuid
            categories = CategoryDisplayOptionList()
            categories.from_dict(json_dict)
            write_display_options_to_json(fname, categories)
            new_data.info.attrs['point_display_uuid'] = json_dict['uuid']  # Store point display UUID in component data.

        if duplicating:
            new_data.commit()

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Args:
            new_main_file (:obj:`str`): The location of the new main file.
            convert_filepaths (:obj:`bool`): False if only the project directory should be updated.

        Returns:
            (:obj:`str`): Message on failure, empty string on success

        """
        new_data = bcd.BcData(new_main_file)
        if not convert_filepaths:
            # This case is to handle opening a package project for the first time.
            comp_folder = os.path.dirname(self.bc_comp.main_file)
            package_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            new_data.info.attrs['proj_dir'] = package_proj_dir
            new_data.commit()  # Save the updated project directory
            return ''
        err_msg = new_data.update_proj_dir()
        # Copy the newly saved file to temp.
        io_util.copyfile(new_main_file, self.bc_comp.main_file)
        return err_msg
