"""MappedTidalData class."""

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as io_util

# 4. Local modules
from xms.adcirc.__version__ import version

MAPPED_TIDAL_MAIN_FILE = 'mapped_tidal_comp.nc'


class MappedTidalData(XarrayBase):
    """Manages data file for the mapped tidal constituent component."""
    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        # Initialize member variables before calling super so they are available for commit() call
        self._filename = data_file
        self._info = None
        self._cons = None
        self._values = None
        # Create the default file before calling super because we have our own attributes to write.
        self._get_default_datasets()
        super().__init__(self._filename)

    @property
    def cons(self):
        """Load the tidal constituent parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the tidal constituent parameters in the main file

        """
        if self._cons is None:
            self._cons = self.get_dataset('cons', False)
        return self._cons

    @cons.setter
    def cons(self, dset):
        """Setter for the cons dataset attribute."""
        if dset:
            self._cons = dset

    @property
    def values(self):
        """Load the nodal tidal constituent values dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the nodal tidal constituent values in the main file

        """
        if self._values is None:
            self._values = self.get_dataset('values', False)
        return self._values

    @values.setter
    def values(self, dset):
        """Setter for the nodal tidal constituent values dataset attribute."""
        if dset:
            self._values = dset

    def _get_default_datasets(self):
        """Create default datasets if needed."""
        if not os.path.exists(self._filename) or not os.path.isfile(self._filename):
            info = {
                'FILE_TYPE': 'ADCIRC_MAPPED_TIDAL',
                'VERSION': version,
            }
            self._info = xr.Dataset(attrs=info)
            con_data = {
                'amplitude': ('con', np.array([], dtype=np.float64)),
                'frequency': ('con', np.array([], dtype=np.float64)),
                'nodal_factor': ('con', np.array([], dtype=np.float64)),
                'equilibrium_argument': ('con', np.array([], dtype=np.float64)),
                'earth_tide_reduction_factor': ('con', np.array([], dtype=np.float64)),
            }
            con_coords = {'con': []}
            self._cons = xr.Dataset(data_vars=con_data, coords=con_coords)

            values_data = {
                'amplitude': ('con', np.array([], dtype=np.float64)),
                'phase': ('con', np.array([], dtype=np.float64)),
            }
            values_coords = {
                'con': [],
                'node_id': [],
            }
            self._values = xr.Dataset(data_vars=values_data, coords=values_coords)
            self.commit()

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._cons is not None:
            self._cons.close()
            self._drop_h5_groups(['cons'])
            self._cons.to_netcdf(self._filename, group='cons', mode='a')
        if self._values is not None:
            self._values.close()
            self._drop_h5_groups(['values'])
            self._values.to_netcdf(self._filename, group='values', mode='a')

    def vacuum(self):
        """Rewrite all data to a new/wiped file to reclaim disk space.

        All datasets that need to be written to the file must be loaded into memory before calling this method.

        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._cons is None:
            self._cons = self.get_dataset('cons', False)
        if self._values is None:
            self._values = self.get_dataset('values', False)
        io_util.removefile(self._filename)  # Delete the existing NetCDF file
        self.commit()  # Rewrite all datasets
