"""DMI component to manage running of an ADCIRC simulation from SMS."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.components.bases.run_base import RunBase
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg

# 4. Local modules
from xms.adcirc.data import sim_data as sid
from xms.adcirc.feedback.solution_worker_thread import SolutionLoadWorkerThread


class SimRunner(RunBase):
    """Class that handles running ADCIRC."""
    def __init__(self, dummy_mainfile=''):
        """Initializes the class.

        Args:
            dummy_mainfile (:obj:`str`): Unused. Just to keep constructor consistent with component classes.
        """
        super().__init__()
        self.build_vertex = None

    def read_solution(self, query, params, win_cont):
        """Reads the ADCIRC Solution.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            params (:obj:`dict`): Generic map of parameters. Contains the structures for various components that
                are required for adding vertices to the Query Context with Add().
            win_cont (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Find out what files we should load as part of this run
        sim_mainfile = params[0]['sim_mainfile']
        worker = SolutionLoadWorkerThread(sim_mainfile, query, win_cont)
        display_text = {
            'title': 'Loading ADCIRC Solution Files',
            'working_prompt': 'Reading ADCIRC solution files. Please wait...',
            'warning_prompt': 'Warning(s) encountered reading solution. Review log output for more details.',
            'error_prompt': 'Error(s) encountered reading solution. Review log output for more details.',
            'success_prompt':
                'Successfully read ADCIRC solution and converted to XMDF format. Close this dialog to '
                'load the applied data into SMS.',
            'note': '',
            'auto_load': 'Auto load solution datasets into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        feedback_dlg.exec()
        return [], []

    def get_executables(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            filelocation (:obj:`str`): The location of input files for the simulation.

        Returns:
            (:obj:`list[xms.api.dmi.ExecutableCommand]`):
                The executable objects to run and the action requests that go with it.
        """
        # Get the simulation hidden component
        sim_comp = query.item_with_uuid(sim.uuid, model_name='ADCIRC', unique_name='Sim_Component')
        data = sid.SimData(sim_comp.main_file)

        # Setup the solution load ActionRequest
        load_sol = self._create_solution_load_action(filelocation, sim_comp.main_file)

        padcirc = data.general.attrs['run_padcirc']
        num_comp_proc = data.general.attrs['num_comp_proc']
        if padcirc and num_comp_proc > 1:  # Set up for a PADCIRC run
            # TODO: Remove PADCIRC I/O processors input from GUI. We do not ever use it. PADCIRC inputs are used as
            #       command line arguments when running the model . They do not exist in any ADCIRC native files. We
            #       believe the number of I/O processors is only relevant in supercomputer runs. Certainly not the case
            #       if launching from SMS.
            # num_io_proc = data.general.attrs['num_io_proc']

            # Get the path to the PADCIRC executable. We need this because it is a commandline argument to mpiexec.exe,
            # which is the actual executable we need to invoke.
            padcirc_path = query.named_executable_path('PADCIRCExec')
            # ADCPREP (part 1)
            prep1_cmd = ExecutableCommand(
                executable='ADCPREP',
                model='ADCIRC',
                executable_order=0,
                display_name='ADCPREP - Partition Mesh',
                run_weight=2
            )
            prep1_cmd.add_commandline_arg('--np')
            prep1_cmd.add_commandline_arg(f'{num_comp_proc}')
            prep1_cmd.add_commandline_arg('--partmesh')
            # ADCPREP (part 2)
            prep2_cmd = ExecutableCommand(
                executable='ADCPREP',
                model='ADCIRC',
                executable_order=1,
                display_name='ADCPREP - Prepare All',
                run_weight=2
            )
            prep2_cmd.add_commandline_arg('--np')
            prep2_cmd.add_commandline_arg(f'{num_comp_proc}')
            prep2_cmd.add_commandline_arg('--prepall')
            # PADCIRC
            padcirc_cmd = ExecutableCommand(
                executable='MPIEXEC',
                executable_order=2,
                display_name='PADCIRC',
                run_weight=80,
                progress_script='xml_entry_points/adcirc_progress.py'
            )
            padcirc_cmd.add_commandline_arg('--np')
            padcirc_cmd.add_commandline_arg(f'{num_comp_proc}')
            padcirc_cmd.add_commandline_arg('--localonly')
            padcirc_cmd.add_commandline_arg(f'{padcirc_path}')
            padcirc_cmd.add_solution_file(load_sol)
            commands = [prep1_cmd, prep2_cmd, padcirc_cmd]
        else:  # Set up for serial ADCIRC run
            cmd = ExecutableCommand(
                executable='ADCIRC',
                model='ADCIRC',
                executable_order=0,
                display_name='ADCIRC',
                run_weight=100,
                progress_script='xml_entry_points/adcirc_progress.py'
            )
            cmd.add_solution_file(load_sol)
            commands = [cmd]
        return commands

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            filelocation (:obj:`str`): The location of input files for the simulation.

        Returns:
            (:obj:`list[xms.api.dmi.ActionRequest]`): The solution load ActionRequests for the simulation
        """
        # Get the simulation's hidden component
        sim_comp = query.item_with_uuid(sim.uuid, model_name='ADCIRC', unique_name='Sim_Component')
        return [self._create_solution_load_action(filelocation, sim_comp.main_file)]

    def _create_solution_load_action(self, file_location, sim_comp_mainfile):
        """Create an ActionRequest to read a solution.

        Args:
            file_location (:obj:`str`): Path to the project export folder
            sim_comp_mainfile (:obj:`str`): Path to the simulation hidden component's main file

        Returns:
            (:obj:`ActionRequest`): The solution load ActionRequest
        """
        return ActionRequest(
            main_file=os.path.join(file_location, 'fort.15'),
            modality='MODAL',
            class_name='SimRunner',
            module_name='xms.adcirc.dmi.sim_runner',
            method_name='read_solution',
            parameters={'sim_mainfile': sim_comp_mainfile}
        )
