"""Worker thread for the solution importer."""

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.solution_importer import ADCIRCSolutionReader


class SingleSolutionLoadWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing a single ADCIRC solution file."""
    def __init__(self, parent):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.query = Query()

    def _do_work(self):
        """Thread runner that exports the fort.14."""
        # Look for a mapped BC coverage first.
        XmLog().instance.info('Loading ADCIRC solution file...')
        try:
            # Read the solution file and send to XMS. Will be loaded on first compatible geometry found.
            ref_time = self.query.global_time if self.query.global_time is not None else datetime.datetime.now()
            reader = ADCIRCSolutionReader([self.query.read_file], ref_time, self.query, self.query.xms_temp_directory)
            reader.read()
        except Exception:
            XmLog().instance.exception('Error(s) encountered while loading ADCIRC solution files.')


def read_single_solution_file():  # pragma: no cover
    """Read a single solution file.

    Triggered as a drag-n-drop read of a solution file into XMS.
    """
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)

    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope

    worker = SingleSolutionLoadWorkerThread(win_cont)
    display_text = {
        'title': 'Loading ADCIRC Solution File',
        'working_prompt': 'Reading ADCIRC solution file. Please wait...',
        'warning_prompt': 'Warning(s) encountered reading solution. Review log output for more details.',
        'error_prompt': 'Error(s) encountered reading solution. Review log output for more details.',
        'success_prompt':
            'Successfully read ADCIRC solution and converted to XMDF format. Close this dialog to '
            'load the applied data into SMS.',
        'note': '',
        'auto_load': 'Auto load solution dataset into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:
        worker.query.send()
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
