"""Reader for ADCIRC fort.20 geometry files."""

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog


class Fort20Reader:
    """Reads an ADCIRC fort.20 file. River flows."""
    def __init__(self, filename, query=None, datafrom15=None):
        """Initializes the reader.

        If the second argument is provided, so must the third. If the second argument is not
        provided, this is assumed to be a fort.14 control file.

        Args:
            filename (:obj:`str`): Full path and filename of the fort.14 file. If query is not provided, this argument
                will be ignored, and the filename will be obtained from the Query.
            query (:obj:`xms.api.dmi.Query`, optional): Query for communicating with SMS. If none provided, the
                fort.14 will be considered a control file and a new simulation will be created.
            datafrom15 (:obj:`DataForThe14`, optional): Contains cross file dependency data when reading a fort.14
                as part of a fort.15 import.
        """
        self.filename = filename
        self.mapped_bc_dir = ''
        self.lines = []
        self.query = query
        self.is_control = True if not query else False
        self.datafrom15 = datafrom15

    def _setup_query(self):  # pragma: no cover
        """Connects a Query to SMS when the fort.14 is being read as a control file.

        The filename is retrieved from the Query Session, but it is currently always "fort.14" in the cwd.

        Returns:
            (:obj:`bool`): False if connecting to SMS fails.
        """
        self.query = Query()
        self.filename = self.query.read_file

        # Get the XMS temp directory
        comp_dir = os.path.join(self.query.xms_temp_directory, 'Components')
        self.mapped_bc_dir = os.path.join(comp_dir, str(uuid.uuid4()))
        os.makedirs(self.mapped_bc_dir, exist_ok=True)

    def _parse_lines(self, num_river_nodes):
        XmLog().instance.info('Loading fort.20 from ASCII file...')
        with open(self.filename, "r") as f:
            self.lines = f.readlines()
            ts = float(self.lines[0].split()[0])

            # remaining lines are flow values, grouped by timestep
            flows_by_node = [[] for _ in range(num_river_nodes)]
            flow_lines = self.lines[1:]
            num_ts = len(flow_lines) // num_river_nodes
            for i in range(num_ts):
                for node_idx in range(num_river_nodes):
                    line_index = i * num_river_nodes + node_idx
                    flow_value = float(flow_lines[line_index].strip().split()[0])
                    flows_by_node[node_idx].append(flow_value)

        return ts, flows_by_node

    def read(self, num_river_nodes):
        """Top-level function that starts the read of the fort.14 file."""
        # If no Query passed in, create one. This is a control file read.
        try:
            if not self.query:  # pragma: no cover
                self._setup_query()

            if not os.path.isfile(self.filename) or os.path.getsize(self.filename) == 0:
                XmLog().instance.error(f'Error reading fort.20: File not found - {self.filename}')
                return

            # Read the fort.20 and parse the lines
            ts, flows_by_node = self._parse_lines(num_river_nodes)

            # Build the data_objects to send back to SMS.
            XmLog().instance.info('Creating data objects to send to SMS...')
            # If running a recording test, only print the basename.
            filename = os.path.basename(self.filename) if XmEnv.xms_environ_running_tests() == 'TRUE' else self.filename
            XmLog().instance.info(f'Successfully read "{filename}".')
            return ts, flows_by_node
        except Exception:
            XmLog().instance.exception(f'Error reading fort.20: {self.filename}')
