"""Reader for ADCIRC Recording Stations solution output files."""

# 1. Standard Python modules
import os

# 2. Third party modules
import netCDF4
import numpy as np

# 3. Aquaveo modules
from xms.datasets.vectors import vx_vy_to_direction, vx_vy_to_magnitude

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog

ADCIRC_NULL_VALUE = -99999.0
NUM_STATION_NAME_ROWS = 2
NUM_STATION_NAME_COLS = 50


class StationSolutionReader:
    """The ADCIRC station solution file reader."""
    def __init__(self, feature_id):
        """Construct the reader.

        Args:
            feature_id (:obj:`int`): The feature ID of the Recording Station point to read solutions for
        """
        self.feature_id = feature_id
        self.num_ts = 0
        self.num_nodes = 0
        self.read_file = ''

    def find_column_index_from_netcdf(self, file):
        """Returns the index of the column for the feature point we are reading solutions for.

        Args:
            file (:obj:`netCDF4.Dataset`): File handle at the root of the NetCDF file

        Returns:
            (:obj:`int`): 0-base index of the feature points column in the NetCDF arrays, -1 on error
        """
        try:
            if 'station_name' in file.variables:
                name_dset = file['station_name']
                # First try the 0-base index that corresponds to the feature id.
                likely_idx = self.feature_id - 1
                expected_suffix = f'(ID = {self.feature_id})'
                if likely_idx < name_dset.shape[0]:
                    likely_name = name_dset[likely_idx][:].tobytes().decode().strip().upper()
                    if likely_name.endswith(expected_suffix):
                        return likely_idx
                # Loop through the entire dataset
                for idx in range(name_dset.shape[0]):
                    station_name = name_dset[idx][:].tobytes().decode().strip().upper()
                    if station_name.endswith(expected_suffix):
                        return idx
        except Exception:
            pass
        return -1

    def read_ascii_scalars(self, file):
        """Read a scalar dataset from an ASCII file.

        Args:
            file: Open stream to the file to read
        """
        scalars = []
        times = []
        try:
            self.read_header(file)
            if self.num_ts == 0:
                return None  # No data
            for _ in range(self.num_ts):  # loop through the timesteps
                # get the timestep time
                ts_line = file.readline()
                if not ts_line or not ts_line.strip():
                    break  # reached unexpected EOF

                ts_line = ts_line.split()
                ts_time = float(ts_line[0])

                # Read in the values for this timestep
                ts_val = None
                for j in range(self.num_nodes):
                    node_line = file.readline()
                    if not node_line or not node_line.strip():
                        break  # reached unexpected EOF
                    # Find the node we are currently reading solutions for
                    node_line = node_line.split()
                    if ts_val is None and int(node_line[0]) == self.feature_id:
                        ts_val = float(node_line[1])
                        for _ in range(j + 1, self.num_nodes):
                            file.readline()  # Read the rest of the timestep
                        break

                # Add the timestep if enough values were written to the output
                if ts_val is not None:
                    scalars.append(ts_val)
                    times.append(ts_time / (60 * 60))  # Convert seconds to hours for plot
        except Exception:
            XmLog().instance.exception(f'Error reading solution file: {self.read_file}')
            return None
        return [np.array(times), np.array(scalars)]

    def read_ascii_vectors(self, file):
        """Read a vector dataset from an ASCII file.

        Args:
            file: Open stream to the file to read
        """
        times = []
        x_data = []
        y_data = []
        try:
            self.read_header(file)
            for _ in range(self.num_ts):  # loop through the timesteps
                # get the timestep time
                ts_line = file.readline()
                if not ts_line or not ts_line.strip():
                    break  # reached unexpected EOF

                ts_line = ts_line.split()
                ts_time = float(ts_line[0])

                # Read in the values for this timestep
                ts_val = None
                for j in range(self.num_nodes):
                    node_line = file.readline()
                    if not node_line or not node_line.strip():
                        break  # reached unexpected EOF
                    # Find the node we are currently reading solutions for
                    node_line = node_line.split()
                    if ts_val is None and int(node_line[0]) == self.feature_id:
                        ts_val = [float(node_line[1]), float(node_line[2])]
                        for _ in range(j + 1, self.num_nodes):
                            file.readline()  # Read the rest of the timestep
                        break

                # Add the timestep if enough values were written to the output
                if ts_val is not None:
                    x_data.append(ts_val[0])
                    y_data.append(ts_val[1])
                    times.append(ts_time / (60 * 60))  # Convert seconds to hours for plot
        except Exception:
            XmLog().instance.exception(f'Error reading solution file: {self.read_file}')
            return None
        x_data = np.array(x_data)
        y_data = np.array(y_data)
        x_data[x_data == ADCIRC_NULL_VALUE] = np.nan
        y_data[y_data == ADCIRC_NULL_VALUE] = np.nan
        vmag = vx_vy_to_magnitude(x_data, y_data)
        vdir = vx_vy_to_direction(x_data, y_data)
        return [np.array(times), [], x_data, y_data, vmag, vdir]

    def read_netcdf_scalars(self, filename, scalar_path):
        """Read a scalar solution dataset from a NetCDF formatted file.

        Args:
            filename (:obj:`str`): Filesystem path to the NetCDF solution file.
            scalar_path (:obj:`str`): Path in the NetCDF file to the solution dataset

        Returns:
            (:obj:`tuple`): The time x-column, the scalar data values
        """
        with netCDF4.Dataset(filename, 'r') as ncfile:
            column = self.find_column_index_from_netcdf(ncfile)
            if column < 0:
                XmLog().instance.error(f'Unable to find column in file for feature point {self.feature_id}')
                return None
            scalar_data = ncfile[scalar_path][:, column]
            if scalar_data.size == 0:
                XmLog().instance.warning(f'Empty solution data set encountered: {filename}')
                return None  # No data to in the file

            # Replace -99999.0 null values with NaN for numpy operations.
            scalar_data[scalar_data == ADCIRC_NULL_VALUE] = np.nan
            times = ncfile['/time'][:]
            times /= 60 * 60  # Convert to hours
            return [times, scalar_data]

    def read_netcdf_vectors(self, filename, x_path, y_path):
        """Read a vector solution dataset from a NetCDF formatted file.

        Args:
            filename (:obj:`str`): Filesystem path to the NetCDF solution file.
            x_path (:obj:`str`): Path in the NetCDF file to the x-component solution dataset
            y_path (:obj:`str`): Path in the NetCDF file to the y-component solution dataset

        Returns:
            (:obj:`tuple`): The time column, velocity-x column, velocity-y column
        """
        # Read the x and y component values from the NetCDF solution file.
        with netCDF4.Dataset(filename, 'r') as ncfile:
            column = self.find_column_index_from_netcdf(ncfile)
            if column < 0:
                XmLog().instance.error(f'Unable to find column in file for feature point {self.feature_id}')
                return None
            x_data = ncfile[f'{x_path}'][:, column]
            y_data = ncfile[f'{y_path}'][:, column]

            if x_data.size == 0:
                XmLog().instance.warning(f'Empty solution data set encountered: {filename}')
                return None  # No data to in the file

            # Replace -99999.0 and 0.0 null values with NaN for numpy operations.
            x_data[x_data == ADCIRC_NULL_VALUE] = np.nan
            y_data[y_data == ADCIRC_NULL_VALUE] = np.nan
            vmag = vx_vy_to_magnitude(x_data, y_data)
            vdir = vx_vy_to_direction(x_data, y_data)

            times = ncfile['/time'][:]
            times /= 60 * 60  # Convert to hours
            return [times, [], x_data, y_data, vmag, vdir]

    def read_header(self, file):
        """Reader the header line of an ASCII solution file.

        Args:
            file: Open stream to the file to read
        """
        file.readline()  # skip the first header line - nothing we need
        time_line = file.readline()
        time_line = time_line.split()
        self.num_ts = int(time_line[0])
        self.num_nodes = int(time_line[1])  # number of values can be less if sparse
        # don't care about anything else in the header

    def get_reader(self):
        """Get the appropriate reader method based on filename since ADCIRC uses hard-coded filenames."""
        if not os.path.isfile(self.read_file):
            return None

        filename = os.path.basename(self.read_file).lower()
        if 'fort.61.nc' in filename:
            XmLog().instance.info('Reading elevation NetCDF solution at station locations (fort.61.nc)...')
            return self.read_netcdf_elevation
        elif 'fort.62.nc' in filename:
            XmLog().instance.info('Reading velocity NetCDF solution at station locations (fort.62.nc)...')
            return self.read_netcdf_velocity
        elif 'fort.71.nc' in filename:
            XmLog().instance.info('Reading atmospheric pressure NetCDF solution at station locations (fort.71.nc)...')
            return self.read_netcdf_meteor71
        elif 'fort.72.nc' in filename:
            XmLog().instance.info('Reading wind velocity NetCDF solution at station locations (fort.72.nc)...')
            return self.read_netcdf_meteor72
        elif 'fort.61' in filename:
            XmLog().instance.info('Reading elevation ASCII solution at station locations (fort.61)...')
            return self.read_ascii_elevation
        elif 'fort.62' in filename:
            XmLog().instance.info('Reading velocity ASCII solution at station locations (fort.62)...')
            return self.read_ascii_velocity
        elif 'fort.71' in filename:
            XmLog().instance.info('Reading wind pressure ASCII solution at station locations (fort.71)...')
            return self.read_ascii_meteor71
        elif 'fort.72' in filename:
            XmLog().instance.info('Reading wind stress ASCII solution at station locations (fort.72)...')
            return self.read_ascii_meteor72
        else:  # File doesn't have standard hard-coded filename.
            return None

    # Readers returned by get_readers()
    def read_netcdf_elevation(self):
        """Reader for NetCDF elevation solution files (fort.63.nc).

        Returns:
            (:obj:`tuple`): The column headings and the curve value arrays
        """
        data_vals = self.read_netcdf_scalars(self.read_file, '/zeta')
        if data_vals is not None:
            return [['Time', 'Water Surface (eta)'], data_vals]
        return None

    def read_netcdf_velocity(self):
        """Reader for NetCDF velocity solution files (fort.64.nc).

        Returns:
            (:obj:`tuple`): The column headings and the curve value arrays
        """
        data_vals = self.read_netcdf_vectors(self.read_file, '/u-vel', '/v-vel')
        if data_vals is not None:
            return [  # Vector datasets should always come before Vx/Vy
                ['Time', 'Current Vector', 'Current-X', 'Current-Y', 'Current Magnitude', 'Current Direction'],
                data_vals
            ]
        return None

    def read_netcdf_meteor71(self):
        """Reader for NetCDF atmospheric pressure solution files (fort.71.nc).

        Returns:
            (:obj:`tuple`): The column headings and the curve value arrays
        """
        data_vals = self.read_netcdf_scalars(self.read_file, '/pressure')
        if data_vals is not None:
            return [['Time', 'Atmospheric Pressure'], data_vals]
        return None

    def read_netcdf_meteor72(self):
        """Reader for NetCDF wind velocity/stress solution files (fort.72.nc).

        Returns:
            (:obj:`tuple`): The column headings and the curve value arrays
        """
        data_vals = self.read_netcdf_vectors(self.read_file, '/windx', '/windy')
        if data_vals is not None:
            return [  # Vector datasets should always come before Vx/Vy
                ['Time', 'Wind Vector', 'Wind-X', 'Wind-Y', 'Wind Magnitude', 'Wind Direction'],
                data_vals
            ]
        return None

    def read_ascii_elevation(self):
        """Reader for ASCII elevation solution files (fort.63).

        Returns:
            (:obj:`tuple`): The column headings and the curve value arrays
        """
        with open(self.read_file, 'r', buffering=100000) as f:
            data_vals = self.read_ascii_scalars(f)
        if data_vals is not None:
            return [['Time', 'Water Surface (eta) - ASCII'], data_vals]
        return None

    def read_ascii_velocity(self):
        """Reader for ASCII velocity solution files (fort.64)."""
        with open(self.read_file, 'r', buffering=100000) as f:  # fort.64 (velocity) reader
            data_vals = self.read_ascii_vectors(f)
        if data_vals is not None:
            return [  # Vector datasets should always come before Vx/Vy
                ['Time', 'Current Vector - ASCII', 'Current-X - ASCII', 'Current-Y - ASCII',
                 'Current Magnitude - ASCII', 'Current Direction - ASCII'],
                data_vals
            ]
        return None

    def read_ascii_meteor71(self):
        """Reader for ASCII atmospheric pressure solution files (fort.71).

        Returns:
            (:obj:`tuple`): The column headings and the curve value arrays
        """
        with open(self.read_file, 'r', buffering=100000) as f:
            data_vals = self.read_ascii_scalars(f)
        if data_vals is not None:
            return [['Time', 'Atmospheric Pressure - ASCII'], data_vals]
        return None

    def read_ascii_meteor72(self):
        """Reader for ASCII wind velocity/stress solution files (fort.72)."""
        with open(self.read_file, 'r', buffering=100000) as f:
            data_vals = self.read_ascii_vectors(f)
        if data_vals is not None:
            return [  # Vector datasets should always come before Vx/Vy
                ['Time', 'Wind Vector - ASCII', 'Wind-X - ASCII', 'Wind-Y - ASCII', 'Wind Magnitude - ASCII',
                 'Wind Direction - ASCII'],
                data_vals
            ]
        return None

    def read(self, filename):
        """Read a station solution file.

        Args:
            filename (:obj:`str`): Path to the recording station solution file

        Returns:
            (:obj:`tuple`): The list of headings, the solution curve array
        """
        self.read_file = filename
        reader = self.get_reader()
        if reader:
            return reader()
        else:
            XmLog().instance.error(f'Unable to determine format of solution file: {self.read_file}')
            return None
