"""Qt dialog for assigning attributes of a Boundary Conditions coverage arc."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from itertools import combinations
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QCheckBox, QDialog, QHBoxLayout, QLabel, QLineEdit, QPushButton, QRadioButton,
                               QSizePolicy, QSpacerItem, QVBoxLayout, QWidget)
from shapely.geometry import LineString, Point as ShPt
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.coverage.arcs.arc_util import _sum_distances_between_arc_points as sum_dists, arcs_have_compatible_directions
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.message_box import message_with_ok, message_with_ok_cancel
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adcirc.data import bc_data as bcd
from xms.adcirc.gui import gui_util
from xms.adcirc.gui.assign_bc_dlg_ui import Ui_AdcircBcDlg
from xms.adcirc.gui.bc_plots_dialog import BcPlotsDialog
from xms.adcirc.gui.levee_table import LeveeTableWidget


BC_CBX_OPTS = {
    'cbx_bc_type':
        (
            [
                'Unassigned',
                'Ocean',
                'Mainland',
                'Island',
                'River',
                'Levee outflow',
                'Levee',
                'Radiation',
                'Zero normal',
                'Flow and radiation',
            ], [
                bcd.UNASSIGNED_INDEX,
                bcd.OCEAN_INDEX,
                bcd.MAINLAND_INDEX,
                bcd.ISLAND_INDEX,
                bcd.RIVER_INDEX,
                bcd.LEVEE_OUTFLOW_INDEX,
                bcd.LEVEE_INDEX,
                bcd.RADIATION_INDEX,
                bcd.ZERO_NORMAL_INDEX,
                bcd.FLOW_AND_RADIATION_INDEX,
            ]
        ),
    'cbx_bc_options': (['Essential', 'Natural'], [bcd.ESSENTIAL_INDEX, bcd.NATURAL_INDEX]),
}


class AdcircBcDlg(XmsDlg):
    """A dialog showing the Assign BC dialog."""
    def __init__(self, win_cont, bc_data, levee_data, q, msg, selected_ids, old_comp_id, xd, bc_comp, bc_comp_disp):
        """Initializes the Assign BC dialog.

        Args:
            win_cont (:obj:`QWidget`): Parent dialog
            bc_data (:obj:`xarray.Dataset`): Dataset containing the boundary conditions parameters
            levee_data (:obj:`xarray.Dataset`): Dataset containing the levee parameters
            q (:obj:`xarray.Dataset`): Dataset of flow rate
            msg (:obj:`str`): Warning/multi-select message, if any
            selected_ids (:obj:`list`): The feature arc ids of the arcs selected
            old_comp_id (:obj:`int`): The component id of the selected arcs prior to incrementing. Points to the
                existing attributes, if any.
            xd (:obj:`XmsData`): The XMS inter-process communicator
            bc_comp (:obj:`BcComponent`): The BC component
            bc_comp_disp (:obj:'BcComponentDisplay'): The BC component display
        """
        super().__init__(win_cont, 'xmsadcirc.gui.assign_bc_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC_Coverages#Boundary_Condition'
        self.bc_comp = bc_comp
        self.bc_comp_disp = bc_comp_disp
        self.data = bc_data
        self.levee_data = levee_data
        self.levee_table = None
        self.q = q
        self.cbx_view = None
        self.selected_ids = selected_ids
        self.old_comp_id = old_comp_id
        self.msg = msg
        self._xd = xd
        self._cov_arcs = None
        self._levee_arcs = None
        self._levee_locs = None
        self._levee_locs_utm = None
        self._levee_ls = None
        self._reset_bc = False

        if self.data['type'].data[0].item() == bcd.LEVEE_INDEX:
            if not self._check_num_levee_arcs():
                self.reject()
                return

        self.ui = Ui_AdcircBcDlg()
        self.ui.setupUi(self)

        # Setup the dialog
        self._setup_ui()

        if self._reset_bc:
            # change it to unassigned
            index = bcd.UNASSIGNED_INDEX
            self.ui.cbx_bc_type.setCurrentIndex(index)
            self._plot.hide()

    def _check_num_levee_arcs(self):
        """Add the levee table widget."""
        # check the number of assigned arcs to this bc
        num_arcs = len(self.selected_ids)
        if num_arcs == 2:
            return True

        # one arc will be set to unassigned
        if num_arcs == 1:
            if XmEnv.xms_environ_running_tests() != 'TRUE':
                result = message_with_ok_cancel(
                    parent=self, message="This arc is assigned as a levee but has no paired arc. It will be "
                                         "reset to 'Unassigned'.", app_name='SMS')
            else:
                result = 'TRUE'
            if not result:
                return False

            self._reset_bc = True
            return True
        else:
            # if we are here, we have too many arcs
            if not self._do_pairing():
                return False

        return True

    def _do_pairing(self):
        """Add the levee table widget."""
        # compute all unique pair distances
        arc_pts = [[(pt.x, pt.y, pt.z) for pt in arc.get_points(FilterLocation.PT_LOC_ALL)] for arc in self.levee_arcs]

        pair_distances = []
        for id1, id2 in combinations(self.selected_ids, 2):
            arc1 = arc_pts[self.selected_ids.index(id1)]
            arc2 = arc_pts[self.selected_ids.index(id2)]
            pair_distances.append(((id1, id2), sum_dists(arc1, arc2)))

        # sort pairs by ascending distance
        pair_distances.sort(key=lambda x: x[1])

        # select closest pairs
        used_ids = set()
        closest_pairs = []

        for (id1, id2), _ in pair_distances:
            if id1 not in used_ids and id2 not in used_ids:
                closest_pairs.append((id1, id2))
                used_ids.update([id1, id2])

        msg = 'More than two arcs were assigned to this levee pair. The following pairs will be used:\n '
        for id1, id2 in closest_pairs:
            msg += f'Arc {id1} is paired with arc {id2}.\n'

        unpaired = [id for id in self.selected_ids if id not in used_ids]
        if len(unpaired) > 0:
            msg += f"Arc {unpaired[0]} is not paired and will be set to 'Unassigned'."

        if XmEnv.xms_environ_running_tests() != 'TRUE':
            result = message_with_ok_cancel(parent=self, message=msg, app_name='SMS')
        else:
            result = 'TRUE'
        if not result:
            return False

        # create the components for the new pairings
        for i in range(1, len(closest_pairs)):
            # create copies of the bc_data for each pair
            new_comp_id = self.bc_comp.data.add_bc_atts(copy.deepcopy(self.data))

            id1 = closest_pairs[i][0]
            id2 = closest_pairs[i][1]

            self.bc_comp.update_component_id(TargetType.arc, id1, new_comp_id)
            self.bc_comp.update_component_id(TargetType.arc, id2, new_comp_id)

            # Write the updated comp id files for the necessary BC types.
            self.bc_comp_disp._update_id_files(bcd.LEVEE_INDEX, new_comp_id)
            self.bc_comp.data.commit()  # Flush attribute dataset to disk

        if len(unpaired) > 0:
            new_data = copy.deepcopy(self.data)
            new_data['type'].data[0] = bcd.UNASSIGNED_INDEX
            new_comp_id = self.bc_comp.data.add_bc_atts(new_data)
            id = unpaired[0]
            self.bc_comp.update_component_id(TargetType.arc, id, new_comp_id)

            # Write the updated comp id files for the necessary BC types.
            self.bc_comp_disp._update_id_files(bcd.UNINITIALIZED_COMP_ID, new_comp_id)
            self.bc_comp.data.commit()  # Flush attribute dataset to disk

        id1 = closest_pairs[0][0]
        id2 = closest_pairs[0][1]
        self.selected_ids = [id1, id2]
        self._levee_arcs = None
        self._levee_locs = None
        self._levee_locs_utm = None
        self._levee_ls = None

        if len(closest_pairs) > 1 and XmEnv.xms_environ_running_tests() != 'TRUE':
            msg = f'The Boundary Conditions dialog will show data for the arc pair {id1} and {id2}.'
            message_with_ok(parent=self, message=msg, app_name='SMS')

        return True

    @property
    def levee_locs(self):
        """Retrieve levee arc locations if needed."""
        if self._levee_locs is None:
            wkt = self._xd.display_wkt
            coord_sys = self._xd.coordinate_system.upper()
            cov = self._xd.parent_item()
            self._cov_arcs = cov.arcs
            self._levee_locs = []
            self._levee_locs_utm = []
            self._levee_ls = []
            for arc in self.levee_arcs:
                arc_pts = arc.get_points(FilterLocation.PT_LOC_ALL)
                arc_locs = [(pt.x, pt.y, pt.z) for pt in arc_pts]
                self._levee_locs.append(arc_locs)

                if coord_sys == 'GEOGRAPHIC':
                    utm_locs = bcd.convert_lat_lon_pts_to_utm(arc_locs, wkt, None)
                    self._levee_ls.append(LineString(utm_locs))
                    self._levee_locs_utm.append(utm_locs)
                else:
                    self._levee_ls.append(LineString(arc_locs))
                    self._levee_locs_utm.append(arc_locs)

                if len(self._levee_locs) == len(self.selected_ids):
                    break  # all done
        return self._levee_locs

    @property
    def levee_ls(self):
        """Retrieve levee arc locations if needed."""
        if self._levee_ls is None:
            _ = self.levee_locs

        return self._levee_ls

    @property
    def levee_arcs(self):
        """Retrieve levee arc locations if needed."""
        if self._levee_arcs is None:
            self._levee_arcs = []
            if self._cov_arcs is None:
                cov = self._xd.parent_item()
                self._cov_arcs = cov.arcs

            for arc in self._cov_arcs:
                if arc.id in self.selected_ids:
                    self._levee_arcs.append(arc)
                    if len(self._levee_arcs) == len(self.selected_ids):
                        break

        return self._levee_arcs

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Add the message label if not empty
        if self.msg:
            label = QLabel(self.msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.ui.scrollable_area_bc_contents.layout().insertWidget(0, label)

        self._plot = BcPlotsDialog(parent=self, title='BC Plot')
        # Add the levee table widget
        self._add_levee_widgets()
        # Add river stuff
        self._add_river_widgets()
        # Populate comboboxes
        self._add_cbx_opts(self.ui.cbx_bc_type)
        self._disable_levee_option()
        self._add_cbx_opts(self.ui.cbx_bc_options)
        # Connect slots/signals
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.on_bc_type_changed)
        self.ui.cbx_bc_options.currentIndexChanged.connect(self.on_bc_option_changed)
        # Populate widgets from data
        self._load_data()
        self.on_bc_type_changed(self.ui.cbx_bc_type.currentIndex())
        self.on_bc_option_changed(self.ui.cbx_bc_options.currentIndex())
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

        bc_type = self.ui.cbx_bc_type.currentIndex()
        if bc_type == bcd.LEVEE_INDEX:
            self._plot.show()
            self._update_plot()

    def _add_levee_widgets(self):
        """Add the levee table widget."""
        layout = QVBoxLayout()
        coeff_layout = QHBoxLayout()
        sub_label = QLabel('Subcritical flow coeff:')
        self.ui.edt_sub_coeff = QLineEdit('1.0')
        self.ui.edt_sub_coeff.setValidator(QxDoubleValidator(bottom=0.0, parent=self))
        coeff_spacer = QSpacerItem(20, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
        super_label = QLabel('Supercritical flow coeff:')
        self.ui.edt_super_coeff = QLineEdit('1.0')
        self.ui.edt_super_coeff.setValidator(QxDoubleValidator(bottom=0.0, parent=self))
        coeff_layout.addWidget(sub_label)
        coeff_layout.addWidget(self.ui.edt_sub_coeff)
        coeff_layout.addWidget(super_label)
        coeff_layout.addWidget(self.ui.edt_super_coeff)
        coeff_layout.addSpacerItem(coeff_spacer)
        layout.addLayout(coeff_layout)
        # Add the radio buttons
        self.ui.rbt_levee_elevs = QRadioButton(self.ui.grp_levee)
        self.ui.rbt_levee_elevs.setText('Use BC arc node/vertex elevations for crest')
        self.ui.rbt_levee_elevs.toggled.connect(self.on_levee_rbt_changed)
        layout.addWidget(self.ui.rbt_levee_elevs)
        self.ui.rbt_levee_parametric = QRadioButton(self.ui.grp_levee)
        self.ui.rbt_levee_parametric.setText('Specify parametric curve')
        self.ui.rbt_levee_parametric.toggled.connect(self.on_levee_rbt_changed)
        layout.addWidget(self.ui.rbt_levee_parametric)
        # Set up the levee table widget
        dset = self.levee_data.where(self.levee_data.comp_id == self.old_comp_id, drop=True)
        df = dset.to_dataframe()
        gui_util.fix_df_column_names_for_gui(df)
        self.levee_table = LeveeTableWidget(df, self.levee_arcs, self)
        self.levee_table.set_is_levee_outflow(int(self.data['type'].data[0].item()) == bcd.LEVEE_OUTFLOW_INDEX)
        layout.addWidget(self.levee_table)
        # Add a spacer for when the table is not there
        self.levee_spacer = QSpacerItem(40, 10, QSizePolicy.Minimum, QSizePolicy.Expanding)
        layout.addSpacerItem(self.levee_spacer)
        self.ui.grp_levee.setLayout(layout)
        self.ui.rbt_levee_elevs.setChecked(True)
        self.levee_table.model.dataChanged.connect(self._update_plot)
        self.levee_table.table_view.selectionModel().selectionChanged.connect(self._update_plot)

    def _add_river_widgets(self):
        """Add the levee table widget."""
        layout = QVBoxLayout()
        river_layout = QHBoxLayout()
        name_label = QLabel('River name:')
        self.ui.edt_name = QLineEdit('River BC')
        spacer = QSpacerItem(20, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
        wse_label = QLabel('WSE offset:')
        self.ui.edt_wse_offset = QLineEdit('0.0')
        self.ui.edt_wse_offset.setValidator(QxDoubleValidator(bottom=0.0, parent=self))
        river_layout.addWidget(name_label)
        river_layout.addWidget(self.ui.edt_name)
        river_layout.addWidget(wse_label)
        river_layout.addWidget(self.ui.edt_wse_offset)
        river_layout.addSpacerItem(spacer)
        layout.addLayout(river_layout)

        # toggle for constant q or a hydrograph
        self.ui.tog_const_q = QCheckBox(self.ui.grp_river)
        self.ui.tog_const_q.setText('Use constant Q')
        self.ui.tog_const_q.toggled.connect(self.on_river_q_selection_changed)

        self.ui.const_q_container = QWidget()
        self.ui.const_q_layout = QHBoxLayout(self.ui.const_q_container)
        self.ui.edt_const_q = QLineEdit('50.0')
        self.ui.edt_const_q.setValidator(QxDoubleValidator(bottom=0.0, parent=self))
        self.ui.edt_const_q.setFixedWidth(70)
        self.ui.lbl_const_q = QLabel('cms')
        self.ui.const_q_spacer = QSpacerItem(20, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)

        self.ui.const_q_layout.addWidget(self.ui.edt_const_q)
        self.ui.const_q_layout.addWidget(self.ui.lbl_const_q)
        self.ui.const_q_layout.addSpacerItem(self.ui.const_q_spacer)

        self.ui.define_q_container = QWidget()
        self.ui.define_q_layout = QHBoxLayout(self.ui.define_q_container)
        self.ui.btn_define_q = QPushButton('Define...')
        self.ui.btn_define_q.clicked.connect(self.on_btn_define_q)
        self.ui.define_q_spacer = QSpacerItem(20, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
        self.ui.define_q_layout.addWidget(self.ui.btn_define_q)
        self.ui.define_q_layout.addSpacerItem(self.ui.define_q_spacer)

        self.ui.river_spacer = QSpacerItem(40, 10, QSizePolicy.Minimum, QSizePolicy.Expanding)

        layout.addWidget(self.ui.tog_const_q)
        layout.addWidget(self.ui.const_q_container)
        layout.addWidget(self.ui.define_q_container)
        layout.addSpacerItem(self.ui.river_spacer)

        self.ui.grp_river.setLayout(layout)

        q_dset = self.q.where(self.q.comp_id == self.old_comp_id, drop=True)
        self._q_df = q_dset.to_dataframe().reset_index(drop=True)
        self._q_df.index = range(1, len(self._q_df) + 1)
        gui_util.fix_df_column_names_for_gui(self._q_df)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        self._plot.close()
        super().accept()

    def _update_plot(self):
        """Override default accept slot to update persistent dataset."""
        if self.levee_ls is None:
            return

        two_plots = len(self.levee_arcs) == 2
        ave_plots = self.ui.rbt_levee_parametric.isChecked() and two_plots
        if self.ui.rbt_levee_parametric.isChecked():
            parametric_lengths_1 = list(self.levee_table.model.data_frame['Parametric \n Length'])
            length_1 = self.levee_ls[0].length
            lengths_1 = [length_1 * parametric_lengths_1[i] for i in range(len(parametric_lengths_1))]

            if two_plots:
                parametric_lengths_2 = list(self.levee_table.model.data_frame['Parametric \n Length 2'])
                length_2 = self.levee_ls[1].length
                lengths_2 = [length_2 * parametric_lengths_2[i] for i in range(len(parametric_lengths_2))]

            elevs_1 = elevs_2 = list(self.levee_table.model.data_frame['Zcrest (m)'])
        else:
            elevs_1 = [self._levee_locs_utm[0][i][2] for i in range(len(self._levee_locs_utm[0]))]
            lengths_1 = self._get_levee_lengths_from_locs(self.levee_ls[0], self._levee_locs_utm[0])

            if two_plots:
                elevs_2 = [self._levee_locs_utm[1][i][2] for i in range(len(self._levee_locs_utm[1]))]
                lengths_2 = self._get_levee_lengths_from_locs(self.levee_ls[1], self._levee_locs_utm[1])

        sel = self.levee_table.table_view.selectionModel().selectedIndexes()
        dashed_index = sel[0].row() if sel and sel[0].isValid else None

        if two_plots:
            self._plot.update_plot([lengths_1, lengths_2], [elevs_1, elevs_2], [f'id={self.levee_arcs[0].id}',
                                                                                f'id={self._levee_arcs[1].id}'],
                                   dashed_index, ave_plots)
        else:
            self._plot.update_plot([lengths_1], [elevs_1], [f'id={self.levee_arcs[0].id}'], dashed_index, ave_plots)

    def _get_levee_lengths_from_locs(self, ls, locs):
        """Override default accept slot to update persistent dataset."""
        sh_locs = self._sh_pts_from_locs(locs)
        return [ls.project(sh_locs[i]) for i in range(len(sh_locs))]

    def _sh_pts_from_locs(self, locs):
        """Override default accept slot to update persistent dataset."""
        return [ShPt(locs[i][0], locs[i][1], locs[i][2]) for i in range(len(locs))]

    def on_levee_rbt_changed(self):
        """Called when the type of data for the levee is changed."""
        self.levee_table.setVisible(self.ui.rbt_levee_parametric.isChecked())
        self._update_plot()
        # self.levee_spacer.setVisible(self.rbt_levee_elevs.isChecked())

    def on_river_q_selection_changed(self):
        """Called when the type of data for the levee is changed."""
        use_constant = self.ui.tog_const_q.isChecked()

        self.ui.const_q_container.setVisible(use_constant)

        self.ui.define_q_container.setVisible(not use_constant)

    def on_btn_define_q(self):
        """Called when the type of data for the levee is changed."""
        dlg = XySeriesEditor(data_frame=self._q_df, series_name='Q', dialog_title='Define Q', parent=self)
        if dlg.exec() == QDialog.Accepted:
            self._q_df = dlg.model.data_frame

    def on_bc_type_changed(self, index):
        """Called when the BC type combobox is changed.

        Args:
            index (:obj:`int`): Combobox option index of the new BC type
        """
        # Hide/show natural/essential combobox
        enable = False
        if index not in [
            bcd.UNASSIGNED_INDEX, bcd.OCEAN_INDEX, bcd.RADIATION_INDEX, bcd.ZERO_NORMAL_INDEX,
            bcd.FLOW_AND_RADIATION_INDEX
        ]:
            enable = True
        self.ui.cbx_bc_options.setVisible(enable)

        # Hide/show Galerkin toggle
        enable = False
        if index == bcd.ZERO_NORMAL_INDEX:
            enable = True
        self.ui.tog_galerkin.setVisible(enable)

        # Hide/show the levee table
        is_levee_pair = False
        if index == bcd.LEVEE_INDEX:
            arc1 = self.levee_arcs[0]
            arc2 = self.levee_arcs[1]
            if arcs_have_compatible_directions(arc1, arc2):
                is_levee_pair = True
                self._plot.show()
                self._update_plot()
            else:
                # give message that the arcs are incompatible
                if XmEnv.xms_environ_running_tests() != 'TRUE':
                    message_with_ok(parent=self, message="Arc directions incompatible. Swap one arc",
                                    app_name='SMS')
                # change it widget to unassigned or mainland, ...
                index = bcd.UNASSIGNED_INDEX
                self.ui.cbx_bc_type.setCurrentIndex(index)
                self._plot.hide()
        else:
            self._plot.hide()
        self.levee_table.set_is_levee_outflow(not is_levee_pair)
        self.ui.grp_levee.setVisible(index in [bcd.LEVEE_OUTFLOW_INDEX, bcd.LEVEE_INDEX])
        self.ui.grp_river.setVisible(index == bcd.RIVER_INDEX)

        # Update state
        self.on_bc_option_changed(self.ui.cbx_bc_options.currentIndex())

    def on_bc_option_changed(self, index):
        """Called when the BC option combobox is changed.

        Args:
            index (:obj:`int`): Combobox option index of the new BC option (Natural or Essential)
        """
        enable = False
        bc_type = self.ui.cbx_bc_type.currentIndex()
        if bc_type not in [
            bcd.UNASSIGNED_INDEX, bcd.OCEAN_INDEX, bcd.LEVEE_INDEX, bcd.RADIATION_INDEX, bcd.ZERO_NORMAL_INDEX,
            bcd.FLOW_AND_RADIATION_INDEX
        ] and index == bcd.ESSENTIAL_INDEX:
            enable = True
        self.ui.tog_tang_slip.setVisible(enable)

    def _disable_levee_option(self):
        """Disable the Levee option if there are not exactly 2 arcs selected."""
        if len(self.selected_ids) != 2:
            model = self.ui.cbx_bc_type.model()
            item = model.item(bcd.LEVEE_INDEX)
            item.setEnabled(False)

    @staticmethod
    def _add_cbx_opts(cbx):
        """Add a combobox's option texts and user data values.

        Args:
            cbx (:obj:`QCombobox`): The combobox widget.
        """
        cbx_name = cbx.objectName()
        model_vals = BC_CBX_OPTS[cbx_name][1]
        for idx, opt in enumerate(BC_CBX_OPTS[cbx_name][0]):  # Loop through the list of display option texts
            cbx.addItem(opt, model_vals[idx])  # Add the ADCIRC value as UserData

    @staticmethod
    def _set_cbx_opt_from_data(cbx, value):
        """Set a combobox's  current option index based on its user data value.

        Args:
            cbx (:obj:`QCombobox`): The combobox to update
            value (:obj:`int`): Data value associated with the new combobox option
        """
        index = cbx.findData(value)
        if index != -1:
            cbx.setCurrentIndex(index)

    def _load_data(self):
        """Populate widgets from persistent data."""
        self._set_cbx_opt_from_data(self.ui.cbx_bc_type, int(self.data['type'].data[0].item()))
        self._set_cbx_opt_from_data(self.ui.cbx_bc_options, int(self.data['bc_option'].data[0].item()))
        self.ui.tog_tang_slip.setCheckState(
            Qt.Checked if int(self.data['tang_slip'].data[0].item()) == 1 else Qt.Unchecked
        )
        self.ui.tog_galerkin.setCheckState(
            Qt.Checked if int(self.data['galerkin'].data[0].item()) == 1 else Qt.Unchecked
        )
        if 'use_elevs' in self.data.data_vars:
            use_elevs = int(self.data['use_elevs'].data[0].item()) == 1
        else:
            use_elevs = False

        self.ui.rbt_levee_elevs.setChecked(use_elevs)
        self.ui.rbt_levee_parametric.setChecked(not use_elevs)

        if 'subcritical_coeff' in self.data.data_vars:
            subcritical_flow_coeff = float(self.data['subcritical_coeff'].data[0].item())
            supercritical_flow_coeff = float(self.data['supercritical_coeff'].data[0].item())
        else:
            subcritical_flow_coeff = 1.0
            supercritical_flow_coeff = 1.0

        self.ui.edt_sub_coeff.setText(str(subcritical_flow_coeff))
        self.ui.edt_super_coeff.setText(str(supercritical_flow_coeff))

        if 'constant_q' in self.data.data_vars:
            const_q_val = float(self.data['constant_q'].data[0].item())
            use_const_q = const_q_val > 0.0
        else:
            const_q_val = 50.0
            use_const_q = True

        if use_const_q:
            self.ui.edt_const_q.setText(str(const_q_val))

        self.ui.tog_const_q.setChecked(use_const_q)

        if 'wse_offset' in self.data.data_vars:
            wse_offset = float(self.data['wse_offset'].data[0].item())
        else:
            wse_offset = 0.0
        self.ui.edt_wse_offset.setText(str(wse_offset))

        self.on_river_q_selection_changed()

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        df = self.levee_table.model.data_frame
        gui_util.fix_df_column_names_for_io(df)
        dset = df.to_xarray()
        # Rename dimensions and coordinates after pandas <-> xarray conversions.
        if 'index' in dset.dims:
            dset = dset.rename_dims({'index': 'comp_id'})
        if 'index' in dset.coords:
            dset.coords['index'] = [self.data['comp_id'].data.item() for _ in range(self.levee_table.model.rowCount())]
            dset = dset.rename({'index': 'comp_id'})
        else:
            dset.coords['comp_id'] = [
                self.data['comp_id'].data.item() for _ in range(self.levee_table.model.rowCount())
            ]
        rest_of_them = self.levee_data.where(self.levee_data.comp_id != self.data['comp_id'].data[0].item(), drop=True)
        dset = xr.concat([rest_of_them, dset], 'comp_id')
        self.levee_data = dset
        # Will reassign a new component id outside the dialog
        self.data['type'] = self.ui.cbx_bc_type.itemData(self.ui.cbx_bc_type.currentIndex())
        self.data['bc_option'] = self.ui.cbx_bc_options.itemData(self.ui.cbx_bc_options.currentIndex())
        self.data['tang_slip'] = 1 if self.ui.tog_tang_slip.checkState() == Qt.Checked else 0
        self.data['galerkin'] = 1 if self.ui.tog_galerkin.checkState() == Qt.Checked else 0
        self.data['use_elevs'] = 1 if self.ui.rbt_levee_elevs.isChecked() else 0
        self.data['subcritical_coeff'] = float(self.ui.edt_sub_coeff.text())
        self.data['supercritical_coeff'] = float(self.ui.edt_super_coeff.text())

        if self.data['type'].item() == bcd.RIVER_INDEX:
            self.data['wse_offset'] = float(self.ui.edt_wse_offset.text())

            const_q = self.ui.tog_const_q.isChecked()
            self.data['constant_q'] = float(self.ui.edt_const_q.text()) if const_q else -999.0
            if not const_q:
                gui_util.fix_df_column_names_for_io(self._q_df)
                q_dset = self._q_df.to_xarray()
                # Rename dimensions and coordinates after pandas <-> xarray conversions.
                if 'index' in q_dset.dims:
                    q_dset = q_dset.rename_dims({'index': 'comp_id'})
                if 'index' in q_dset.coords:
                    q_dset.coords['index'] = [self.data['comp_id'].data.item() for _ in
                                              range(len(list(self._q_df['Time'])))]
                    q_dset = q_dset.rename({'index': 'comp_id'})
                else:
                    q_dset.coords['comp_id'] = [
                        self.data['comp_id'].data.item() for _ in range(len(list(self._q_df['Time'])))
                    ]
                if len(self.q) > 0:
                    ignore_vals = [self.data['comp_id'].data[0].item(), self.old_comp_id]
                    mask = ~self.q.comp_id.isin(ignore_vals)
                    rest_of_them = self.q.where(mask, drop=True)
                    if len(rest_of_them['Flow']):
                        q_dset = xr.concat([rest_of_them, q_dset], 'comp_id')
                self.bc_comp.data.q = q_dset
            else:
                self.q = self.q.where(self.q['comp_id'] != self.old_comp_id, drop=True)
