"""The levee plots dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QDialogButtonBox, QHBoxLayout, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


class BcPlotsDialog(XmsDlg):
    """A dialog for showing station point solution plots."""
    def __init__(self, parent, title):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            title (str): Title to show at the top
        """
        super().__init__(parent, 'xms.adcirc.gui.bc_plots_dialog')

        self.figure = None  # matplotlib.figure Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self.ax = None  # matplotlib Axes

        # self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC#Output_files'
        self.widgets = {}
        self.setWindowTitle(title)
        self._setup_ui()

    def _setup_ui(self):
        """Sets up the dialog controls."""
        self.setWindowTitle('Levee Elevations Viewer')
        self.setMinimumSize(300, 300)
        self.widgets['layout'] = QVBoxLayout()
        self.figure = Figure()
        self.figure.set_tight_layout(True)  # Frames the plots
        self.canvas = FigureCanvas(self.figure)
        self.canvas.setMinimumWidth(300)  # So user can't resize it to nothing
        self.widgets['layout'].addWidget(self.canvas)
        self.ax = self.figure.add_subplot(111)
        self.setLayout(self.widgets['layout'])

        # add the ok, cancel, help... buttons at the bottom of the dialog
        self._setup_ui_bottom_button_box()

    def _setup_ui_bottom_button_box(self):
        """Add buttons to the bottom of the dialog."""
        # Add Import and Export buttons
        self.widgets['btn_horiz_layout'] = QHBoxLayout()
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(QDialogButtonBox.Close | QDialogButtonBox.Help)
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['btn_box'].helpRequested.connect(self._help_requested)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_box'])
        self.widgets['layout'].addLayout(self.widgets['btn_horiz_layout'])

    def _help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def update_plot(self, x, y, list_legend, x_label, y_label, dash_index=None, ave_plots=False,
                    convert_to_steps=False):  # pragma: no cover
        """Called when the Help button is clicked."""
        self._draw_plot(x, y, list_legend, x_label, y_label, dash_index, ave_plots, convert_to_steps)

    def _draw_plot(self, x, y, ids, x_label, y_label, vertical_dashed_idx, ave_plots, convert_to_steps):
        """Draws all plots base on selection in the list boxes."""
        self.ax.clear()

        if len(x) != len(y) or len(x) == 0:
            return

        if ave_plots and len(x) == 2:
            y = [[(y[0][i] + y[1][i]) / 2.0 for i in range(len(y[0]))]]
            x = [[(x[0][i] + x[1][i]) / 2.0 for i in range(len(x[0]))]]

        dashed_lengths = []
        if vertical_dashed_idx is not None:
            max_y = max(y[0])
            min_y = min(y[0])
            if len(y) > 1:
                max_y = max(max_y, max(y[1]))
                min_y = min(min_y, min(y[1]))
            for i in range(len(x)):
                dashed_lengths.append(x[i][vertical_dashed_idx])

        if convert_to_steps is True:
            x, y = self._convert_to_steps(x, y)

        if len(x) == 2:
            colors = ['blue', 'red']
        for i in range(len(x)):
            if len(x[i]) > 0:
                self.ax.plot(x[i], y[i], label=ids[i], color=colors[i] if len(x) == 2 else None)

        for i in range(len(dashed_lengths)):
            self.ax.plot([dashed_lengths[i], dashed_lengths[i]], [min_y, max_y],
                         linestyle='--', color=colors[i])

        self.ax.set_xlabel(x_label)
        self.ax.set_ylabel(y_label)
        self.ax.legend()
        self.canvas.draw()

    def _convert_to_steps(self, lengths, elevs):
        """Draws all plots base on selection in the list boxes."""
        # find all of the length midpoints
        new_lengths = []
        new_elevs = []
        for i, length in enumerate(lengths):
            if len(length) == 0:
                continue
            elev = elevs[i]

            # first point
            new_length = [0.0]
            new_elev = [elev[0]]

            # middle points
            for j in range(1, len(length)):
                mid_point = (length[j] + length[j - 1]) / 2.0
                new_length.append(mid_point)
                new_elev.append(elev[j - 1])

                # move up the step
                new_length.append(mid_point)
                new_elev.append(elev[j])

            # last point
            new_length.append(length[-1])
            new_elev.append(elev[-1])

            new_lengths.append(new_length)
            new_elevs.append(new_elev)

        return new_lengths, new_elevs
