"""Qt dialog for entering in the Boundary Conditions coverage tidal and flow forcing options."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QLabel, QSizePolicy

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.tides.gui.user_defined_table import (
    COL_AMP, COL_EQUILIBRIUM_ARGUMENT, COL_FREQUENCY, COL_NAME, COL_NODAL_FACTOR, COL_PHASE, UserDefinedWidget
)

# 4. Local modules
from xms.adcirc.gui import gui_util
from xms.adcirc.gui.forcing_options_dlg_ui import Ui_ForcingOptionsDlg


class ForcingOptionsDlg(XmsDlg):
    """A dialog showing the Forcing Options dialog."""
    def __init__(self, bc_data, win_cont, pe_tree):
        """Initializes the sediment material list and properties dialog.

        Args:
            bc_data (:obj:`BcData`): Boundary conditions component data
            win_cont (:obj:`QWidget`): Parent dialog
            pe_tree (:obj:`xms.api.tree.tree_node.TreeNode`): project explorer tree
        """
        super().__init__(win_cont, 'xmsadcirc.gui.forcing_options_dlg')
        self.pe_tree = pe_tree
        self.flow_cons_table = None
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC'
        self.data = bc_data
        self.ui = Ui_ForcingOptionsDlg()
        self.ui.setupUi(self)

        # Setup the dialog
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        self.ui.btn_box.helpRequested.connect(self.help_requested)

        # Add the flow forcing table widget
        self._add_flow_table()

        # set up the time units combo box
        self.ui.cbx_time_units.addItems(['seconds', 'minutes', 'hours', 'days'])

        # Connect slots/signals
        self.ui.tog_periodic_tidal.stateChanged.connect(self.on_tidal_forcing_changed)
        self.ui.tog_periodic_flow.stateChanged.connect(self.on_flow_forcing_changed)
        self.ui.btn_fort19.clicked.connect(self.on_select_fort19)

        # Populate widgets from data
        self._load_data()
        self.on_tidal_forcing_changed(self.ui.tog_periodic_tidal.checkState())
        self.on_flow_forcing_changed(self.ui.tog_periodic_flow.checkState())

    def _add_flow_table(self):
        """Add the flow constituents table widget."""
        df = self.data.flow_cons.to_dataframe()
        gui_util.fix_df_column_names_for_gui(df)
        columns = [COL_NAME, COL_FREQUENCY, COL_NODAL_FACTOR, COL_EQUILIBRIUM_ARGUMENT, COL_AMP, COL_PHASE]
        self.flow_cons_table = UserDefinedWidget(
            data_frame=df, pe_tree=self.pe_tree, reftime=None, parent=self, columns=columns
        )
        self.ui.verticalLayout_2.addWidget(self.flow_cons_table)  # Couldn't find layout in Designer to give a name
        self.lbl_msg = QLabel(self.ui.grp_flow_forcing)
        self.lbl_msg.setText('A constant flow rate can be set directly by turning off the period flow forcing toggle '
                             'and specifying a constant flow on the river arc.')
        self.lbl_msg.setStyleSheet('color: red;')
        self.ui.verticalLayout_2.addWidget(self.lbl_msg)

        size_policy = self.ui.grp_flow_forcing.sizePolicy()
        size_policy.setVerticalPolicy(QSizePolicy.MinimumExpanding)
        self.ui.grp_flow_forcing.setSizePolicy(size_policy)

    def _load_data(self):
        """Populate widgets from persistent data."""
        self.ui.tog_periodic_tidal.setCheckState(
            Qt.Checked if self.data.info.attrs['periodic_tidal'] == 1 else Qt.Unchecked
        )
        self.ui.tog_periodic_flow.setCheckState(
            Qt.Checked if self.data.info.attrs['periodic_flow'] == 1 else Qt.Unchecked
        )
        self.ui.tog_hot_start_flow.setCheckState(
            Qt.Checked if self.data.info.attrs['hot_start_flow'] == -1 else Qt.Unchecked
        )
        fort19_file = self.data.info.attrs['fort.19']
        if fort19_file and self.data.does_file_exist(fort19_file):
            self.ui.lbl_fort19_file.setText(fort19_file)

        flow_ts = self.data.info.attrs['flow_ts']
        self.ui.edt_flow_ts.setValidator(QxDoubleValidator(bottom=0.0, parent=self))
        flow_ts_units = self.data.info.attrs['flow_ts_units']
        self.ui.edt_flow_ts.setText(str(flow_ts))
        self.ui.cbx_time_units.setCurrentText(flow_ts_units)

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        df = self.flow_cons_table.model.data_frame
        gui_util.fix_df_column_names_for_io(df)
        dset = df.to_xarray()
        self.data.flow_cons = dset

        self.data.info.attrs['periodic_tidal'] = 1 if self.ui.tog_periodic_tidal.checkState() == Qt.Checked else 0
        self.data.info.attrs['periodic_flow'] = 1 if self.ui.tog_periodic_flow.checkState() == Qt.Checked else 0
        self.data.info.attrs['hot_start_flow'] = -1 if self.ui.tog_hot_start_flow.checkState() == Qt.Checked else 0
        fort19_file = self.ui.lbl_fort19_file.text()
        if fort19_file != gui_util.NULL_SELECTION:
            self.data.info.attrs['fort.19'] = fort19_file

        if self.ui.tog_periodic_flow.checkState() == Qt.Unchecked:
            self.data.info.attrs['flow_ts'] = float(self.ui.edt_flow_ts.text())
            self.data.info.attrs['flow_ts_units'] = str(self.ui.cbx_time_units.currentText())

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()

    def on_select_fort19(self):
        """Called when the fort.19 file selector button is clicked."""
        gui_util.select_file(
            self, self.ui.lbl_fort19_file, 'Select a non-periodic water level forcing file', 'fort.19;;All files (*.*)',
            self.data.info.attrs['proj_dir']
        )

    def on_tidal_forcing_changed(self, state):
        """Called when the tidal forcing option is changed."""
        enable_fort19 = False
        lbl_txt = "Periodic tidal constituents will be extracted from a simulation's applied Tidal Constituents."
        if state == Qt.Unchecked:
            lbl_txt = 'Select a fort.19 file containing the non-periodic water levels:'
            enable_fort19 = True
        self.ui.file_fort19.setVisible(enable_fort19)
        self.ui.lbl_tidal_forcing.setText(lbl_txt)

    def on_flow_forcing_changed(self, state):
        """Called when the flow forcing option is changed."""
        use_periodic = state == Qt.Checked
        self.ui.flow_forcing.setVisible(not use_periodic)
        self.ui.tog_hot_start_flow.setVisible(not use_periodic)
        self.flow_cons_table.setVisible(use_periodic)
        self.lbl_msg.setVisible(use_periodic)
