"""Subclass of ProcessFeedback dialog in xms.guipy for use with fort.14 and fort.15 imports."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QCheckBox

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.settings import SettingsManager

# 4. Local modules

INTERFACE_WARNING_REG_KEY = 'show_new_interface_import_warning'


class ImportProcessFeedbackDlg(ProcessFeedbackDlg):
    """A dialog for viewing GUI feedback during a long running process."""
    def __init__(self, display_text, logger_name, worker, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            display_text (:obj:`dict`): Text to be displayed in the GUI
                ::
                    {
                        'title': 'The dialog window title',

                        'working_prompt': 'Initial "Please wait..." message in prompt label',

                        'error_prompt': 'Prompt message to display when ERROR or CRITICAL message logged.',

                        'warning_prompt': 'Prompt message to display when WARNING message logged.',

                        'success_prompt': 'Prompt message to display when processing completes successfully.',

                        'note': 'Text in operation specific additional notes label',

                        'auto_load': 'Text for the auto load toggle. If empty no toggle.',

                        'log_format': '%(levelname)-8s - %(asctime)s - %(name)s - %(message)s',

                        'use_colors': False,  # If True, warnings are green, errors are red and don't have ******
                    }
            logger_name (:obj:`str`): Name of the top-level logger to echo to dialog text widget. Should match
                name of the logging module logger.
            worker (:obj:`QThread`): The processing worker thread. Needs to have a signal named 'processing_finished'
                that is emitted whenever the worker thread finishes its stuff. If errors occur in the worker thread, use
                the logging module to log them at ERROR or CRITICAL level. User will be notified and prompted to check
                the log output window. Catch your exceptions in the worker thread and use the logging module to log
                an ERROR or CRITICAL level message. Use logger's exception convenience method inside except clauses to
                log an ERROR level message with a nice traceback for debuggin. DEBUG, INFO, and WARNING level messages
                are echoed to the dialog's log output window, but they are not presented to the user as fatal to the
                processing operation.
            parent (Something derived from :obj:`QWidget`): The parent window

        """
        self._dumb_warning = True
        self._tog_show_dumb_warning = None
        self._warning_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC_Interface_Changes_for_SMS_13.1'
        super().__init__(display_text, logger_name, worker, parent)
        self.display_text['use_colors'] = True

    def _setup_ui(self):
        """Add the programmatic widgets."""
        super()._setup_ui()
        # Add the 'Do not show import warning again toggle', if it has not been set before.
        self._show_dumb_warning()

    def _show_dumb_warning(self):
        """Retrieves the 'Do not show again' registry setting for the new interface warning."""
        # Don't show warning if XMS is running tests.
        if os.environ.get(XmEnv.ENVIRON_RUNNING_TESTS, 'FALSE').upper() == 'TRUE':
            self._dumb_warning = False
            return

        settings = SettingsManager()
        self._dumb_warning = bool(settings.get_setting('xmsadcirc', INTERFACE_WARNING_REG_KEY, 1))
        if self._dumb_warning:
            # Add a 'Do not show again' toggle
            self._tog_show_dumb_warning = QCheckBox('Do not show interface update warning again')
            self._tog_show_dumb_warning.setCheckState(Qt.Unchecked)
            self.ui.horizontalLayout.insertWidget(1, self._tog_show_dumb_warning)

    def accept(self):  # noqa: N802
        """Save the "Do not show again" toggle value to registry if needed."""
        if self.finished and self._dumb_warning and self._tog_show_dumb_warning.checkState() == Qt.Checked:
            settings = SettingsManager()
            settings.save_setting('xmsadcirc', INTERFACE_WARNING_REG_KEY, 0)
        super().accept()

    def processing_finished(self):
        """Called after mapping operation completes. Closes dialog if auto load option enabled."""
        if self._dumb_warning:
            # Log a warning about the drastically different ADCIRC interface.
            self.logger.warning(
                f'\n\tThe ADCIRC interface in SMS has significant changes from previous versions.\n\tSee '
                f'{self._warning_url} for more details.\n\tCheck the "{self._tog_show_dumb_warning.text()}" '
                f'toggle to prevent this warning during future imports.'
            )
        super().processing_finished()
