"""Qt table model using a pandas.DataFrame for storage."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
import pandas
from PySide2.QtCore import QModelIndex, Qt

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules


NO_QMODELINDEX = QModelIndex()
TBL_COL_LENGTH = 0
TBL_COL_LENGTH2 = 1  # Hidden
TBL_COL_ZCREST = 2
TBL_COL_SUB_COEF = 3  # Hidden if levee outflow
TBL_COL_SUPER_COEF = 4
TBL_COL_SIDE_FLAG = 5  # Hidden
TBL_COL_SIDE_LOCS = 6  # Hidden


class LeveeTableModel(QxPandasTableModel):
    """Class derived from QxPandasTableModel to handle a pandas DataFrame with levee data."""
    def __init__(self, data_frame, levee_arcs, parent=None):
        """Initializes the class.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The pandas DataFrame.
            levee_arcs (list): List of two levee arcs
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        self._headers = None
        if len(levee_arcs) > 1:
            self._headers = [f'Parametric \n Length (id={levee_arcs[0].id})',
                             f'Parametric \n Length (id={levee_arcs[1].id})',
                             'Zcrest (m)',
                             'Subcritical\n Flow Coef',
                             'Supercritical\n low Coef']
        super().__init__(data_frame, parent)

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (:obj:`int`): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (:obj:`int`): The role.

        Returns:
            The data.
        """
        if self._headers and role == Qt.DisplayRole and orientation == Qt.Horizontal:
            return self._headers[section]
        return super().headerData(section, orientation, role)

    def setData(self, index, value, role=Qt.EditRole):  # noqa: N802, C901
        """Adjust the data (set it to <value>) depending on index and role.

        Args:
            index (:obj:`QModelIndex`): The index.
            value : The value.
            role (:obj:`int`): The role.

        Returns:
            (:obj:`bool`): Returns True if successful; otherwise False.
        """
        if not index.isValid():
            return False
        table_row = index.row()
        table_column = index.column()

        if table_column in self.read_only_columns:
            return False

        t = (table_row, table_column)
        if t in self.read_only_cells:
            return False

        if role == Qt.EditRole or role == Qt.CheckStateRole:
            row = None
            col = None
            if len(self.data_frame.index) > table_row:
                row = self.data_frame.index[table_row]
            if len(self.data_frame.columns) > table_column:
                col = self.data_frame.columns[table_column]
            if row is None or col is None:
                return False

            dtype = self.data_frame[col].dtype

            if index.column() in self.checkbox_columns:
                value = 1 if value else 0  # Assume checkbox columns are integers 0 and 1
            elif index.column() in self.combobox_columns and isinstance(value, str):
                value = self._match_value_to_combo_box_item(value, table_column)
                if np.issubdtype(dtype, np.integer):  # Check for integer combobox option indices
                    value = self._match_value_to_combo_box_index(table_column, value)
            elif dtype != object:
                try:
                    value = None if value == '' else dtype.type(value)
                except ValueError:
                    return False

            slice_ds = self.data_frame[col][row]
            value_type = type(slice_ds)
            if value_type == pandas.Series:
                if self.data_frame[col][row].iloc[table_row] != value:
                    self.data_frame[col][row].iloc[table_row] = value
                    self.dataChanged.emit(index, index)
            else:
                if self.data_frame[col][row] != value:
                    self.data_frame[col][row] = value
                    self.dataChanged.emit(index, index)
            return True

        return False

    def insertRows(self, row, count, parent=None):  # noqa: N802
        """Inserts <count> rows into the model before 'row'.

         Will append to the bottom if row == row_count.

        Args:
            row (:obj:`int`): The row.
            count (:obj:`int`): The number of rows to insert.
            parent (:obj:`QObject`): Qt parent

        Returns:
            (:obj:`bool`): Returns True if rows successfully inserted; otherwise False.
        """
        row_count = self.rowCount()
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        # Get column info and create a new index
        columns, defaults = self.get_column_info()
        new_index = []
        for x in range(count):
            new_index.append(row + x + 1)
        # Determine an appropriate default length
        row_idx = row - 1
        if row_count == 0:
            defaults['Parametric \n Length'] = 0.0
            defaults['Parametric \n Length 2'] = 0.0
        elif row_count == 1:
            defaults['Parametric \n Length'] = self.data_frame['Parametric \n Length'].iloc[row_idx] + 1.0
            defaults['Parametric \n Length 2'] = self.data_frame['Parametric \n Length 2'].iloc[row_idx] + 1.0
        elif row_count > 1:
            para_length = (
                self.data_frame['Parametric \n Length'].iloc[row_idx] -  # NOQA W504
                self.data_frame['Parametric \n Length'].iloc[row_idx - 1]
            )
            para_length2 = (
                self.data_frame['Parametric \n Length 2'].iloc[row_idx] -  # NOQA W504
                self.data_frame['Parametric \n Length 2'].iloc[row_idx - 1]
            )
            para_length += self.data_frame['Parametric \n Length'].iloc[row_idx]
            para_length2 += self.data_frame['Parametric \n Length 2'].iloc[row_idx]
            defaults['Parametric \n Length'] = para_length
            defaults['Parametric \n Length 2'] = para_length2
        line = pandas.DataFrame(data=defaults, index=new_index, columns=columns)

        # Create the new DataFrame through concatenation
        if row < row_count:  # Insert above row
            df2 = pandas.concat([self.data_frame.loc[:row], line, self.data_frame.loc[row + 1:]],
                                sort=False).reset_index(drop=True)
            df2.index = df2.index + 1  # Start index at 1, not 0
        elif self.data_frame.shape[0] == 0:  # Empty dataframe
            df2 = line
        else:  # Append to bottom
            df2 = pandas.concat([self.data_frame.loc[:], line], sort=False)

        self.data_frame = df2
        self.submit()
        last = self.createIndex(row, self.columnCount())
        self.dataChanged.emit(QModelIndex(), last)  # Needed to update the table view
        self.endInsertRows()
        return True

    def removeRows(self, row, count, parent=None):  # noqa: N802
        """Removes count rows starting with the given row from the model.

        Args:
            row (:obj:`int`): The row.
            count (:obj:`int`): The number of rows to remove.
            parent (:obj:`QObject`): Qt parent

        Returns:
            (:obj:`bool`): Returns True if rows successfully removed; otherwise False.
        """
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)

        if row == 0:
            df2 = self.data_frame.iloc[count:].reset_index(drop=True)
        else:
            df2 = pandas.concat([self.data_frame.iloc[:row], self.data_frame.iloc[row + count:]],
                                sort=False).reset_index(drop=True)

        self.data_frame = df2
        self.data_frame.index = self.data_frame.index + 1  # Start index at 1, not 0
        self.submit()
        last = self.createIndex(row + count, self.columnCount())
        self.dataChanged.emit(QModelIndex(), last)  # Needed to update the table view
        self.endRemoveRows()
        return True
