"""Table widget and associated classes for viewing the levee attributes of an applied boundary condition."""

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from PySide2.QtCore import QSortFilterProxyModel, Qt, Signal
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adcirc.gui.adcirc_table import AdcircTableWidget

TBL_COL_NODE1 = 0
TBL_COL_NODE2 = 1  # Hidden if levee outflow
TBL_COL_ZCREST = 2
TBL_COL_SUB_COEF = 3  # Hidden if levee outflow
TBL_COL_SUPER_COEF = 4
TBL_COL_PIPE = 5  # Hidden if levee outflow
TBL_COL_ZPIPE = 6  # Hidden if levee outflow
TBL_COL_PIPE_DIAMETER = 7  # Hidden if levee outflow
TBL_COL_PIPE_COEF = 8  # Hidden if levee outflow

TBL_RIV_COL_TS = 0


class LeveePandasTableModel(QxPandasTableModel):
    """Class derived from QAbstractTableModel to handle a pandas DataFrame."""
    zcrest_changed = Signal(bool)

    def __init__(self, data_frame, nstr_ids, parent=None):
        """Initializes the class.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The pandas DataFrame.
            nstr_ids (list): List of nodestring ids
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        self._headers = None
        if len(nstr_ids) == 2:
            self._headers = [f'NSTR {nstr_ids[0]} \n Node ID',
                             f'NSTR {nstr_ids[1]} \n Node ID',
                             'Zcrest (m)',
                             'Subcritical\n Flow Coef',
                             'Supercritical \n Flow Coef',
                             'Pipe', 'Zpipe (m)', 'Pipe \n Diameter (m)', 'Bulk \n Coefficient']
        super().__init__(data_frame, parent)

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Args:
            index (:obj:`QModelIndex`): The index.
            role (:obj:`int`): The role.

        Returns:
            The data at index, or None.
        """
        if not index.isValid():
            return None

        if role == Qt.DisplayRole or role == Qt.EditRole:

            # Don't display anything in checkbox columns other than the checkboxes
            if index.column() in self.checkbox_columns:
                return ''
            if index.column() in self.combobox_columns:  # Check for integer combobox option indices
                value = self.data_frame.iloc[index.row(), index.column()]
                if np.issubdtype(type(value), np.integer):
                    s = self._match_index_to_combo_box_value(index.column(), value)
                    if s:
                        return s

            col = self.data_frame.columns[index.column()]
            if self.data_frame[col].dtype == int:
                s = str(int(self.data_frame.iloc[index.row(), index.column()]))
                return s
            else:
                s = str(self.data_frame.iloc[index.row(), index.column()])
                if s == 'nan' and self.show_nan_as_blank:
                    s = ''
                return s
        elif role == Qt.CheckStateRole:
            if index.column() in self.checkbox_columns:
                i = self.data_frame.iloc[index.row(), index.column()]
                return Qt.Checked if i else Qt.Unchecked

        return None  # super().data(index, role)

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (:obj:`int`): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (:obj:`int`): The role.

        Returns:
            The data.
        """
        if self._headers and role == Qt.DisplayRole and orientation == Qt.Horizontal:
            return self._headers[section]
        return super().headerData(section, orientation, role)

    def setData(self, index, value, role=Qt.EditRole):  # noqa: N802
        """Adjust the data (set it to <value>) depending on index and role.

        Args:
            index (:obj:`QModelIndex`): The index.
            value : The value.
            role (:obj:`int`): The role.

        Returns:
            (:obj:`bool`): True if successful; otherwise False.
        """
        if not index.isValid():
            return False

        if index.column() in self.read_only_columns:
            return False

        t = (index.row(), index.column())
        if t in self.read_only_cells:
            return False

        if role == Qt.EditRole or role == Qt.CheckStateRole:
            col = self.data_frame.columns[index.column()]
            dtype = self.data_frame[col].dtype

            if index.column() in self.checkbox_columns:
                value = 1 if value else 0  # Assume checkbox columns are integers 0 and 1
            elif index.column() in self.combobox_columns and isinstance(value, str):
                value = self._match_value_to_combo_box_item(value, index.column())
                if np.issubdtype(dtype, np.integer):  # Check for integer combobox option indices
                    value = self._match_value_to_combo_box_index(index.column(), value)
            elif dtype != object:
                value = None if value == '' else dtype.type(value)

            if self.data_frame.iloc[index.row(), index.column()] != value:
                self.data_frame.iloc[index.row(), index.column()] = value
                self.dataChanged.emit(index, index)
                if col == 'Zcrest (m)':
                    self.zcrest_changed.emit(True)
            return True

        return super().setData(index, value, role)


class MappedLeveeTableDisableEditModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        super().__init__(parent)

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        col = index.column()
        if col in [TBL_COL_ZPIPE, TBL_COL_PIPE_DIAMETER, TBL_COL_PIPE_COEF]:
            row = index.row()
            is_pipe = index.model().index(row, TBL_COL_PIPE).data(Qt.CheckStateRole) == Qt.Checked
            if is_pipe:
                ret_flags |= Qt.ItemIsEnabled
            else:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        return ret_flags


class MappedLeveeTableWidget(AdcircTableWidget):
    """The harmonic analysis table widget."""
    def __init__(self, data_frame, list_nstr_ids, parent=None):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
            list_nstr_ids (list): List of nodestring ids

        """
        super().__init__(
            parent, data_frame, TBL_COL_NODE1, {
                'Node1 Id': -1,
                'Node2 Id': -1,
                'Zcrest (m)': 0.0,
                'Subcritical\n Flow Coef': 0.0,
                'Supercritical\n Flow Coef': 0.0,
                'Pipe': 0,
                'Zpipe (m)': 0.0,
                'Pipe \n Diameter (m)': 0.0,
                'Bulk \n Coefficient': 0.0,
            }
        )
        self.model = LeveePandasTableModel(data_frame, list_nstr_ids, self)
        self.model.read_only_columns = {
            TBL_COL_NODE1,
            TBL_COL_NODE2,
        }
        self.filter_model = MappedLeveeTableDisableEditModel(self)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        dbl_validator = QxDoubleValidator(bottom=0.0, top=10.0, parent=self)
        dbl_delegate = EditFieldValidator(dbl_validator, self)
        pos_dbl_validator = QxDoubleValidator(bottom=0.0, parent=self)
        pos_dbl_delegate = EditFieldValidator(pos_dbl_validator, self)
        tog_delegate = CheckBoxNoTextDelegate(self)
        tog_delegate.state_changed.connect(self.use_pipe_toggled)
        delegates = {
            TBL_COL_ZCREST: dbl_delegate,
            TBL_COL_SUB_COEF: pos_dbl_delegate,
            TBL_COL_SUPER_COEF: pos_dbl_delegate,
            TBL_COL_PIPE: tog_delegate,
            TBL_COL_ZPIPE: dbl_delegate,
            TBL_COL_PIPE_DIAMETER: pos_dbl_delegate,
            TBL_COL_PIPE_COEF: pos_dbl_delegate,
        }
        self.model.set_checkbox_columns({TBL_COL_PIPE})
        super()._setup_ui(delegates, False, True, self.filter_model)
        self.model.set_read_only_columns({TBL_COL_NODE1, TBL_COL_NODE2})
        self.table_view.verticalHeader().hide()
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def set_is_levee_outflow(self, is_outflow):
        """Set flag indicating whether the BC type is an outflow levee.

        If the BC type is levee outflow, the subcritical coefficient column will be hidden.

        Args:
            is_outflow (:obj:`bool`): True if the BC type is a levee outflow

        """
        self.table_view.setColumnHidden(TBL_COL_SUB_COEF, is_outflow)
        self.table_view.setColumnHidden(TBL_COL_NODE2, is_outflow)
        self.table_view.setColumnHidden(TBL_COL_PIPE, is_outflow)
        self.table_view.setColumnHidden(TBL_COL_ZPIPE, is_outflow)
        self.table_view.setColumnHidden(TBL_COL_PIPE_DIAMETER, is_outflow)
        self.table_view.setColumnHidden(TBL_COL_PIPE_COEF, is_outflow)

    def use_pipe_toggled(self):
        """Refresh the table view when the user toggles a pipe on/off."""
        # This guy will not update the pipe columns on toggling of the pipe checkbox. I tried emitting dataChanged
        # from various places (is already getting emitted). Scrolling the row out of view updates the columns when
        # this hack is not in place.
        self.table_view.hide()
        self.table_view.show()


class MappedRiverTableWidget(AdcircTableWidget):
    """The harmonic analysis table widget."""
    def __init__(self, data_frame, list_nstr_ids, parent=None):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
            list_nstr_ids (list): List of nodestring ids

        """
        self._list_nstr_ids = list_nstr_ids
        flow_dict = {f'Node {node}': 0.0 for node in list_nstr_ids}
        super().__init__(parent, data_frame, TBL_RIV_COL_TS, flow_dict)
        self.model = LeveePandasTableModel(data_frame, list_nstr_ids, self)
        self.model.read_only_columns = {TBL_RIV_COL_TS}
        self.filter_model = MappedLeveeTableDisableEditModel(self)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        pos_dbl_validator = QxDoubleValidator(bottom=0.0, parent=self, decimals=4)
        pos_dbl_delegate = EditFieldValidator(pos_dbl_validator, self)
        delegates = {col: pos_dbl_delegate for col in range(len(self._list_nstr_ids) + 1)}
        super()._setup_ui(delegates, False, True, self.filter_model)
        self.model.set_read_only_columns({col for col in range(len(self._list_nstr_ids) + 1)})
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
