"""Dialog to view the attributes of applied tidal constituents."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import os
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import QScrollArea, QTabWidget, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.time_format import datetime_to_qdatetime, ISO_DATETIME_FORMAT
from xms.tides.data import tidal_data as td

# 4. Local modules
from xms.adcirc.gui.constituent_properties_table import ConstituentPropertiesTableWidget
from xms.adcirc.gui.constituent_values_table import ConstituentValuesWidget
from xms.adcirc.gui.mapped_tidal_dlg_ui import Ui_MappedTidalDlg


class MappedTidalDlg(XmsDlg):
    """A dialog for viewing mapped tidal data."""
    def __init__(self, mapped_cons_data, tidal_cons, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            mapped_cons_data (:obj:`XarrayBase`): Should be a MappedTidalData or MappedFlowData
            parent (Something derived from :obj:`QWidget`): The parent window
            tidal_cons (:obj:`bool`): True if this dialog is for tidal constituents, False for flow constituents
        """
        super().__init__(parent, 'xmsadcirc.gui.mapped_tidal_dlg')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Tidal_Constituents'
        self.tidal_cons = tidal_cons
        self.ui = Ui_MappedTidalDlg()
        self.ui.setupUi(self)
        self.mapped_cons_data = mapped_cons_data
        self.con_props_widget = None
        self.con_values_widgets = {}
        if not tidal_cons:
            self.setWindowTitle('Applied Flow Constituents')
        self._setup_ui()

    def _setup_ui(self):
        """Add the widgets."""
        if self.tidal_cons:
            # Read values from the source tidal data file.
            source_data = td.TidalData(
                os.path.join(os.path.dirname(self.mapped_cons_data._filename), td.DEFAULT_TIDAL_DATA_FILE)
            )
            # Set the non-editable source constituent data labels.
            self.ui.lbl_source_value.setText(td.TDB_SOURCES[int(source_data.info.attrs['source'])])
            ref_date = datetime_to_qdatetime(
                datetime.datetime.strptime(source_data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
            )
            self.ui.date_ref_time.setDateTime(ref_date)
        else:  # Hide the source group if flow constituents
            self.ui.grp_source_data.setVisible(False)

        # Add the constituent properties table.
        v_layout = QVBoxLayout()
        self.con_props_widget = ConstituentPropertiesTableWidget(
            self.mapped_cons_data.cons.to_dataframe(), self.tidal_cons, self
        )
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setWidget(self.con_props_widget)
        v_layout.addWidget(scroll_area)
        self.ui.grp_con_props.setLayout(v_layout)

        # Add the constituent values tabs and tables.
        con_values_widget = QTabWidget()
        for idx in range(self.mapped_cons_data.values.sizes['con']):
            con_name = self.mapped_cons_data.values.coords['con'][idx].item()
            con_dset = self.mapped_cons_data.values.sel(con=con_name).drop('con')
            con_widget = ConstituentValuesWidget(con_dset.to_dataframe(), self)
            self.con_values_widgets[con_name] = con_widget
            con_values_widget.addTab(con_widget, con_name.upper())
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setWidget(con_values_widget)
        values_v_layout = QVBoxLayout()
        values_v_layout.addWidget(scroll_area)
        self.ui.grp_values.setLayout(values_v_layout)

        self.ui.btn_box.helpRequested.connect(self.help_requested)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        super().accept()
        self._save_constituent_properties()
        self._save_constituent_values()

    def _save_constituent_properties(self):
        """Update the constituent properties Dataset on accept."""
        if self.tidal_cons:
            self.mapped_cons_data.cons = self.con_props_widget.model.data_frame.rename(
                columns={
                    'Tidal\nPotential\nAmplitude': 'amplitude',
                    'Frequency': 'frequency',
                    'Nodal Factor': 'nodal_factor',
                    'Equilibrium\nArgument': 'equilibrium_argument',
                    'Earth Tide\nReduction\nFactor': 'earth_tide_reduction_factor'
                }
            ).to_xarray()
        else:  # Flow constituents
            self.mapped_cons_data.cons = self.con_props_widget.model.data_frame.rename(
                columns={
                    'Frequency': 'frequency',
                    'Nodal Factor': 'nodal_factor',
                    'Equilibrium\nArgument': 'equilibrium_argument',
                }
            ).to_xarray()

    def _save_constituent_values(self):
        """Update the constituent values Dataset on accept."""
        for con_name, con_data in self.con_values_widgets.items():
            self.mapped_cons_data.values['amplitude'].loc[con_name] = con_data.model.data_frame['Amplitude']
            self.mapped_cons_data.values['phase'].loc[con_name] = con_data.model.data_frame['Phase']
