"""Dialog for viewing the levee attributes of an applied boundary condition arc."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from PySide2.QtWidgets import QLabel, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.adcirc.data.bc_data import convert_lat_lon_pts_to_utm
from xms.adcirc.gui import mapped_levee_table
from xms.adcirc.gui.bc_plots_dialog import BcPlotsDialog
from xms.adcirc.gui.view_bc_dlg_ui import Ui_ViewBCDlg

DEFAULT_WIDTH = 450


class ViewBCDlg(XmsDlg):
    """A dialog for viewing mapped levee data."""
    def __init__(self, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.

        """
        super().__init__(parent, 'xmsadcirc.gui.view_bc_dlg')


class ViewRiverDlg(ViewBCDlg):
    """A dialog for viewing mapped river data."""
    def __init__(self, river_flow_data, nodestring_id, node_ids, xd, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            river_flow_data (:obj:`pandas.DataFrame`): The river data to view
            nodestring_id (:obj:`int`): Id of the levee BC
            node_ids (list): List of the nodes that make up the nodestring
            xd (XmsData): Object for communicating with XMS
            parent (Something derived from :obj:`QWidget`): The parent window.

        """
        super().__init__(parent)
        self.ui = Ui_ViewBCDlg()
        self.ui.setupUi(self)
        self.xd = xd
        self._river_flow_data = river_flow_data
        self._node_ids = node_ids
        self.nodestring_id = nodestring_id
        self._setup_ui()

    def _setup_ui(self):
        """Add the widgets."""
        # Add the tree item picker.
        v_layout = QVBoxLayout()
        river_msg = QLabel(f'Viewing flow rates at each mapped node in nodestring {self.nodestring_id}.')
        v_layout.addWidget(river_msg)
        self._tbl_widget = mapped_levee_table.MappedRiverTableWidget(self._river_flow_data, self._node_ids, self)
        v_layout.addWidget(self._tbl_widget)
        self.ui.grp_bc.setLayout(v_layout)

        # self._tbl_widget.model.zcrest_changed.connect(self._update_plot)
        # self._tbl_widget.table_view.selectionModel().selectionChanged.connect(self._update_plot)
        #

        self._plot = BcPlotsDialog(parent=self, title='Unit Flows')
        self._plot.show()
        self._update_plot()

    def _update_plot(self):
        """Override default accept slot to update persistent dataset."""
        times = self._river_flow_data.index.tolist()
        times = [times.copy() for _ in range(len(self._node_ids))]
        flows = [self._river_flow_data[col].tolist() for col in self._river_flow_data.columns]

        self._plot.update_plot(times, flows, self._node_ids, 'Time', 'Flow')


class ViewLeveeDlg(ViewBCDlg):
    """A dialog for viewing mapped levee data."""
    def __init__(self, levee_data, nodestring_id, partner_id, ugrid, xd, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            levee_data (:obj:`pandas.DataFrame`): The levee data to view
            nodestring_id (:obj:`int`): Id of the levee BC
            partner_id (:obj:`int`): Id of the levee pair partner if not a levee outflow else -1
            ugrid (UGrid): Ugrid used for mapping
            xd (XmsData): Object for communicating with XMS
            parent (Something derived from :obj:`QWidget`): The parent window.

        """
        super().__init__(parent)
        self.ui = Ui_ViewBCDlg()
        self.ui.setupUi(self)
        self.ugrid = ugrid
        self.xd = xd
        self.levee_data = levee_data
        self.nodestring_id = nodestring_id
        self.partner_id = partner_id
        self.lengths = []
        self._levee_dists = None
        self._tbl_widget = None
        self._setup_ui()

    def _setup_ui(self):
        """Add the widgets."""
        # Add the tree item picker.
        v_layout = QVBoxLayout()
        if self.partner_id < 0:
            msg = f'Viewing levee outflow attributes of nodestring {self.nodestring_id}.'
        else:
            msg = f'Viewing levee pair attributes of nodestrings {self.nodestring_id} and {self.partner_id}.'
        levee_msg = QLabel(msg)
        v_layout.addWidget(levee_msg)
        self._tbl_widget = mapped_levee_table.MappedLeveeTableWidget(self.levee_data,
                                                                     [self.nodestring_id, self.partner_id], self)
        self._tbl_widget.set_is_levee_outflow(self.partner_id < 0)
        v_layout.addWidget(self._tbl_widget)
        self.ui.grp_bc.setLayout(v_layout)
        self._tbl_widget.model.zcrest_changed.connect(self._update_plot)
        self._tbl_widget.table_view.selectionModel().selectionChanged.connect(self._update_plot)

        self._plot = BcPlotsDialog(parent=self, title='ADCIRC Levee Plot')
        self._plot.show()
        self._update_plot()

    def _update_plot(self):
        """Override default accept slot to update persistent dataset."""
        zcrests = list(self.levee_data['Zcrest (m)'])
        id = f'NSTR {self.nodestring_id}'
        if self.partner_id >= 0:
            id = f'{id}, {self.partner_id}'

        sel = self._tbl_widget.table_view.selectionModel().selectedIndexes()
        dashed = sel[0].row() if sel and sel[0].isValid() else None
        self._plot.update_plot(self.levee_dists, [zcrests], [id], 'Length', 'Zcrest', dashed, convert_to_steps=True)

    @property
    def levee_dists(self):
        """Retrieve levee arc locations if needed."""
        if self._levee_dists is None:
            locs1 = [self.ugrid.get_point_location(node1 - 1) for node1 in list(self.levee_data['Node1 Id'])]
            locs2 = [self.ugrid.get_point_location(node2 - 1) for node2 in list(self.levee_data['Node2 Id'])]
            ave_locs = [self._get_ave_loc(locs1[i], locs2[i]) for i in range(len(locs1))]

            self._levee_dists = []
            projection = self.xd.display_projection
            wkt = projection.well_known_text
            coord_sys = projection.coordinate_system.upper()
            if coord_sys == 'GEOGRAPHIC':
                ave_locs = convert_lat_lon_pts_to_utm(ave_locs, wkt, None)

            num_pts = len(ave_locs)
            prev_pt = (ave_locs[0][0], ave_locs[0][1])
            actual_lengths = [0.0]
            total_length = 0.0
            for i in range(1, num_pts):
                now_pt = (ave_locs[i][0], ave_locs[i][1])
                diff_pts = np.subtract(prev_pt, now_pt)
                dist = np.linalg.norm(diff_pts)
                total_length += dist
                actual_lengths.append(total_length)
                prev_pt = now_pt
            self._levee_dists.append(actual_lengths)

        return self._levee_dists

    def _get_ave_loc(self, loc1, loc2):
        """Retrieve levee arc locations if needed."""
        return [(loc1[0] + loc2[0]) / 2.0, (loc1[1] + loc2[1]) / 2.0]

    @property
    def levee_dists_old(self):
        """Retrieve levee arc locations if needed."""
        if self._levee_dists is None:
            locs1 = [self.ugrid.get_point_location(node1 - 1) for node1 in list(self.levee_data['Node1 Id'])]
            locs2 = [self.ugrid.get_point_location(node2 - 1) for node2 in list(self.levee_data['Node2 Id'])]

            self._levee_dists = []
            projection = self.xd.display_projection
            wkt = projection.well_known_text
            coord_sys = projection.coordinate_system.upper()
            if coord_sys == 'GEOGRAPHIC':
                locs1 = convert_lat_lon_pts_to_utm(locs1, wkt, None)
                locs2 = convert_lat_lon_pts_to_utm(locs2, wkt, None)

            levee_pair_locs = [locs1, locs2]
            for levee_locs in levee_pair_locs:
                num_pts = len(levee_locs)
                prev_pt = (levee_locs[0][0], levee_locs[0][1])
                actual_lengths = [0.0]
                total_length = 0.0
                for i in range(1, num_pts):
                    now_pt = (levee_locs[i][0], levee_locs[i][1])
                    diff_pts = np.subtract(prev_pt, now_pt)
                    dist = np.linalg.norm(diff_pts)
                    total_length += dist
                    actual_lengths.append(total_length)
                    prev_pt = now_pt
                self._levee_dists.append(actual_lengths)

        return self._levee_dists

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._plot.close()
        super().accept()

    def reject(self):
        """Override default accept slot to update persistent dataset."""
        self._plot.close()
        super().reject()
