"""XMS DMI component for migrating old ADCIRC SMS projects."""

# 1. Standard Python modules
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.bases.migrate_base import MigrateBase
from xms.components.runners import migrate_runner as mgrun
from xms.data_objects.parameters import Coverage, FilterLocation

# 4. Local modules


class MigrateAdcirc(MigrateBase):
    """Convert old SMS projects to component-based interface."""
    def __init__(self):
        """Constructor.

        Note that this implementation currently only deletes any ADCIRC simulations and converts any BC coverages to
        Area Property types. We do not support migrating pre-13.1 ADCIRC SMS projects.
        """
        super().__init__()
        self._new_covs = []
        self._delete_uuids = []
        self._xms_data = None  # Old data to migrate. Get from Query or pass in for testing.

    def _change_bc_coverage_type(self, old_cov):
        """Change all BC coverage types to Area Property.

        Args:
            old_cov (:obj:`Coverage`): Geometry of the old BC coverage
        """
        new_cov = Coverage(name=old_cov.name, uuid=str(uuid.uuid4()), projection=old_cov.projection)
        new_cov.polygons = old_cov.polygons
        new_cov.arcs = old_cov.arcs
        new_cov.set_points(old_cov.get_points(FilterLocation.PT_LOC_DISJOINT))
        new_cov.complete()
        self._new_covs.append(new_cov)

    def _get_data_to_migrate(self, old_items, query):
        """Get dumps of all the coverages we need to migrate.

        Args:
            old_items (:obj:`dict`): This is a map from uuid's to object attributes.
            query (:obj:`xms.dmi.Query`): Object for communicating with SMS.
        """
        self._xms_data = {'cov_dumps': []}
        for item_uuid, info in old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == 'ADCIRC':
                self._delete_uuids.append(item_uuid)
                if info[2] == 'Coverage':
                    cov_dump = query.item_with_uuid(item_uuid)
                    if not cov_dump:
                        continue
                    self._xms_data['cov_dumps'].append(cov_dump)

    def map_to_typename(
        self, uuid_map, widget_map, take_map, main_file_map, hidden_component_map, material_atts, material_polys, query
    ):
        """This is the first pass of a two-pass system.

        This function performs all the data object migrations necessary for use between versions of model interfaces.

        Args:

            uuid_map (:obj:`dict{uuid -> object}`): This is a map from uuid's to object attributes.
                This is a collection of things you want to find the conversions for.

            widget_map (:obj:`dict{uuid -> object}`): This is a larger dictionary mapping uuid's to
                their conversions.

            take_map (:obj:`dict{uuid -> object}`): This is a map from uuid's of taking
                objects to a list of uuids of objects taken by the taker.

            main_file_map (:obj:`dict{uuid -> str}`): This is a map from uuid's to main
                files of components.

            hidden_component_map (:obj:`dict{uuid -> object}`): This is a map from uuid's of owning
                objects to a list of uuids of hidden components owned by the object.

            material_atts (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is tuple of display options (name, r, g, b, alpha, texture)

            material_polys (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is a set of polygon ids with that material assignment.

            query (:obj:`xms.data_objects.parameters.Query`): This is used to communicate with XMS.
                Do not call the 'send' method on this instance as 'send' will be called later.

        Returns:
            (:obj:`dict{uuid -> object}`): This is the map of delete and replacement requests.
        """
        if not query:  # Testing. Initialize XMS data.
            self._xms_data = mgrun.xms_migration_data
        else:
            self._get_data_to_migrate(uuid_map, query)

        for coverage in self._xms_data['cov_dumps']:
            self._change_bc_coverage_type(coverage)

        return {}  # Don't care. There is no model that needs this information.

    def send_replace_map(self, replace_map, query):
        """This is the second pass of a two-pass system.

        Args:
            replace_map (:obj:`dict{uuid -> object}`): This is the map of delete and replace requests.
                This can be generated by map_to_typename().

            query (:obj:`xms.data_objects.parameters.Query`): This is used to communicate with SMS.

        """
        if not query:
            return

        # Add delete requests to the Query
        for delete_uuid in self._delete_uuids:
            query.delete_item(delete_uuid)

        # Add old BC coverage geometries as Area Property coverages
        for new_cov in self._new_covs:
            query.add_coverage(new_cov)

    def get_messages_and_actions(self):
        """Called at the end of migration.

        This is for when a message needs to be given to the user about some change due to migration.
        Also, an ActionRequest can be given if there is ambiguity in the migration.

        Returns:
            (:obj:`tuple(list[str],list[xms.api.dmi.ActionRequest])`):
                messages, action_requests:

                    Where messages is a list of tuples with the first element of the tuple being
                    the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

                    action_requests is a list of actions for XMS to perform.
        """
        messages = []
        # If there was any ADCIRC data, give an error message. We do not support migration of pre-13.1 projects.
        if self._delete_uuids:
            url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC_Interface_Changes_for_SMS_13.1#Importing_Native_ADCIRC_Model'
            messages.append(
                (
                    'ERROR', 'Opening SMS 13.0 ADCIRC projects in later versions of SMS is not supported. All ADCIRC '
                    'simulations will be deleted and all ADCIRC Boundary Conditions coverages will be converted '
                    "to Area Property coverages. Read the project simulations' native ADCIRC input files to load "
                    f'the ADCIRC model data in this version of SMS. See {url} for more details.'
                )
            )
        return messages, []
