"""Module contains reader for migrating old ADCIRC custom interface SMS projects to the new DMI ADCIRC interface.

This is a one time operation per project. Once the project has been migrated and saved in SMS, no further migration
will be necessary until the next interface update.
"""

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules
import h5py
from PySide2.QtCore import QDateTime

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.adcirc.file_io.fort15_reader import Fort15Reader
from xms.adcirc.gui import gui_util


class ADCIRCXmsProjectReader:
    """Migrates an old custom ADCIRC interface project to a new DMI simulation(s).

    Attributes:
        query (:obj:`xms.api.dmi.Query`): Query object used to communicate with SMS
    """
    def __init__(self):
        """Construct the project migration reader."""
        self.query = Query()

    def read(self):
        """Top-level function that will trigger the read of an old custom ADCIRC interface project.

        The ADCIRC meshes in the project will be converted to DMI simulations in SMS. This is a one time
        only operation if the project is then saved.
        """
        extra_h5 = self.query.read_file

        # Get the SMS temp directory.
        comp_dir = os.path.join(self.query.xms_temp_directory, 'Components')
        os.makedirs(comp_dir, exist_ok=True)

        # Get global time. Set reference date to this if not in ADCIRC input files.
        global_time = self.query.global_time
        if global_time is not None:
            global_time = gui_util.datetime_to_qdatetime(global_time)
        else:
            global_time = QDateTime.currentDateTime()

        # Get the fort.14 and fort.15 filenames.
        geom_uuid = str(uuid.uuid4())
        storm_track_uuid = None
        filenames = {}
        overwrite_widgets = {}
        with h5py.File(extra_h5, "r") as f:
            proj_grp = f.require_group("ADCIRC_Project_Files")
            # Get the fort.14 filename
            if proj_grp.__contains__("ADCIRC Project File"):
                filenames[14] = str(proj_grp["ADCIRC Project File"][0].decode('UTF-8'))
            # Get the ADCIRC mesh UUID
            if proj_grp.__contains__("GUID"):
                geom_uuid = str(proj_grp["GUID"][0].decode('UTF-8'))
            # Get the fort.15 filename
            if proj_grp.__contains__("ADCIRC Control File"):
                # Get the widget values in the extra.h5 that are not in the fort.15
                if f.__contains__("Extra"):
                    extra_group = f["Extra"]
                    # Get the interpolation reference date.
                    if extra_group.__contains__("ColdStartDateTime2"):
                        overwrite_widgets["dateInterpRef"] = [{"": str(float(extra_group["ColdStartDateTime2"][0]))}]

                    # Get the number of computational and I/O processors for PADCIRC runs.
                    if extra_group.__contains__("ComputerProcessors"):
                        num_comp_processors = int(extra_group["ComputerProcessors"][0])
                        overwrite_widgets["edtNumCompProcessor"] = [{"": num_comp_processors}]
                        # Turn on PADCIRC run if number of computational processors is greater than 1.
                        if num_comp_processors > 1:
                            overwrite_widgets["togRunPadcirc"] = [{"": 1}]
                    if extra_group.__contains__("IOProcessors"):
                        overwrite_widgets["edtNumCommProcessor"] = [{"": int(extra_group["IOProcessors"][0])}]

                    # TODO: Datasets on the mesh (NWS=(1,2,4,5) and nodal attributes) will be saved in the "_datasets"
                    # TODO:     project folder. These will be read before the delayed _extra.h5, which contains the
                    # TODO:     geometry in the fort.14. Obviously, reading the XMDF datasets will fail without a
                    # TODO:     geometry, but those datasets do get read in and linked up correctly by the model-native
                    # TODO:     read scripts. The user sees errors though and solution and/or other datasets that
                    # TODO:     happened to be on the mesh would have to be loaded in manually. Figure this out.
                    # # Get paths of the mesh wind datasets if they exist. Grid datasets were never selected in the
                    # # custom interface, so we won't be able to select the appropriate datasets for those widgets.
                    # if extra_group.__contains__("ScalarDataset"):
                    #     overwrite_widgets["edtNumCommProcessor"] = [{"": int(extra_group["IOProcessors"][0])}]

                    # Get the UUID of the storm track coverage if present.
                    if extra_group.__contains__("HurricaneCoverage/GUID"):
                        storm_track_uuid = str(extra_group["HurricaneCoverage/GUID"][0].decode('UTF-8'))

                filenames[15] = str(proj_grp["ADCIRC Control File"][0].decode('UTF-8'))
                sim_name = os.path.basename(filenames[15])
                sim_name = os.path.splitext(sim_name)[0]

                # Read the fort.15
                fort15reader = Fort15Reader(
                    filename=filenames[15],
                    query=self.query,
                    comp_dir=comp_dir,
                    global_time=global_time,
                    geom_uuid=geom_uuid,
                    sim_name=sim_name,
                    storm_track_uuid=storm_track_uuid,
                    alt_filenames=filenames,
                    overwrite_mc_widgets=overwrite_widgets,
                    read_project_dsets=True
                )
                fort15reader.set_mesh_name(sim_name)
                if os.path.isfile(filenames[15]):
                    fort15reader.read()
                    fort15reader.send()
