"""A base component class with common functionality for components in AdH."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.display_options_io import read_display_option_ids

# 4. Local modules
from xms.adh.file_io import io_util


class AdHComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the AdH model."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file.strip('"\''))
        self.data = None
        self.tree_commands = []  # [(menu_text, menu_method)...]
        self.polygon_commands = []  # [(menu_text, menu_method)...]
        self.arc_commands = []  # [(menu_text, menu_method)...]
        self.point_commands = []  # [(menu_text, menu_method)...]
        self.uuid = os.path.basename(os.path.dirname(self.main_file))

    @property
    def component_path(self) -> str:
        """Retrieves the path of the component files.

        Returns:
            The directory path as a string.
        """
        return str(Path(self.main_file).parent)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        messages = []
        action_requests = []

        # Check if we are already in the new location
        if not self.copy_component_folder(new_path):  # We are already in the new location (simple save)
            return self.main_file, messages, action_requests

        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))
        return new_main_file, messages, action_requests

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list` of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.

        """
        if not main_file_list or not self.tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # Add all the project explorer menus
        for command_text, command_method in self.tree_commands:
            action = ActionRequest(
                modality='MODAL',
                method_name=command_method,
                class_name=self.class_name,
                module_name=self.module_name,
                main_file=self.main_file
            )
            menu_item = MenuItem(text=command_text, action=action)
            menu_list.append(menu_item)
        return menu_list

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two strings where the first is the file
                location of the XMS coverage id binary file. The second string is file location of the component
                coverage id binary file. Only applicable for coverage selections. The file will be deleted after the
                event. The file must be copied if it needs to persist.

        Returns:
            menu_items (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.

        """
        menu_list = [None]  # None == spacer
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                id_files[entity] = ('', '')
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        menu_list = self.add_menu_items_for_selection(menu_list, selection, unpacked_id_files)
        if not menu_list:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return menu_list

    def add_menu_items_for_selection(self, menu_list, selection, unpacked_id_files):
        """Adds menu items based on the selected feature objects.

        Args:
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                    that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None"
                    may be added to the list to indicate a separator.
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                    The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            unpacked_id_files (dict): A dictionary with the key being an entity type, and the value being a tuple.
                    The first value of the tuple is a temporary file of XMS ids. The second value of the tuple is
                    a temporary file of component ids.

        Returns:
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.
        """
        if 'POLYGON' in selection and selection['POLYGON']:
            poly_id_files = unpacked_id_files['POLYGON'] if 'POLYGON' in unpacked_id_files else None
            for command_text, command_method in self.polygon_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters={
                        'id_files': poly_id_files,
                        'selection': selection['POLYGON']
                    }
                )
                menu_item = MenuItem(text=command_text, action=action)
                menu_list.append(menu_item)
        if 'ARC' in selection and selection['ARC']:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for command_text, command_method in self.arc_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters={
                        'id_files': arc_id_files,
                        'selection': selection['ARC']
                    }
                )
                menu_item = MenuItem(text=command_text, action=action)
                menu_list.append(menu_item)
        if ('POINT' in selection and selection['POINT']) or ('NODE' in selection and selection['NODE']):
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
            node_id_files = unpacked_id_files['NODE'] if 'NODE' in unpacked_id_files else None
            for command_text, command_method in self.point_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters={
                        'id_files': point_id_files,
                        'selection': selection['POINT'],
                        'node_id_files': node_id_files,
                        'node_selection': selection['NODE']
                    }
                )
                menu_item = MenuItem(text=command_text, action=action)
                menu_list.append(menu_item)
        return menu_list

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        if self.tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            action = ActionRequest(
                main_file=self.main_file,
                modality='MODAL',
                class_name=self.class_name,
                module_name=self.module_name,
                method_name=self.tree_commands[0][1]
            )
            actions.append(action)
        return messages, actions

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double click in the main display area of XMS happened.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)
        unpacked_id_files = {}

        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                id_files[entity] = ('', '')
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        actions = []
        if 'POLYGON' in selection:
            poly_id_files = unpacked_id_files['POLYGON'] if 'POLYGON' in unpacked_id_files else None
            for _, command_method in self.polygon_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters={
                        'id_files': poly_id_files,
                        'selection': selection['POLYGON']
                    }
                )
                actions.append(action)
                break  # Only expecting one dialog ActionRequest on double-click
        if 'ARC' in selection:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for _, command_method in self.arc_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters={
                        'id_files': arc_id_files,
                        'selection': selection['ARC']
                    }
                )
                actions.append(action)
                break  # Only expecting one dialog ActionRequest on double-click
        if not actions:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return [], actions

    def get_display_refresh_action(self):
        """Get an ActionRequest that will refresh the XMS display list for components with display."""
        action = ActionRequest(
            modality='NO_DIALOG',
            method_name='get_initial_display_options',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            comp_uuid=self.uuid
        )
        return action

    @staticmethod
    def remove_id_files(file_dict):
        """Clean up the files dumped by XMS.

        Args:
            file_dict (:obj:`dict`): A dictionary with str entity type as key. Value is a tuple of two filenames.
        """
        for filenames in file_dict.values():
            io_util.remove(filenames[0])
            io_util.remove(filenames[1])

    def clean_all(self, query, params):
        """Query XMS for a dump of all the current component ids of the specified entity type.

        Query needs to be at the Component Context level (or any
        other node with a "ComponentCoverageIds" ContextDefinition out edge).

        Args:
            query (xmsdmi.dmi.Query): Query for communicating with XMS.
            params (:obj:`dict'): Generic map of parameters.

        Returns:
            (dict): Key is stringified entity type and value is tuple of xms and component id files.

        """
        return [], []

    def _get_att_and_comp_ids(self, params):
        """Reads the files referenced in the ActionRequest arguments which hold the XMS attribute ids and component ids.

        Args:
            params (): Arguments from the ActionRequest, extracted from the list.

        Returns:
            A pair of lists, the first list of XMS attribute ids, the second a list of component ids.
        """
        att_ids = []
        comp_ids = []
        if 'id_files' in params and params['id_files']:
            att_ids = read_display_option_ids(params['id_files'][0])
            comp_ids = read_display_option_ids(params['id_files'][1])
        elif 'selection' in params:
            att_ids = params['selection']
            comp_ids = [-1 for _ in att_ids]
        return att_ids, comp_ids
