"""A component for the sediment transport constituents."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.gui.sediment_constituents_dialog import SedimentConstituentsDialog


class SedimentConstituentsComponent(AdHComponent):
    """A Dynamic Model Interface (DMI) component for the AdH sediment transport constituents."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data: SedimentConstituentsIO = SedimentConstituentsIO(self.main_file)
        if not os.path.exists(main_file):
            self.data.commit()
        self.tree_commands = [('Constituents...', 'open_constituents')]  # [(menu_text, menu_method)...]

    def open_constituents(self, _query, _params, win_cont):
        """Opens the Transport Constituents dialog and saves component data state on OK.

        Args:
            _query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            _params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        self.data = SedimentConstituentsIO(self.main_file)
        dlg = SedimentConstituentsDialog(win_cont, self.data)
        if dlg.exec():
            self.data = dlg.data
            self.data.commit()
        return [], []
