"""A component for the transport constituents."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.data.transport_constituents_io import TransportConstituentsIO
from xms.adh.gui.transport_constituents_dialog import TransportConstituentsDialog


class TransportConstituentsComponent(AdHComponent):
    """A Dynamic Model Interface (DMI) component for the AdH transport constituents."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = TransportConstituentsIO(self.main_file)
        if not os.path.exists(main_file):
            self.data.commit()
        self.tree_commands = [('Constituents...', 'open_constituents')]  # [(menu_text, menu_method)...]

    def open_constituents(self, query, params, win_cont):
        """Opens the Transport Constituents dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        self.data = TransportConstituentsIO(self.main_file)
        dlg = TransportConstituentsDialog(win_cont, self.data)
        if dlg.exec():
            self.data = dlg.data
            self.data.commit()
        return [], []
