"""AdH data protocol."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import TYPE_CHECKING

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid as CoGrid
from xms.data_objects.parameters import Coverage

# 4. Local modules
from xms.adh.components.vessel_component import VesselComponent
from xms.adh.data.adh_data import AdhData
from xms.adh.data.bc_io import BcIO
from xms.adh.data.materials_io import MaterialsIO
from xms.adh.data.model_control import ModelControl
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.data.sediment_materials_io import SedimentMaterialsIO
from xms.adh.data.transport_constituents_io import TransportConstituentsIO

if TYPE_CHECKING:
    from xms.adh.data.xms_data import XmsData


class AdhQueryData(AdhData):
    """AdH query data."""
    def __init__(self, xms_data: 'XmsData'):
        """Initialize AdhQueryData."""
        super().__init__()
        self._xms_data = xms_data

    def _load_co_grid(self) -> CoGrid | None:
        """
        Loads the grid.

        Returns:
            CoGrid: The grid to be loaded.
        """
        return self._xms_data.co_grid

    def _load_geom_uuid(self) -> str | None:
        """
        Loads the geom uuid.

        Returns:
            str: The geom uuid.
        """
        return self._xms_data.geom_uuid

    def _load_model_control(self) -> ModelControl | None:
        """
        Loads the model control.

        Returns:
            ModelControl: The model control.
        """
        data = self._xms_data.sim_component.data if self._xms_data.sim_component else None
        return data

    def _load_bc_io(self) -> BcIO | None:
        """
        Loads the bc io.

        Returns:
            BcIO: The bc io.
        """
        if self._xms_data.bc_component is not None:
            return self._xms_data.bc_component.data
        return None

    def _load_materials_io(self) -> MaterialsIO | None:
        """
        Loads material IO data.

        Returns:
            The loaded material data, or None if material coverage is not available.
        """
        if self._xms_data.material_component is not None:
            return self._xms_data.material_component.data
        return None

    def _load_transport_constituents_io(self) -> TransportConstituentsIO | None:
        """
        Loads the transport constituents IO.

        Returns:
            The transport constituents IO.
        """
        if self._xms_data.transport_component is not None:
            return self._xms_data.transport_component.data
        return None

    def _load_sediment_materials_io(self) -> SedimentMaterialsIO | None:
        """
        Loads the sediment materials IO.

        Returns:
            The sediment materials IO.
        """
        if self._xms_data.sediment_material_component is not None:
            return self._xms_data.sediment_material_component.data
        return None

    def _load_sediment_constituents_io(self) -> SedimentConstituentsIO | None:
        """
        Loads the sediment constituents IO.

        Returns:
            The sediment constituents IO.
        """
        if self._xms_data.sediment_constituents_component is not None:
            return self._xms_data.sediment_constituents_component.data
        return None

    def _load_vessel_coverages(self) -> list[Coverage] | None:
        """
        Loads the vessel coverages.

        Returns:
            The list of vessel coverages.
        """
        return self._xms_data.vessel_coverages

    def _load_vessel_components(self) -> list[VesselComponent] | None:
        """
        Loads the vessel components.

        Returns:
            The list of vessel components.
        """
        return self._xms_data.vessel_components
