"""Write the AdH simulation files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import traceback
from typing import Optional

# 2. Third party modules
from adhparam import file_io

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.adh.data.hot_start_helper import merge_hot_start_datasets
from xms.adh.data.vessel_data_to_adh_param import VesselDataToAdhParam
from xms.adh.data.xms_data import XmsData
from xms.adh.data.xms_data_to_adhparam import XmsDataToAdhParam
from xms.adh.data.xms_query_data import XmsQueryData


def adh_write(query: Optional[Query] = None, logger: Optional[logging.Logger] = None):
    """
    Reads an AdH bc file and converts it to an XMS AdH model.

    Args:
        query: The object that communicates with XMS. Can be None for testing.
        logger: An instance of the logging.Logger class to log any error messages.
    """
    if query is None:
        query = Query()
    xms_data = XmsQueryData(query)
    write_adh_simulation(xms_data, logger)


def write_adh_simulation(xms_data: XmsData, logger: logging.Logger = None) -> bool:
    """
    Writes AdH simulation files.

    Args:
        xms_data (XmsData): The XMS data.
        logger (logging.Logger): The logger object for logging messages. Default is None.

    Returns:
        True if writing was successful.
    """
    success = True
    try:
        if logger is None:
            logger = logging.getLogger("xms.adh")
        logger.info("Building AdH data.")
        data_getter = XmsDataToAdhParam(xms_data, logger)
        data_getter.get_data()
        if not data_getter.valid_inputs():
            return False
        param_adh, sim_name, hotstarts = data_getter.get_adh_param()
        logger.info("Writing boundary condition file.")
        file_io.write_bc_file(sim_name + ".bc", param_adh.boundary_conditions, param_adh.model_control, True)
        logger.info("Writing nodal output mapping file.")
        file_io.write_nodal_output_file(sim_name + ".prn_map", param_adh.model_control)
        logger.info("Writing mesh file.")
        file_io.write_mesh_file(sim_name + ".3dm", param_adh.mesh)
        logger.info("Writing hot start file.")
        hotstarts = merge_hot_start_datasets(hotstarts, logger)
        file_io.write_hot_start_file(sim_name + ".hot", hotstarts)

        vessel_getter = VesselDataToAdhParam(xms_data, logger)
        vessel_list = vessel_getter.get_data()
        if vessel_list.vessels:
            logger.info("Writing vessel file.")
            file_io.write_bt_file(sim_name + ".bt", vessel_list)

        # data_getter.save_xms_data(base_path="c:/temp/xms_data")
    except Exception as ex:
        logger.error("Unable to write AdH simulation.")
        with open(f'adh_export_error_{os.getpid()}.log', 'a') as file:
            traceback.print_exception(type(ex), ex, ex.__traceback__, file=file)
        raise
    return success
