"""This dialog allows users to assign boundary conditions to a set of points."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from adhparam.time_series import TimeSeries
import pandas as pd

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.adh.data.card_info import CardInfo
from xms.adh.gui.bc_point_attributes_dialog_ui import Ui_PointAttributesDialog
import xms.adh.gui.resources.adhqrc  # noqa F401
from xms.adh.gui.time_series_editor import TimeSeriesEditor


class BcPointDialog(XmsDlg):
    """A dialog for assigning materials to polygons."""
    def __init__(self, win_cont, title):
        """Allows the user to edit arc boundary conditions.

        Args:
            win_cont (QWidget): Parent window
            title (str): Window title
            data (MaterialsIO): Materials object
        """
        super().__init__(win_cont, 'xms.adh.gui.bc_point_attributes_dialog')
        self.ui = Ui_PointAttributesDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Boundary_Condition_Assignment'

        self.setWindowTitle(title)
        self.wind_units = ['seconds', 'minutes', 'hours', 'days', 'weeks']
        # 0 = seconds, 1 = minutes, 2 = hours, 3 = days, 4 = weeks
        self.point_types = CardInfo.point_types
        self.card_to_index = CardInfo.point_card_to_index
        # tuple has card name, series button
        self.CARD = 0
        self.SERIES_BTN = 1
        self.WIDGETS = 2
        self.XY_COLUMNS = [3, 3, 3, 3, 3, 4]
        self.index_to_card = [
            (None, (), self.ui.off_page, None),
            # ('OVL', (self.ui.ovl_x, self.ui.ovl_y), self.ui.ovl_page),  # not in AdH 4.6
            (
                'OVH', (self.ui.ovh_x, self.ui.ovh_y, self.ui.ovh_depth), self.ui.ovh_page, [
                    {
                        'X': 'Time (sec)',
                        'Y': 'Vx'
                    }, {
                        'X': 'Time (sec)',
                        'Y': 'Vy'
                    }, {
                        'X': 'Time (sec)',
                        'Y': 'Depth'
                    }
                ]
            ),
            ('LDE', (self.ui.lde_elevation, ), self.ui.lde_page, [{
                'X': 'Time (sec)',
                'Y': 'Elevation'
            }]),
            ('LDH', (self.ui.ldh_depth, ), self.ui.ldh_page, [{
                'X': 'Time (sec)',
                'Y': 'Depth'
            }]),
            ('LID', (self.ui.lid_draft, ), self.ui.lid_page, [{
                'X': 'Time (sec)',
                'Y': 'Draft'
            }]),
            ('WND', (self.ui.wnd_wind, ), self.ui.wnd_page, [{
                'X': 'Time',
                'Y': 'X-component',
                'Z': 'Y-component'
            }])
        ]
        self.series_data = {}
        self._setup()

    def _setup(self):
        """Sets up the signals for the UI and adjusts sizes."""
        self.ui.point_type.addItems(self.point_types)
        self.ui.point_type.currentIndexChanged[int].connect(self._point_type_changed)

        self.ui.wnd_wind_units.addItems(self.wind_units)
        # Units combo is set later, because the time series data will be set later

        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.point_type.setCurrentIndex(self.card_to_index['OVH'])
        self.adjustSize()
        self.ui.point_type.setCurrentIndex(self.card_to_index[''])
        for idx, card in enumerate(self.index_to_card):
            for series_idx, btn in enumerate(card[self.SERIES_BTN]):
                btn.clicked.connect(self._get_open_series_dialog_lambda(idx, series_idx))

    def _get_open_series_dialog_lambda(self, idx, series_idx):
        """Sets the type of the current point.

        Args:
            idx (int): The card index.
            series_idx (int): The series index for the card.

        Returns:
            A lambda for the open series dialog.
        """
        return lambda: self._open_series_dialog(idx, series_idx)

    def set_point_type(self, card_type, is_mixed):
        """Sets the type of the current point.

        Args:
            card_type (str): The text of the card.
            is_mixed (bool): True if the attributes are for multiple things.
        """
        if is_mixed:
            self.ui.warning_label.setText('All selected points will be reassigned to the current options on ok.')
            self.ui.warning_label.setStyleSheet("QLabel { color : red; }")
        else:
            self.ui.warning_label.hide()
        self.ui.point_type.setCurrentIndex(self.card_to_index[card_type])
        if card_type == '':
            self._point_type_changed(self.card_to_index[card_type])

    def set_series(self, all_series):
        """Sets the series of the current card.

        Args:
            all_series (list): A list of all of the series used by the current card.
        """
        idx = self.ui.point_type.currentIndex()
        self.series_data[idx] = all_series

    def set_units_for_series(self):
        """Sets the units combo box."""
        # 5 = 'WND'
        card_idx = 5
        # Only one series for wind
        series_idx = 0
        ts = self._get_time_series(card_idx, series_idx)
        self.ui.wnd_wind_units.setCurrentIndex(ts.units)
        self.ui.wnd_wind_units.currentIndexChanged[int].connect(self._wnd_wind_units_changed)

    def get_series(self, series_idx):
        """Gets the series for the current card index.

        Args:
            series_idx (int): The series index for the current card.

        Returns:
            An XY series, or None if not found.
        """
        idx = self.ui.point_type.currentIndex()
        if idx in self.series_data:
            if series_idx < len(self.series_data[idx]):
                return self.series_data[idx][series_idx]
            else:
                return None
        else:
            return None

    def get_point_card_type(self):
        """Gets the series for the current card index.

        Returns:
            A string of the current card.
        """
        idx = self.ui.point_type.currentIndex()
        return self.index_to_card[idx][self.CARD]

    def _point_type_changed(self, index):
        """Called when the point type combobox selection changed.

        Args:
            index (int): The index for the current card.
        """
        self.ui.card_options_stack.setCurrentWidget(self.index_to_card[index][self.WIDGETS])

    def _wnd_wind_units_changed(self, index):
        """Called when the wind unit combobox selection changed.

        Args:
            index (int): The index for the time units.
        """
        # 5 = 'WND'
        card_idx = 5
        # Only one series for wind
        series_idx = 0
        ts = self._get_time_series(card_idx, series_idx)
        ts.units = index

    def _get_time_series(self, card_index, series_index):
        """Returns the time series.

        Args:
            card_index (int): The current card index.
            series_index (int): The series index for the current card.
        """
        if card_index not in self.series_data:
            self.series_data[card_index] = []
            for _ in self.index_to_card[card_index][self.SERIES_BTN]:
                self.series_data[card_index].append(TimeSeries())
                if self.XY_COLUMNS[card_index] == 3:
                    df = pd.DataFrame([[0.0, 0.0]], columns=['X', 'Y'])
                else:
                    df = pd.DataFrame([[0.0, 0.0, 0.0]], columns=['X', 'Y', 'Z'])
                self.series_data[card_index][-1].time_series = pd.concat(
                    [self.series_data[card_index][-1].time_series, df]
                )
        ts = self.series_data[card_index][series_index]

        return ts

    def _open_series_dialog(self, card_index, series_index):
        """Opens the XY series editor.

        Args:
            card_index (int): The current card index.
            series_index (int): The series index for the current card.
        """
        ts = self._get_time_series(card_index, series_index)
        series = ts.time_series

        column_map = self.index_to_card[card_index][3][series_index]
        series = series.rename(columns=column_map)

        time_series_editor = TimeSeriesEditor(series, icon=self.windowIcon(), parent=self, column_map=column_map)
        if time_series_editor.run():
            self.series_data[card_index][series_index].time_series = time_series_editor.series

    def get_number_of_series(self):
        """Gets the number of possible series for the current card index.

        Returns:
            The number of XY series used by the card.
        """
        idx = self.ui.point_type.currentIndex()
        return len(self.index_to_card[idx][self.SERIES_BTN])

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when the Ok button is clicked."""
        return super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        return super().reject()
