"""This module defines a filter model that can be used for defining bed layers."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QModelIndex, Qt

# 3. Aquaveo modules
from xms.guipy.models.rename_model import RenameModel

# 4. Local modules


class BedLayerModel(RenameModel):
    """A model to rename header titles, hide unwanted options."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        self._ID_SOURCE_COLUMN = 0
        self._THICKNESS_SOURCE_COLUMN = 1
        self._POROSITY_SOURCE_COLUMN = 2
        self._CRITCAL_SHEAR_SOURCE_COLUMN = 3
        self._EROSION_CONSTANT_SOURCE_COLUMN = 4
        self._EROSION_EXPONENT_SOURCE_COLUMN = 5
        self._ID_COLUMN = 1
        self._THICKNESS_COLUMN = 0
        self.show_cohesion = True
        super().__init__(
            ['Thickness', 'Constituents', 'Porosity', 'Critical Shear', 'Erosion Constant', 'Erosion Exponent'], parent
        )

    def filterAcceptsColumn(self, source_column: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Filters out the 'ID' column.

        Args:
            source_column (int): The column index in the source model.
            source_parent (QModelIndex): The parent index of the view.

        Returns:
            True if the model should keep the column, false if the column should be hidden.
        """
        if source_column in [
            self._POROSITY_SOURCE_COLUMN, self._CRITCAL_SHEAR_SOURCE_COLUMN, self._EROSION_CONSTANT_SOURCE_COLUMN,
            self._EROSION_EXPONENT_SOURCE_COLUMN
        ]:
            return self.show_cohesion
        return True

    def flags(self, index: QModelIndex):
        """Decides flags for things like enabled state.

        Args:
            index (QModelIndex): The index in the

        Returns:
            Item flags for the model index.
        """
        local_flags = super(BedLayerModel, self).flags(index)
        return local_flags

    def set_visible_cohesive_properties(self, use_cohesion):
        """Changes the visible columns based on whether we are using cohesion.

        Args:
            use_cohesion (bool): True if cohesive properties are to be used.
        """
        self.show_cohesion = use_cohesion
        self.invalidateFilter()

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (int): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (int): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole or orientation == Qt.Horizontal:
            return super().headerData(section, orientation, role)

        if section == 0:
            return 'Top'
        elif section == self.rowCount() - 1:
            return 'Bottom'
        else:
            return '-'

    def mapFromSource(self, source_index: QModelIndex) -> QModelIndex:  # noqa: N802
        """Maps the index of the source model to this model. This is for swapping the first two columns in the table.

        Args:
            source_index (QModelIndex): The index in the source model that needs to be mapped.

        Returns:
            A QModelIndex of the source model for the given index of our model.
        """
        proxy_column = source_index.column()
        if proxy_column == self._THICKNESS_SOURCE_COLUMN:
            proxy_column = self._THICKNESS_COLUMN
        elif proxy_column == self._ID_SOURCE_COLUMN:
            proxy_column = self._ID_COLUMN
        return self.index(source_index.row(), proxy_column, source_index.parent())

    def mapToSource(self, proxy_index: QModelIndex) -> QModelIndex:  # noqa: N802
        """Maps the index of this model to the source model. This is for swapping the first two columns in the table.

        Args:
            proxy_index (QModelIndex): The index in this model that needs to be mapped.

        Returns:
            A QModelIndex of the source model for the given index of our model.
        """
        source_column = proxy_index.column()
        if source_column == self._THICKNESS_COLUMN:
            source_column = self._THICKNESS_SOURCE_COLUMN
        elif source_column == self._ID_COLUMN:
            source_column = self._ID_SOURCE_COLUMN
        return self.sourceModel().index(proxy_index.row(), source_column, proxy_index.parent())

    def set_bed_layer_type(self, protocol):
        """Sets the header text based on the bed layer assignment protocol.

        Args:
            protocol (int): 0 if bed layer thickness assigned directly on SB cards, 1 if bed layer thickness assigned
            by strata elevation horizon on SB cards.
        """
        if protocol == 0:
            self.column_names[self._THICKNESS_COLUMN] = 'Thickness'
        else:
            self.column_names[self._THICKNESS_COLUMN] = 'Elevation'
        self.headerDataChanged.emit(Qt.Horizontal, self._THICKNESS_COLUMN, self._THICKNESS_COLUMN)
