"""A dialog for the global sediment options."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa: AQU103
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.data.sediment_materials_io import SedimentMaterialsIO
from xms.adh.gui.global_sediment_dialog_ui import Ui_GlobalSedimentDialog
from xms.adh.gui.widgets.bed_layer_table_widget import BedLayerTableWidget
from xms.adh.gui.widgets.consolidation_table_widget import ConsolidationTableWidget
from xms.adh.gui.widgets.transport_constituent_assignment_widget import TransportConstituentAssignmentWidget


class GlobalSedimentDialog(XmsDlg):
    """A dialog for setting global sediment options."""
    def __init__(self, win_cont, pe_tree, sediment_data):
        """Allows the user to global sediment options.

        Args:
            win_cont (QWidget): Parent window
            pe_tree (TreeNode): The project explorer tree.
            sediment_data (SedimentMaterialsIO): The sediment options with materials.
        """
        super().__init__(win_cont, 'xms.adh.gui.global_sediment_dialog')
        self.ui = Ui_GlobalSedimentDialog()
        self.ui.setupUi(self)
        self.old_constituents_uuid = ''
        self.data = sediment_data
        self.sediment = TransportConstituentAssignmentWidget(self, pe_tree, True, True)
        self.sediment.hide_for_selection_only()
        self.bed_layer_table = BedLayerTableWidget(self, self.data, 0, self.sediment.constituents)
        self.sediment.changed_sediment_constituents[SedimentConstituentsIO].connect(
            self.bed_layer_table.constituents_changed
        )
        self.consolidation_table = ConsolidationTableWidget(self, self.data, 0)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Sediment_Library_Control'

        self.dbl_valid = QxDoubleValidator()
        self.hid_dbl_valid = QxDoubleValidator()
        self.number_corrector = NumberCorrector(self)
        self.hid_dbl_valid.setDecimals(10)
        self.dbl_valid.setDecimals(10)
        self.hid_dbl_valid.setBottom(0.0)
        self.hid_dbl_valid.setTop(1.0)

        self._setup()

    def _setup(self):
        """Sets up the sediment transport constituent assignment widget."""
        self.ui.bed_layer_layout.insertWidget(0, self.sediment)
        self.ui.bed_layer_layout.addWidget(self.bed_layer_table)
        self.ui.consolidation_layout.addWidget(self.consolidation_table)

        self._setup_combo_boxes()
        self._setup_edit_fields()
        self._setup_toggles()

        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def _setup_combo_boxes(self):
        """Sets up the combo boxes with their options, current text, and connections."""
        # Set up the options in the combo boxes in the general tab.
        self.ui.csv_combo.addItems(SedimentMaterialsIO.CSV_OPTIONS)
        self.ui.wws_combo.addItems(SedimentMaterialsIO.WWS_OPTIONS)
        self.ui.nse_combo.addItems(SedimentMaterialsIO.NSE_OPTIONS)
        self.ui.nbe_combo.addItems(SedimentMaterialsIO.NBE_OPTIONS)
        self.ui.hid_combo.addItems(SedimentMaterialsIO.HID_OPTIONS)
        self.ui.bed_layer_type_combo.addItems(SedimentMaterialsIO.MP_NBL_OPTIONS)

        # Set the current option for each combo box from the data that was passed into this dialog.
        self.ui.csv_combo.setCurrentText(self.data.info.attrs['cohesive_settling_velocity_method'])
        self.ui.wws_combo.setCurrentText(self.data.info.attrs['wind_wave_shear_method'])
        self.ui.nse_combo.setCurrentText(self.data.info.attrs['noncohesive_suspended_method'])
        self.ui.nbe_combo.setCurrentText(self.data.info.attrs['noncohesive_bedload_method'])
        self.ui.hid_combo.setCurrentText(self.data.info.attrs['noncohesive_hiding_method'])
        self.ui.bed_layer_type_combo.setCurrentIndex(self.data.info.attrs['bed_layer_assignment_protocol'])

        # Connect the combo boxes to slots for making other widgets visible or not depending on the current option.
        self.ui.csv_combo.currentTextChanged[str].connect(self.csv_changed)
        self.ui.nbe_combo.currentTextChanged[str].connect(self.nbe_changed)
        self.ui.hid_combo.currentTextChanged[str].connect(self.hid_changed)
        self.ui.bed_layer_type_combo.currentIndexChanged[int].connect(self.bed_layer_table.bed_layer_type_changed)

        # Make sure the right widgets are visible for the current options.
        self.csv_changed(self.ui.csv_combo.currentText())
        self.nbe_changed(self.ui.nbe_combo.currentText())
        self.hid_changed(self.ui.hid_combo.currentText())
        self.bed_layer_table.bed_layer_type_changed(self.ui.bed_layer_type_combo.currentIndex())

    def _setup_edit_fields(self):
        """Sets up the edit fields with their ranges, and current text."""
        # Set the current values.
        self.ui.a_edit.setText(str(self.data.info.attrs['a_csv']))
        self.ui.b_edit.setText(str(self.data.info.attrs['b_csv']))
        self.ui.m_edit.setText(str(self.data.info.attrs['m_csv']))
        self.ui.n_edit.setText(str(self.data.info.attrs['n_csv']))
        self.ui.critical_sand_edit.setText(str(self.data.info.attrs['critical_shear_sand']))
        self.ui.critical_clay_edit.setText(str(self.data.info.attrs['critical_shear_clay']))
        self.ui.hid_factor_edit.setText(str(self.data.info.attrs['hiding_factor']))
        self.ui.sif_edit.setText(str(self.data.info.attrs['sediment_infiltration_factor']))

        # Set the validators.
        self.ui.a_edit.setValidator(self.dbl_valid)
        self.ui.b_edit.setValidator(self.dbl_valid)
        self.ui.m_edit.setValidator(self.dbl_valid)
        self.ui.n_edit.setValidator(self.dbl_valid)
        self.ui.critical_sand_edit.setValidator(self.dbl_valid)
        self.ui.critical_clay_edit.setValidator(self.dbl_valid)
        self.ui.hid_factor_edit.setValidator(self.hid_dbl_valid)
        self.ui.sif_edit.setValidator(self.dbl_valid)

        # Set number correctors on the edit fields so bad entries can't be entered.
        self.ui.a_edit.installEventFilter(self.number_corrector)
        self.ui.b_edit.installEventFilter(self.number_corrector)
        self.ui.m_edit.installEventFilter(self.number_corrector)
        self.ui.n_edit.installEventFilter(self.number_corrector)
        self.ui.critical_sand_edit.installEventFilter(self.number_corrector)
        self.ui.critical_clay_edit.installEventFilter(self.number_corrector)
        self.ui.hid_factor_edit.installEventFilter(self.number_corrector)
        self.ui.sif_edit.installEventFilter(self.number_corrector)

    def _setup_toggles(self):
        """Sets up the check boxes with their current state."""
        use_cohesive_bed_layers = self.data.info.attrs['use_cohesive_bed_layers'] != 0
        use_consolidation = self.data.info.attrs['use_consolidation'] != 0
        use_infiltration = self.data.info.attrs['use_sediment_infiltration_factor'] != 0
        self.ui.use_cohesion.setCheckState(Qt.Checked if use_cohesive_bed_layers else Qt.Unchecked)
        self.ui.use_consolidation.setCheckState(Qt.Checked if use_consolidation else Qt.Unchecked)
        self.ui.use_sif.setCheckState(Qt.Checked if use_infiltration else Qt.Unchecked)

        self.ui.use_cohesion.stateChanged[int].connect(
            self.bed_layer_table.table_view.filter_model.set_visible_cohesive_properties
        )
        self.bed_layer_table.table_view.filter_model.set_visible_cohesive_properties(use_cohesive_bed_layers)

        self.ui.use_consolidation.stateChanged[int].connect(self.consolidation_table.setEnabled)
        self.consolidation_table.setEnabled(use_consolidation)

        # setup the dependency for the sediment infiltration widgets.
        self.ui.use_sif.stateChanged[int].connect(self.ui.sif_label.setVisible)
        self.ui.use_sif.stateChanged[int].connect(self.ui.sif_edit.setVisible)
        self.ui.sif_label.setVisible(use_infiltration)
        self.ui.sif_edit.setVisible(use_infiltration)

    def csv_changed(self, csv_text):
        """This method handles the widget visible changes for the different SP CSV options.

        Args:
            csv_text (str): the current text in the csv combo box.
        """
        visible = False
        spacing = 0
        if csv_text == 'Hwang and Mehta':
            visible = True
            spacing = 6
        settling_widget_count = self.ui.settling_coeff_layout.count()
        for layout_idx in range(settling_widget_count):
            self.ui.settling_coeff_layout.itemAt(layout_idx).widget().setVisible(visible)
        self.ui.settling_coeff_layout.setVerticalSpacing(spacing)

    def nbe_changed(self, nbe_text):
        """This method handles the widget visible changes for the different SP NBE options.

        Args:
            nbe_text (str): the current text in the nbe combo box.
        """
        visible = False
        spacing = 0
        if nbe_text == 'Wilcock':
            visible = True
            spacing = 6
        shear_widget_count = self.ui.critcal_shear_layout.count()
        for layout_idx in range(shear_widget_count):
            self.ui.critcal_shear_layout.itemAt(layout_idx).widget().setVisible(visible)
        self.ui.critcal_shear_layout.setVerticalSpacing(spacing)

    def hid_changed(self, hid_text):
        """This method handles the widget visible changes for the different SP HID options.

        Args:
            hid_text (str): the current text in the hid combo box.
        """
        visible = False
        spacing = 0
        if hid_text == 'Parker and Klingeman':
            visible = True
            spacing = 6
        hiding_widget_count = self.ui.hiding_factor_layout.count()
        for layout_idx in range(hiding_widget_count):
            self.ui.hiding_factor_layout.itemAt(layout_idx).widget().setVisible(visible)
        self.ui.hiding_factor_layout.setVerticalSpacing(spacing)

    def set_transport(self, constituents_uuid, constituents, transport_name):
        """Sets the transport constituents and current assignments.

        This method should only be called by the owner of the dialog as part of setting up.

        Args:
            constituents_uuid (str): The uuid of the transport constituents component.
            constituents (SedimentConstituentsIO): The sediment transport constituents data.
            transport_name (str): The name of the transport constituents component as it shows in the project explorer.
        """
        self.old_constituents_uuid = constituents_uuid
        self.sediment.set_transport(constituents_uuid, constituents, transport_name=transport_name)
        self.bed_layer_table.allow_grain_size_reset = True
        self.bed_layer_table.grain_delegate.set_constituents(constituents)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Saves data from the widgets to be stored for later."""
        # Save text from the current combo box options.
        self.data.info.attrs['cohesive_settling_velocity_method'] = self.ui.csv_combo.currentText()
        self.data.info.attrs['wind_wave_shear_method'] = self.ui.wws_combo.currentText()
        self.data.info.attrs['noncohesive_suspended_method'] = self.ui.nse_combo.currentText()
        self.data.info.attrs['noncohesive_bedload_method'] = self.ui.nbe_combo.currentText()
        self.data.info.attrs['noncohesive_hiding_method'] = self.ui.hid_combo.currentText()
        self.data.info.attrs['bed_layer_assignment_protocol'] = self.ui.bed_layer_type_combo.currentIndex()

        # Save text from the edit fields
        self.data.info.attrs['a_csv'] = float(self.ui.a_edit.text())
        self.data.info.attrs['b_csv'] = float(self.ui.b_edit.text())
        self.data.info.attrs['m_csv'] = float(self.ui.m_edit.text())
        self.data.info.attrs['n_csv'] = float(self.ui.n_edit.text())
        self.data.info.attrs['critical_shear_sand'] = float(self.ui.critical_sand_edit.text())
        self.data.info.attrs['critical_shear_clay'] = float(self.ui.critical_clay_edit.text())
        self.data.info.attrs['hiding_factor'] = float(self.ui.hid_factor_edit.text())
        self.data.info.attrs['sediment_infiltration_factor'] = float(self.ui.sif_edit.text())

        # Save the check box states.
        self.data.info.attrs['use_cohesive_bed_layers'] = 1 if self.ui.use_cohesion.checkState() == Qt.Checked else 0
        self.data.info.attrs['use_consolidation'] = 1 if self.ui.use_consolidation.checkState() == Qt.Checked else 0
        self.data.info.attrs['use_sediment_infiltration_factor'] = \
            1 if self.ui.use_sif.checkState() == Qt.Checked else 0

        # Save the sediment constituents UUID
        self.data.info.attrs['sediment_transport_uuid'] = self.sediment.constituents_uuid

        # 0 is the default unassigned, or global, material.
        self.data.materials[SedimentMaterialsIO.UNASSIGNED_MAT].bed_layers = self.bed_layer_table.model.data_frame
        self.data.materials[SedimentMaterialsIO.UNASSIGNED_MAT].consolidation = \
            self.consolidation_table.model.data_frame

        # Clean out old sediment constituent distributions if the sediment constituents changed.
        if self.old_constituents_uuid != self.data.info.attrs['sediment_transport_uuid']:
            for mat_id in self.data.materials.keys():
                if mat_id == SedimentMaterialsIO.UNASSIGNED_MAT:
                    continue
                self.data.materials[mat_id].constituents = self.data.materials[mat_id].constituents.iloc[0:0]

        # Update the bed layers and consolidation times.
        self.data.update_from_global()
        return super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        return super().reject()
