"""DataFrameTable class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QModelIndex, Signal
from PySide2.QtWidgets import QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.file_selector_delegate import FileSelectorButtonDelegate
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.delegates.spin_box_delegate import SpinBoxDelegate
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.tool_core.table_definition import (
    ChoicesColumnType, FloatColumnType, InputFileColumnType, IntColumnType, StringColumnType, TableDefinition
)

# 4. Local modules


class DataFrameTable(QWidget):
    """Widget for pandas data frame table."""

    # Signals
    data_changed = Signal(QModelIndex, QModelIndex)

    def __init__(self, parent=None) -> None:
        """Initializes the class.

        Args:
            parent (QWidget): Parent window.
        """
        super().__init__(parent)
        self.setObjectName("hot_start_table")
        self.resize(400, 300)
        self._vertical_layout = QVBoxLayout(self)
        self._vertical_layout.setObjectName("_vertical_layout")
        self._table = QxTableView(self)
        self._table.setObjectName("table")
        self._vertical_layout.addWidget(self._table)

        self._table_def = None
        self._actions = None

    def setup(self, table_definition: TableDefinition, data_frame: pd.DataFrame):
        """Initializes the class.

        Args:
            table_definition (TableDefinition): Defines the column types.
            data_frame(pandas.DataFrame): The data as a pandas DataFrame.
        """
        self._table_def = table_definition
        widget_builder.style_table_view(self._table)
        self._add_data_to_table(data_frame)
        self._hide_columns()
        self._add_delegates()
        self._adjust_column_sizes()
        self._add_tool_tips()

    def get_values(self) -> pd.DataFrame:
        """Returns a copy of the values in the table as a pandas DataFrame."""
        return self._table.model().data_frame.copy()

    def set_values(self, values: pd.DataFrame):
        """Sets up the table with the values and the existing table definition (hopefully they're compatible)."""
        self.setup(table_definition=self._table_def, data_frame=values)

    def _add_data_to_table(self, data_frame: pd.DataFrame) -> None:
        """Adds data to the table.

        Args:
            data_frame(pandas.DataFrame): The data as a pandas DataFrame.
        """
        if data_frame is None:
            data_frame = self._table_def.to_pandas()  # Create an empty dataframe with the appropriate columns

        model = QxPandasTableModel(data_frame)
        self._table.setModel(model)

        # Set other model stuff
        table_def = self._table_def  # for short
        model.set_default_values({col_type.header: col_type.default for col_type in table_def.column_types})
        model.set_read_only_columns({i for i, col_type in enumerate(table_def.column_types) if not col_type.enabled})
        model.dataChanged.connect(self._on_data_changed)

    def _on_data_changed(self, top_left: QModelIndex, bottom_right: QModelIndex) -> None:
        """Called when the model sends the dataChanged signal.

        Args:
            top_left (QModelIndex): Top left of range modified.
            bottom_right (QModelIndex): Bottom right of range modified.
        """
        self.data_changed.emit(top_left, bottom_right)

    def _hide_columns(self):
        """Hides columns if necessary."""
        for i, column in enumerate(self._table_def.column_types):
            if isinstance(column, ChoicesColumnType):
                self._table.setColumnHidden(i, True)

    def _add_delegates(self) -> None:
        """Creates the column delegates using the column types."""
        check_box_columns = set()
        for col_idx, column_type in enumerate(self._table_def.column_types):
            delegate = None
            if isinstance(column_type, StringColumnType) and column_type.choices:
                delegate = QxCbxDelegate(self)
                if isinstance(column_type.choices, int):
                    delegate.set_choices_column(column_type.choices, self._table.model())
                else:
                    delegate.set_strings(column_type.choices)
                self._table.model().set_combobox_column(col_idx, column_type.choices)
            elif isinstance(column_type, IntColumnType):
                if column_type.spinbox:
                    delegate = SpinBoxDelegate(self, minimum=column_type.low, maximum=column_type.high)
                elif column_type.checkbox:
                    delegate = CheckBoxNoTextDelegate(self)
                    check_box_columns.add(col_idx)
            elif isinstance(column_type, FloatColumnType):
                validator = QxDoubleValidator(bottom=column_type.low, top=column_type.high, parent=self)
                delegate = EditFieldValidator(validator, self)
            elif isinstance(column_type, InputFileColumnType):
                project_file = XmsEnvironment.xms_environ_project_path()
                project_dir = Path(project_file).parent if project_file else ''
                delegate = FileSelectorButtonDelegate(
                    proj_dir=project_dir, caption='Select File', parent=self, file_filters=column_type.file_filter
                )

            if delegate:
                self._table.setItemDelegateForColumn(col_idx, delegate)

        if check_box_columns:
            self._table.model().set_checkbox_columns(check_box_columns)

    def _adjust_column_sizes(self) -> None:
        """Tries to adjust column sizes appropriately."""
        self._table.resizeColumnsToContents()
        self._table.horizontalHeader().setStretchLastSection(True)
        self._table.adjustSize()  # Seems necessary

    def _add_tool_tips(self) -> None:
        """Adds tool tips to the column headers."""
        tool_tips_dict = {index: column_type.tool_tip for index, column_type in enumerate(self._table_def.column_types)}
        self._table.model().set_horizontal_header_tooltips(tool_tips_dict)
