"""A base class widget for assigning transport constituents."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Signal
from PySide2.QtWidgets import QMessageBox, QWidget

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import ComponentItem, ProjectExplorerItem
from xms.api.tree import tree_util
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.data.transport_constituents_io import TransportConstituentsIO
from xms.adh.gui.widgets.transport_constituent_assignment_widget_ui import Ui_TransportConstituentAssignmentWidget


def _is_transport_constituent(item):
    """Checks if a project explorer item is an AdH Transport Constituents component.

    Args:
        item (TreeNode): An item from the project explorer.
    """
    if isinstance(item.data, ComponentItem):
        return item.unique_name == 'TransportConstituents' and item.model_name == 'AdH'
    else:
        return type(item.data) is ProjectExplorerItem


def _is_sediment_constituent(item):
    """Checks if a project explorer item is an AdH Sediment Transport Constituents component.

    Args:
        item (TreeNode): An item from the project explorer.
    """
    if isinstance(item.data, ComponentItem):
        return item.unique_name == 'SedimentConstituents' and item.model_name == 'AdH'
    else:
        return type(item.data) is ProjectExplorerItem


class TransportConstituentAssignmentWidget(QWidget):
    """A dialog for assigning transport constituents to strings."""
    changed_constituents = Signal(TransportConstituentsIO)
    changed_sediment_constituents = Signal(SedimentConstituentsIO)
    changed_use_constituents = Signal(bool)

    def __init__(self, parent, pe_tree, use_transport, is_sediment=False):
        """Allows the user to edit which transport constituents are used and how.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
            pe_tree (): The project explorer tree.
            use_transport (bool): True if transport is being used.
            is_sediment (bool): True if the transport is sediment transport.
        """
        super().__init__(parent)
        self.ui = Ui_TransportConstituentAssignmentWidget()
        self.ui.setupUi(self)
        self.is_sediment = is_sediment
        self.constituents = None
        self.constituents_uuid = None
        self.assignments = None
        self.model = None
        self.pe_tree = pe_tree
        if pe_tree:
            if is_sediment:
                tree_util.filter_project_explorer(self.pe_tree, _is_sediment_constituent)
            else:
                tree_util.filter_project_explorer(self.pe_tree, _is_transport_constituent)
        self.ui.constituents_button.pressed.connect(self._constituents_button_pressed)
        # use_transport is sometimes a panda dataframe
        # this causes problems, so let's get the bool and use that
        if not isinstance(use_transport, bool):
            use_transport_bool = use_transport.values[0]
        else:
            use_transport_bool = use_transport
        self.ui.transport_constituents_group.setChecked(use_transport_bool)
        self.ui.transport_constituents_group.toggled.connect(self.changed_use_constituents)
        if is_sediment:
            self.ui.transport_constituents_group.setTitle('Sediment transport constituents')
            self.ui.constituents_label.setText('Sediment transport constituents:')
            self.ui.change_warning_label.setText(
                'Note: Changing the sediment transport constituents here will change '
                'the sediment transport constituents for all feature objects.'
            )

    def set_transport(self, constituents_uuid, constituents, assignments=None, transport_name=''):
        """Sets the transport constituents and current assignments.

        Args:
            constituents_uuid (str): The uuid of the transport constituents component.
            constituents (TransportConstituentsIO/SedimentConstituentsIO): The transport constituents data.
            assignments (pandas.DataFrame): The current assignments to edit.
            transport_name (str): The name of the transport constituents component as it shows in the project explorer.
        """
        self.constituents = constituents
        self.constituents_uuid = constituents_uuid
        if transport_name:
            self.ui.current_constituents_label.setText(transport_name)
        else:
            self.ui.current_constituents_label.setText('(none selected)')
        if self.is_sediment:
            self.changed_sediment_constituents.emit(constituents)
        else:
            self.changed_constituents.emit(constituents)

    def _constituents_button_pressed(self):
        """Opens the dialog for the user to select a transport constituent from the project explorer."""
        if self.is_sediment:
            title_text = 'Select Sediment Transport Constituents'
        else:
            title_text = 'Select Transport Constituents'
        dlg = TreeItemSelectorDlg(
            title_text, ComponentItem, self.pe_tree, previous_selection=self.constituents_uuid, parent=self
        )
        if dlg.exec():
            new_uuid = dlg.get_selected_item_uuid()
            if new_uuid and new_uuid != self.constituents_uuid:
                # get the transport constituent
                comp_node = tree_util.find_tree_node_by_uuid(self.pe_tree, new_uuid)
                if comp_node:
                    main_file = comp_node.main_file
                    if self.is_sediment:
                        transport_comp = SedimentConstituentsIO(main_file)
                    else:
                        transport_comp = TransportConstituentsIO(main_file)
                    transport_name = comp_node.name
                    self.set_transport(new_uuid, transport_comp, transport_name=transport_name)
                else:
                    msg = QMessageBox(self)
                    msg.setWindowTitle('Error')
                    sed_text = 'sediment ' if self.is_sediment else ''
                    msg.setText(f'The AdH {sed_text}transport constituent component could not be retrieved.')
                    msg.exec_()
            elif not new_uuid:
                self.set_transport('', None)

    def hide_for_selection_only(self):
        """Hides the checkbox on the group and the table.

        Use this if this widget is only being used to select a transport constituent component.
        """
        self.ui.constituents_table.setVisible(False)
        self.ui.transport_constituents_group.setCheckable(False)
