"""Python wrapping for xms.api._xmsapi.dmi.ExecutableCommand."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.api._xmsapi.dmi import ExecutableCommand as CExecutableCommand
from xms.api.dmi.ActionRequest import ActionRequest


class ExecutableCommand:
    """The pure Python wrapper for C++ exposed xms.api._xmsapi.dmi.ExecutableCommand objects."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Args:
            **kwargs:

                executable (str): See executable.setter

                model (str): See model.setter

                executable_order (int): See executable_order.setter

                display_name (str): See display_name.setter

                run_weight (int): See run_weight.setter

                progress_script (str): See progress_script.setter

                executable_is_script (bool): See executable_is_script.setter

                plot_group (str): See plot_group.setter

                instance (CExecutableCommand): The C++ object to wrap

        """
        self._instance = kwargs.get('instance')
        if not self._instance:
            self._instance = CExecutableCommand()

        if 'executable' in kwargs:
            self._instance.SetExecutable(kwargs['executable'])
        if 'model' in kwargs:
            self._instance.SetModel(kwargs['model'])
        if 'executable_order' in kwargs:
            self._instance.SetExecutableOrder(kwargs['executable_order'])
        if 'display_name' in kwargs:
            self._instance.SetDisplayName(kwargs['display_name'])
        if 'run_weight' in kwargs:
            self._instance.SetRunWeight(kwargs['run_weight'])
        if 'progress_script' in kwargs:
            self._instance.SetProgressScript(kwargs['progress_script'])
        if 'executable_is_script' in kwargs:
            self._instance.SetExecutableIsScript(kwargs['executable_is_script'])
        if 'plot_group' in kwargs:
            self._instance.SetPlotGroup(kwargs['plot_group'])

    @property
    def executable(self):
        """Returns the XML-defined name of the executable or the absolute path to the executable Python script."""
        return self._instance.GetExecutable()

    @executable.setter
    def executable(self, exe_name):
        """Sets the name of the executable.

        Args:
            exe_name (str): Name of the executable. Must match an executable name defined in the model XML if
                executable is not a Python script. If executable is a Python script, should be the absolute path
                to the script file containing a __main__ entry point.
        """
        self._instance.SetExecutable(exe_name)

    @property
    def model(self):
        """Returns the name of the DMI model that runs this executable."""
        return self._instance.GetModel()

    @model.setter
    def model(self, model_name):
        """Sets the name of the DMI model that runs this executable.

        Args:
            model_name (str): Name of the executable's DMI model as defined in the model XML. Empty string if the
                executable runs through a resource installed with XMS (e.g. MPIexec)
        """
        self._instance.SetModel(model_name)

    @property
    def executable_order(self):
        """Returns the position of the executable in the model's run sequence."""
        return self._instance.GetExecutableOrder()

    @executable_order.setter
    def executable_order(self, order):
        """Sets the position of the executable in the model's run sequence.

        Args:
            order (int): Index of the executable in the model's run sequence
        """
        self._instance.SetExecutableOrder(order)

    @property
    def display_name(self):
        """Returns the name of the DMI model that runs this executable."""
        return self._instance.GetDisplayName()

    @display_name.setter
    def display_name(self, gui_text):
        """Sets the executable name text that will appear in the XMS GUI.

        Args:
            gui_text (str): The executable's display name text
        """
        self._instance.SetDisplayName(gui_text)

    @property
    def run_weight(self):
        """Returns the run progress weight factor for the executable as an int."""
        return self._instance.GetRunWeight()

    @run_weight.setter
    def run_weight(self, weight):
        """Sets the run progress weight factor for the executable.

        Args:
            weight (int): The estimated weight factor this executable process will be assigned in the model's run
                sequence. Overall progress percent is computed from an executable's run weight relative to the other
                executables in the model run sequence.
        """
        self._instance.SetRunWeight(weight)

    @property
    def progress_script(self):
        """Returns the import path to the progress script monitoring this executable."""
        return self._instance.GetProgressScript()

    @progress_script.setter
    def progress_script(self, script_path):
        """Sets the path to the progress script monitoring this executable.

        Args:
            script_path (str): Path to the executable's progress script relative to the model XML definition. Script
                should have an __main__ entry point that launches an xms.api.dmi.ProgressLoop progress loop.
        """
        self._instance.SetProgressScript(script_path)

    @property
    def executable_is_script(self):
        """Returns True if the executable is a Python script."""
        return self._instance.GetExecutableIsScript()

    @executable_is_script.setter
    def executable_is_script(self, is_script):
        """Sets the flag indicating whether this executable is a Python script or not.

        Args:
            is_script (bool): True if the executable is a Python script
        """
        self._instance.SetExecutableIsScript(is_script)

    @property
    def plot_group(self):
        """Returns the XML-defined plot group used with this executable."""
        return self._instance.GetPlotGroup()

    @plot_group.setter
    def plot_group(self, group):
        """Sets the XML-defined plot group used with this executable.

        Args:
            group (str): The XML-defined plot group to use with this executable (the 'unique_name' attribute of the
                <declare_plot_group> element).
        """
        self._instance.SetPlotGroup(group)

    @property
    def run_id(self):
        """Returns the executable's run id."""
        return self._instance.GetRunId()

    @property
    def run_group(self):
        """Returns the executable's run group id."""
        return self._instance.GetRunGroup()

    @property
    def commandline_args(self):
        """Returns a list of the executable commandline argument strings."""
        return self._instance.GetCommandlineArguments()

    @property
    def progress_args(self):
        """Returns a list of the executable's progress script commandline argument strings."""
        return self._instance.GetProgressArguments()

    @property
    def solution_files(self):
        """Returns a list of the executable's solution load ActionRequest objects."""
        return [ActionRequest(instance=solution) for solution in self._instance.GetSolutionFiles()]

    def set_run_group_and_id(self, run_group, run_id):
        """Set the run group and run id indexes.

        Note setting this attributes is only required for iterative model runs.

        Args:
            run_group (int): The index of the group this executable should run with if the model run is an iterative
                process.
            run_id (int): The index of this executable in its run group
        """
        self._instance.SetRunGroupAndId(run_group, run_id)

    def add_commandline_arg(self, cmd_arg):
        """Add a commandline argument to be passed to the executable.

        Args:
            cmd_arg (str): The unquoted commandline argument string. The entire argument will be quoted internally
                as required by the OS.
        """
        self._instance.AddCommandlineArgument(cmd_arg)

    def add_progress_arg(self, progress_arg):
        """Add a commandline argument to be passed to the executable's progress script.

        Args:
            progress_arg (str): The unquoted progress script commandline argument string. The entire argument will be
                quoted internally as required by the OS.
        """
        self._instance.AddProgressArgument(progress_arg)

    def add_solution_file(self, solution_action):
        """Add an ActionRequest to read solution file(s) after the model run.

        Args:
            solution_action (ActionRequest): A solution load ActionRequest to handle the loading of
                model outputs after model run is complete.
        """
        self._instance.AddSolutionFile(solution_action._instance)
