"""Python wrapping for xms.api._xmsapi.dmi.OrderBy."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.api._xmsapi.dmi import ExportSortOrder, OrderBy as COrderBy

ExportSortOrderStrings = {
    'ESO_NONE': ExportSortOrder.ESO_NONE,
    'ESO_CWISE': ExportSortOrder.ESO_CWISE,
    'ESO_CCWISE': ExportSortOrder.ESO_CCWISE,
    'ESO_IJ': ExportSortOrder.ESO_IJ,
    'ESO_JI': ExportSortOrder.ESO_JI,
}


class OrderBy:
    """The pure Python wrapper for C++ exposed xms.api._xmsapi.dmi.OrderBy objects."""
    def __init__(self, instance=None):
        """Construct the wrapper.

        Args:
            instance (xms.api._xmsapi.dmi.OrderBy): The C++ object to wrap
        """
        if instance is None:
            self._instance = COrderBy()
        else:
            self._instance = instance

    @property
    def i_ascending(self):
        """Returns True if order is ascending in the i-direction."""
        return self._instance.IsIAscending()

    @i_ascending.setter
    def i_ascending(self, ascending=True):
        """If True, i-direction is ascending."""
        self._instance.SetIAscending(ascending)

    @property
    def j_ascending(self):
        """Returns True if order is ascending in the i-direction."""
        return self._instance.IsJAscending()

    @j_ascending.setter
    def j_ascending(self, ascending=True):
        """If True, j-direction is ascending."""
        self._instance.SetJAscending(ascending)

    @property
    def sort_order(self):
        """Returns the export order enum as a string."""
        c_enum = self._instance.GetSortOrder()
        return next(key for key, value in ExportSortOrderStrings.items() if value == c_enum)

    @sort_order.setter
    def sort_order(self, order):
        """Sets the Python string constant associated with the OrderBy object's C++ ExportSortOrder enum.

        Args:
            order (str): One of the ExportSortOrderStrings keys
        """
        # Convert the Python string constant to the associated C++ enum ExportSortOrderStrings keys.
        order_enum = ExportSortOrderStrings.get(order.upper())
        if order_enum is None:
            raise ValueError(f'order must be one of the following options: {ExportSortOrderStrings.keys()}')
        self._instance.SetSortOrder(order_enum)

    def add_location_order(self, location):
        """Adds a location flag to the set of entities to be ordered.

        Args:
            location (xms.data_objects.parameters.FilterLocation): The location bit flag to include
        """
        self._instance.AddLocationOrder(location)

    def clear_location_orders(self):
        """Clear all the location flags of entities to be ordered."""
        self._instance.ClearLocationOrders()
