"""Utilities for operating on TreeNode representations of the XMS project explorer tree."""
# 1. Standard python modules
from typing import Collection
import warnings

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.api.tree import TreeNode

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


def _add_node(query_item, children):
    """Returns a TreeNode created from the query_item.

    Args:
        query_item (`xms.api.dmi.ProjectExplorerItem): The data_object representing a project explorer
            tree item
        children (:obj:`list` of :obj:`TreeNode`): Children of the query_item.

    Returns:
        See description.
    """
    node = TreeNode(query_item)
    for child in children:
        child.parent = node
    node.children = children
    return node


def _get_children(query_items):
    """Returns a list of children of a TreeNode.

    Args:
        query_items (list of xms.api.dmi.ProjectExplorerItem): Project explorer dump as returned by an XMS Query.

    Returns:
         See description.
    """
    children_list = []
    for tree_item in query_items:
        children = _get_children(tree_item.GetChildren())
        children_list.append(_add_node(tree_item, children))
    return children_list


class ProjectExplorerTreeCreator:
    """Creates a tree (TreeNode) of the Project Explorer info."""
    def create(self, query):
        """Returns a dict containing info about the Project Explorer.

        Note that this method should never be called directly. If you have a Query, access the project tree via
        the 'project_tree' property.

        Args:
            query (QueryImpl): Object for communicating with XMS.

        Returns:
             A dict of the Project Explorer as {item_name: (uuid, item_type, children_dict)}.
        """
        project = query._instance.Get('project_explorer')['project_explorer']
        if not project:
            return None

        project_tree = _get_children(project)
        if project_tree and len(project_tree) == 1:
            return project_tree[0]
        return None

    def copy(self, tree_node):
        """Creates a deep copy of an existing TreeNode.

        Args:
            tree_node (TreeNode): Root of the tree to copy

        Returns:
             TreeNode: Deep copy of the input tree
        """
        warnings.warn('ProjectExplorerTreeCreator.copy() is deprecated. Use tree_util.copy_tree().',
                      category=DeprecationWarning, stacklevel=2)
        return copy_tree(tree_node)


class ProjectExplorerTreeWriter:
    """Writes the project tree to a file for debugging and testing."""
    def __init__(self):
        """Construct the writer."""
        self._verbosity = 0

    def _write_node(self, node, lines, level):
        """Writes the tree node info to the file.

        Args:
            node (TreeNode): The node
            lines (list): Output list to append lines to
            level (int): Level of recursion. Used for indentation
        """
        spaces = ' ' * (level * 2)
        if self._verbosity <= 0:
            lines.append(f'{spaces}{node.name}\n')
        else:
            lines.append(f'{spaces}Name = {node.name}\n')
            lines.append(f'{spaces}UUID = {node.uuid}\n')
            lines.append(f'{spaces}Type = {type(node.data)}\n')
            if self._verbosity > 1:
                # Simulation, component, and coverage specific attributes - empty string if not applicable
                lines.append(f'{spaces}Model name = {node.model_name}\n')
                # Component specific attributes - empty string if not applicable
                lines.append(f'{spaces}Unique name = {node.unique_name}\n')
                lines.append(f'{spaces}Main file = {node.main_file}\n')
                # UGrid specific attributes - empty string if not applicable
                lines.append(f'{spaces}Num cells = {node.num_cells}\n')
                lines.append(f'{spaces}Num points = {node.num_points}\n')
                # Dataset specific attributes - empty string if not applicable
                lines.append(f'{spaces}Num times = {node.num_times}\n')
                lines.append(f'{spaces}Num components = {node.num_components}\n')
                lines.append(f'{spaces}Num vals = {node.num_vals}\n')
                lines.append(f'{spaces}Data location = {node.data_location}\n')
                # item_typename attribute added with data_objects 1.2.0. Defined in XMS.
                lines.append(f'{spaces}XMS item typename = {node.item_typename}\n')
        for child in node.children:
            self._write_node(child, lines, level + 1)

    def write(self, project_tree, verbosity=0):
        """Writes the project tree to a file.

        Args:
            project_tree (TreeNode): The starting node of the project explorer tree.
            verbosity (int): How verbose the output is. Higher numbers are more verbose.
        """
        self._verbosity = verbosity
        lines = []
        self._write_node(project_tree, lines, 0)
        return ''.join(lines)

    def write_to_file(self, project_tree, filename, verbosity=0):
        """Writes the project tree to a file.

        Args:
            project_tree (TreeNode): The starting node of the project explorer tree.
            filename (str): File path to write project tree to
            verbosity (int): How verbose the output is. Higher numbers are more verbose.
        """
        self._verbosity = verbosity
        lines = []
        self._write_node(project_tree, lines, 0)
        with open(filename, 'w') as file:
            file.writelines(lines)


def find_tree_node_by_uuid(tree_node, uuid):
    """Returns the Project Explorer node with the given uuid.

    Args:
        tree_node (TreeNode): A node of the Project Explorer tree.
        uuid (str): A uuid.

    Returns:
        See description.
    """
    if not tree_node:
        return None
    elif tree_node.uuid == uuid and not tree_node.is_ptr_item:
        return tree_node
    else:
        for child in tree_node.children:
            node = find_tree_node_by_uuid(child, uuid)
            if node:
                return node
        return None


def find_linked_pointers_by_uuid(tree_node, uuid, pointer_items):
    """Returns the Project Explorer linked pointer nodes with the given uuid.

    Args:
        tree_node (TreeNode): A node of the Project Explorer tree.
        uuid (str): UUID of pointer items to search for
        pointer_items (list): The linked pointer items with specified UUID

    Returns:
        See description.
    """
    if not tree_node:
        return
    if tree_node.is_ptr_item and tree_node.uuid == uuid:
        pointer_items.append(tree_node)
    for child in tree_node.children:
        find_linked_pointers_by_uuid(child, uuid, pointer_items)


def traverse_tree(tree_node, action):
    """Traverses the tree calling the action.

    Args:
        tree_node (TreeNode): A node of the Project Explorer tree.
        action: Function to call on each tree node.

    Returns:
        See description.
    """
    action(tree_node)
    for child in tree_node.children:
        traverse_tree(child, action)


def get_project_explorer_tree(query):
    """Returns a TreeNode with info about the Project Explorer items.

    Args:
        query (xms.api.dmi.Query): Object for communicating with GMS.

    Returns:
        See description.
    """
    creator = ProjectExplorerTreeCreator()
    project_tree = creator.create(query)
    return project_tree


def ancestor_of_type(starting_node, type_=None, xms_types: Collection[str] | None = None):
    """Searches up through the tree for a node whose item_type is type_ and returns it or None.

    Args:
        starting_node (TreeNode): The starting node.
        type_ (type): The TreeNode object type
        xms_types (Collection[str]): Collection (e.g. list, set) of str to compare to TreeNode.item_typename.

    Returns:
        A TreeNode if found, else None.
    """
    node = starting_node
    while node:
        if not node.is_ptr_item and (type(node.data) == type_ or xms_types and node.item_typename in xms_types):
            return node
        node = node.parent
    return None


def sibling_with_unique_name(starting_node, unique_name):
    """Searches starting_node's siblings for a node whose unique_name is unique_name.

    Args:
        starting_node (TreeNode): The starting node.
        unique_name (str): The TreeNode.unique_name

    Returns:
        A TreeNode if found, else None.
    """
    if starting_node and starting_node.parent and starting_node.parent.children:
        for child in starting_node.parent.children:
            if child.unique_name == unique_name:
                return child
    return None


def child_from_name(parent, child_name):
    """Returns the child with the given name.

    Args:
        parent (TreeNode): A tree node.
        child_name (str): Name of the child we're looking for.

    Returns:
        See description.
    """
    if not parent:
        return None
    for child in parent.children:
        if child.name == child_name:
            return child
    return None


def child_with_unique_name_in_list(parent, unique_name_list):
    """Searches the children for one with unique_name in unique_name_list.

    Args:
        parent (TreeNode):
        unique_name_list (list of str): List of strings.

    Returns:
        The child TreeNode or None if not found.
    """
    if parent and parent.children:
        for child in parent.children:
            if child.unique_name in unique_name_list:
                return child
    return None


def trim_project_explorer(pe_tree_root, parent_uuid):
    """Trim a project explorer tree so that it only includes items below a parent.

    Note:
        pe_tree_root will be changed. Make a copy via ProjectExplorerTreeCreator.copy before calling if you need
        to preserve the original tree.

    Args:
        pe_tree_root (TreeNode): Root of project explorer tree to trim
        parent_uuid (parent_uuid): UUID of the new root in the trimmed tree

    Returns:
        (TreeNode): Root of the trimmed project explorer tree, None if the parent is not found.
    """
    if not parent_uuid:
        return None  # Don't traverse the tree if we aren't going to find anything.

    if pe_tree_root.uuid == parent_uuid:
        pe_tree_root.parent = None
        return pe_tree_root  # This is the item we are looking for.

    for child in pe_tree_root.children:  # Check if item we are looking for is in children of this item.
        if child.uuid == parent_uuid:
            child.parent = None
            return child
        one_of_childs_children = trim_project_explorer(child, parent_uuid)
        if one_of_childs_children:
            return one_of_childs_children
    return None


def trim_tree_to_items_of_type(start_node, types=None, xms_types: Collection[str] | None = None):
    """Keeps only items whose type is in types and their ancestors.

    Note:
        pe_tree_root will be changed. Make a copy via ProjectExplorerTreeCreator.copy before calling if you need
        to preserve the original tree.

    Args:
        start_node (TreeNode): Root of project explorer tree to trim
        types: Collection of types
        xms_types (Collection[str]): Collection (e.g. list, set) of str to compare to TreeNode.item_typename.

    Returns:
        The root of the new tree.
    """
    if not start_node:
        return None

    children = start_node.children
    if not children:  # Terminal tree item
        return None

    xms_types = xms_types if xms_types else []
    root = TreeNode(other=start_node)  # Create a copy of the root tree item node.
    root.children = []  # Clear children in copy of root item. Want to filter down to selectable types.
    for child in children:
        childs_tree = trim_tree_to_items_of_type(child, types, xms_types)
        if childs_tree:
            root.add_child(childs_tree)
        elif not child.is_ptr_item and (types and type(child.data) in types or child.item_typename in xms_types):
            # Currently excluding all pointer items. Either the xmsapi object type or the XMS tree item type
            # matches the target types.
            root.add_child(child)

    root_matches = not root.is_ptr_item and (types and type(root.data) in types or root.item_typename in xms_types)
    return_root = root.children or root_matches
    return root if return_root else None


def filter_project_explorer(pe_tree_root, condition):
    """Filter a project explorer tree with a conditional method.

    Note:
        pe_tree_root will be changed. Make a copy via ProjectExplorerTreeCreator.copy before calling if you need
        to preserve the original tree.

    Args:
        pe_tree_root (TreeNode): Root of project explorer tree to filter
        condition: Method to apply to tree items. Method should take a TreeNode and
            return a bool. If True, the item and its children (if they satisfy the condition) will be included
            in the trimmed tree.
    """
    if not pe_tree_root:
        return
    filtered_children = []
    for child in pe_tree_root.children:
        if not condition(child):
            continue
        filter_project_explorer(child, condition)
        filtered_children.append(child)
    pe_tree_root.children = filtered_children


def copy_tree(tree_node: TreeNode) -> TreeNode | None:
    """Returns a deep copy of an existing TreeNode.

    Args:
        tree_node: Root of the tree to copy.

    Returns:
        See description.
    """
    if not tree_node:
        return None

    project_tree = _get_children([tree_node.data])
    if project_tree and len(project_tree) == 1:
        return project_tree[0]
    return None


def tree_path(node: TreeNode) -> str:
    """Returns the path of the tree node.

    Args:
        node (TreeNode): The tree node.

    Returns:
        (str): See description.
    """
    path_list = []
    while node:
        path_list.append(node.name)
        node = node.parent
    return '/'.join(reversed(path_list))


def build_tree_path(pe_tree_root, uuid):
    """Finds the node by uuid and returns its tree path.

    Args:
        pe_tree_root (TreeNode): Root of project explorer tree containing the item to build a path for
        uuid (str): UUID of the tree item to build a path for.

    Returns:
        (str): Path to the item.
    """
    if not pe_tree_root or not uuid:
        return ''
    path = _build_tree_path_recursive(pe_tree_root, uuid, pe_tree_root.name)
    return path


def _build_tree_path_recursive(pe_tree_root, uuid, path):
    """Recursive method for building tree item paths.

    Don't call this. Call build_tree_path instead.

    Args:
        pe_tree_root (TreeNode): Root of project explorer tree containing the item to build a path for
        uuid (str): UUID of the tree item to build a path for.
        path (str): Parent path to this level of the tree.

    Returns:
        (str): Path to the item.
    """
    for child in pe_tree_root.children:
        my_path = f'{path}/{child.name}'
        if child.uuid == uuid:
            return my_path
        found_child_path = _build_tree_path_recursive(child, uuid, my_path)
        if found_child_path:
            return found_child_path
    return ''


def item_from_path(tree, path):
    """Returns the item in the tree at path, or None if not found.

    Args:
        tree (TreeNode): Node of a tree.
        path (str): e.g. 'Project/Mesh Data/BC/Z'

    Returns:
        See description.
    """
    if not path or not tree:
        return None

    path_list = path.strip('"\'').replace('\\', '/').split('/')
    if tree.name != path_list[0]:
        return None

    item = tree
    for word in path_list[1:]:
        found = False
        for child in item.children:
            if child.name == word:
                item = child
                found = True
                break
        if not found:
            item = None
            break

    return item


def descendants_of_type(
    tree_root,
    tree_type=None,
    xms_types: Collection[str] | None = None,
    unique_name=None,
    model_name=None,
    coverage_type=None,
    allow_pointers=False,
    recurse=True,
    only_first=False
):
    """Get all child items of specified type, recursively.

    Args:
        tree_root (TreeNode): Root of the tree to start searching
        tree_type (type): C++ object type of the descendants to gather
        xms_types (Collection[str]): Collection (e.g. list, set) of str to compare to TreeNode.item_typename.
        unique_name (str): XML unique name of the descendant items to gather
        model_name (str): XML model name of the descendant items to gather
        coverage_type (str): XML coverage type name of descendant coverages to
            gather
        allow_pointers (bool): If True, linked pointer items will be gathered
        recurse (bool): If False, only direct descendants of
        only_first (bool): If True, only the first descendant of the specified type is returned.

    Returns:
        Union[TreeNode, list, None]: Children of the root that are of the specified type. If only_first=True, return
        value is a single TreeNode or None.
    """
    children = []
    if not tree_root:
        return None if only_first else children

    for child in tree_root.children:
        if not child.is_ptr_item or allow_pointers:
            match = True
            if unique_name:  # Make sure unique name matches if specified
                match = child.unique_name == unique_name
            if match and model_name:  # Make sure model name matches if specified
                match = child.model_name == model_name
            if match and coverage_type:  # Make sure coverage type matches if specified
                match = child.coverage_type == coverage_type
            if match and tree_type:  # Make sure C++ tree item object type matches if specified
                match = type(child.data) == tree_type
            if match and xms_types:  # Make XMS tree type enum in list if specified
                match = child.item_typename in xms_types

            if match:
                if only_first:
                    return child  # Done if we are only looking for a single child
                else:
                    children.append(child)

        if recurse:
            recurse_result = descendants_of_type(
                child,
                tree_type=tree_type,
                xms_types=xms_types,
                unique_name=unique_name,
                model_name=model_name,
                coverage_type=coverage_type,
                allow_pointers=allow_pointers,
                recurse=recurse,
                only_first=only_first
            )
            if recurse_result:
                if only_first:  # Done if one of our children was of the specified type
                    return recurse_result
                else:  # Continue building list of descendants with specified type
                    children.extend(recurse_result)

    return None if only_first else children


def first_descendant_with_name(tree, name):
    """Returns the first descendant found with specified name.

    Args:
        tree (TreeNode): Node of a tree.
        name (str): Name of the child tree item to search for. If multiple descendants with the same name, most direct
            descendant is returned.

    Returns:
        See description.
    """
    for child in tree.children:
        if child.name == name:
            return child
        child = first_descendant_with_name(child, name)
        if child is not None:
            return child
    return None
