"""The Arc properties tab in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO
import os
import webbrowser

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QSortFilterProxyModel
from PySide2.QtWidgets import QHBoxLayout, QPushButton

# 3. Aquaveo modules
from xms.guipy.widgets.table_with_tool_bar import TableWithToolBar

# 4. Local modules
from xms.bridge.gui.struct_dlg.preview_html import PreviewHtml
import xms.bridge.structure_util as su


class ArcPropsBase:
    """The arc properties tab in the structure dialog."""
    def __init__(self, parent, widgets):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            widgets (:obj:`dict`): Dictionary of widgets
        """
        self._parent = parent
        self._widgets = widgets
        self._table_def = {}
        self._df = None
        self._table_widget = None
        self._filter_model = None
        self._table_def = self._table_definition()
        self._col_names = [col.header for col in self._table_def.column_types]

    def _table_definition(self):
        raise RuntimeError('This method must be overridden in a derived class.')

    def _get_default_arc_data(self, arc_ids):
        """Get the default arc data."""
        raise RuntimeError('This method must be overridden in a derived class.')

    def _get_arc_property_csv_str(self):
        """Get the arc property CSV string."""
        raise RuntimeError('This method must be overridden in a derived class.')

    def _get_filter_model(self):
        return QSortFilterProxyModel(self._parent)

    def add_widgets(self, vlayout=None):
        """Setup the UI."""
        self._get_arc_data()
        self._table_widget = TableWithToolBar(parent=self._parent)
        self._widgets['arc_prop_table'] = self._table_widget
        self._table_widget.setup(self._table_def, self._df)
        self._table_widget.ui.tool_bar.hide()
        tab_view = self._table_widget.ui.table
        self._filter_model = self._get_filter_model()
        self._filter_model.setSourceModel(tab_view.model())
        tab_view.setModel(self._filter_model)
        vlayout.addWidget(self._table_widget)
        self._widgets['arc_preview_layout'] = QHBoxLayout()
        vlayout.addLayout(self._widgets['arc_preview_layout'])
        self._widgets['arc_prop_btn_preview'] = QPushButton('Preview coverage arcs...')
        self._widgets['arc_prop_btn_preview'].clicked.connect(self._on_preview_coverage_arcs)
        self._widgets['arc_preview_layout'].addWidget(self._widgets['arc_prop_btn_preview'])
        # show preview in browser
        self._widgets['arc_prop_btn_preview_browser'] = QPushButton('Preview coverage arcs (compatibilty mode)...')
        self._widgets['arc_prop_btn_preview_browser'].clicked.connect(self._on_preview_coverage_arcs_browser)
        self._widgets['arc_preview_layout'].addWidget(self._widgets['arc_prop_btn_preview_browser'])
        self._widgets['arc_preview_layout'].addStretch()
        vlayout.addStretch()

    def _on_preview_coverage_arcs_browser(self):
        """Preview the coverage arcs."""
        self._on_preview(True)

    def _on_preview_coverage_arcs(self):
        """Preview the coverage arcs."""
        self._on_preview()

    def _on_preview(self, browser=False):
        """Preview the coverage arcs."""
        self._parent.close_map_viewer_proc()
        cov = self._parent.arc_data['coverage']
        arc_data = self._parent.arc_data
        if len(cov.arcs) > 0:
            dlg_data = {
                'ugrid': None,
                'projection': arc_data['projection'],
                'url': os.path.join(arc_data['tmp_dir'], 'map.html'),
                'arc_data': arc_data,
                'tool_arc_data': None,
                'arc_df': self._df,
                'struct_type': self._parent.data.data_dict['structure_type']
            }
            PreviewHtml(dlg_data)
            if browser:
                webbrowser.open(dlg_data['url'])
            else:
                self._parent.run_map_viewer_proc(dlg_data['url'])

    def _get_arc_data(self):
        """Get the arc data from the coverage."""
        col_names = self._col_names
        cov = self._parent.arc_data['coverage']
        arc_ids = [1] if not cov else [arc.id for arc in cov.arcs]
        default_vals = self._get_default_arc_data(arc_ids)
        self._df = pd.DataFrame(default_vals)
        comp = self._parent.arc_data['component']
        csv_str = self._get_arc_property_csv_str()
        if csv_str:
            df = pd.read_csv(StringIO(csv_str))
            rename_cols = {f'col_{i}': col for i, col in enumerate(col_names)}
            df.rename(columns=rename_cols, inplace=True)
            df = su.update_data_frame_arc_ids_from_component(df, comp)
            df.set_index('Arc ID', inplace=True)
            self._df.set_index('Arc ID', inplace=True)
            self._df.update(df)
            self._df.reset_index(inplace=True)
        self._df.index += 1

    def get_arc_data_csv(self):
        """Get the arc data as a CSV string."""
        df2 = self._df.copy()
        cols = df2.columns.tolist()
        rename_cols = {col: f'col_{i}' for i, col in enumerate(cols)}
        df2.rename(columns=rename_cols, inplace=True)
        return df2.to_csv(index=False)
