"""A dialog for displaying feature map lines."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import argparse
import os.path
import sys

# 2. Third party modules
from PySide2.QtCore import QUrl
from PySide2.QtWebEngineWidgets import QWebEngineView
from PySide2.QtWidgets import QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists, XmsDlg

# 4. Local modules


class MapViewerDialog(XmsDlg):
    """A dialog for displaying feature map lines."""
    def __init__(self, parent, url):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            url (:obj:`str`): the html file to display in the dialog
        """
        super().__init__(parent, 'xms.bridge.gui.map_view_dialog')
        self._widgets = {}
        self._url = url

        # Add widgets to dialog
        self._setup_ui()
        self._widgets['plot_view'].setUrl(QUrl.fromLocalFile(url))
        self._widgets['plot_view'].show()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        self.setWindowTitle('Preview Bridge Mesh')
        self.setMinimumSize(300, 300)
        self._widgets['vert_layout'] = QVBoxLayout()
        self._widgets['plot_view'] = QWebEngineView(self)
        self._widgets['vert_layout'].addWidget(self._widgets['plot_view'])
        self.setLayout(self._widgets['vert_layout'])


def map_viewer_file_fullpath():
    """Returns the full path to the file."""
    return os.path.abspath(__file__)


class _MainRunner:
    """Main runner for the MapViewerDialog."""
    def __init__(self, args):
        """Initialize the main runner."""
        self.sys_args = args
        self.args = None
        self.app = None
        self.dlg = None

    def _parse_args(self):
        """Parse command-line arguments."""
        parser = argparse.ArgumentParser(description="Map Viewer Dialog")
        parser.add_argument("--parent_id", type=str, required=True, help="The window ID of the parent process.")
        parser.add_argument("--url", type=str, required=True, help="The URL of the bridge process.")
        self.args = parser.parse_args(self.sys_args)

    def run(self):
        """Run the main function."""
        self._parse_args()

        self.app = ensure_qapplication_exists()
        parent_win_id = int(self.args.parent_id)
        parent_window = QWidget.find(parent_win_id)

        self.dlg = MapViewerDialog(parent_window, self.args.url)
        self.dlg.show()
        self._sys_exit()

    def _sys_exit(self):
        sys.exit(self.app.exec_())


if __name__ == "__main__":  # pragma: no cover
    r = _MainRunner(sys.argv[1:])
    r.run()
