"""Export a 3D Bridge deck geometry for the SMS Bridge Scour coverage."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import argparse
from io import StringIO

# 2. Third party modules

# 3. Aquaveo modules
import pandas as pd

# 4. Local modules
from xms.bridge.bridge_io import BridgeIo
from xms.bridge.grid_builder import GridBuilder
from xms.bridge.structure_component import StructureComponent

BRIDGE_PROFILE_ERROR = -2
BRIDGE_GENERAL_ERROR = -1
BRIDGE_SUCCESS = 0


def export_for_bridge_scour_coverage(arg_str=None):
    """Export bridge deck geometry for the SMS Bridge Scour coverage.

    Args:
        arg_str (:obj:`list`): command strings

    Returns:
        (:obj:`int`): return code
    """
    try:
        desc = "3D bridge profile exporter for SMS Bridge Scour coverage."
        arguments = argparse.ArgumentParser(description=desc)
        arguments.add_argument(dest='command', type=str, help='script to run')
        arguments.add_argument(dest='output_file', type=str, help='Path to file where profiles will be written')
        arguments.add_argument(dest='arc_file', type=str, help='Upstream/downstream arc definition file')
        arguments.add_argument(dest='bridge_mainfile', type=str, help='Bridge component main file')
        parsed_args = arguments.parse_args(arg_str)

        # If this is a 3D Structure then just export the profiles.
        main_file = parsed_args.bridge_mainfile.strip('"')
        out_file = parsed_args.output_file.strip('"')
        if main_file.endswith('structure.nc'):
            return _process_3d_structure(main_file, out_file)

        # Instantiate the bridge component
        bridge_io = BridgeIo()
        bridge_comp = bridge_io.read(main_file)

        # Make sure profile curves are defined for top, downstream, and upstream.
        if bridge_comp.df_top.shape[0] < 2 or bridge_comp.df_upstream.shape[0] < 2 \
                or bridge_comp.df_downstream.shape[0] < 2:
            return BRIDGE_PROFILE_ERROR  # Can't continue, give unique return code.

        # Read the upstream and downstream arc definitions.
        upstream_pts = []
        downstream_pts = []
        read_upstream = False
        with open(parsed_args.arc_file.strip('"'), 'r') as in_file:
            for line in in_file:
                line = line.strip()
                if line == 'UPSTREAM':
                    continue
                elif line == 'DOWNSTREAM':
                    read_upstream = True
                elif not read_upstream:
                    line_data = line.split()
                    upstream_pts.append((float(line_data[0]), float(line_data[1])))
                else:
                    line_data = line.split()
                    downstream_pts.append((float(line_data[0]), float(line_data[1])))
        grid_builder = GridBuilder()
        grid_builder.write_parametric_profiles_for_bridge_scour(bridge_comp, upstream_pts, downstream_pts, out_file)
        return BRIDGE_SUCCESS
    except Exception:  # Give SMS non-zero exit code
        return BRIDGE_GENERAL_ERROR


def _process_3d_structure(main_file, out_file):
    """Export profiles for the 3d structure.

    Args:
        main_file (str): The main file for the 3D structure.
        out_file (str): The output file for the profiles.

    Returns
        (int): return code
    """
    struct = StructureComponent(main_file)
    comp_id = _structure_centerline_comp_id(struct)
    # Make sure profile curves are defined for top and upstream.
    top = struct.data.curves['top_profile']
    up = struct.data.curves['upstream_profile']
    if len(top) < 2 or len(up) < 2:
        return BRIDGE_PROFILE_ERROR

    grid_builder = GridBuilder()
    top = grid_builder.get_parametric_line(top)
    up = grid_builder.get_parametric_line(up)
    parametric_vals = list({p[0] for p in top} | {p[0] for p in up})
    parametric_vals.sort()
    grid_builder.add_parametric_lengths_to_line(parametric_vals, top)
    grid_builder.add_parametric_lengths_to_line(parametric_vals, up)
    with open(out_file, 'w') as file:
        if comp_id != -1:
            file.write(f'COMP_ID {comp_id}\n')
        file.write('TOP_PROFILE\n')
        for top_val in top:
            file.write(f'{top_val[0]} {top_val[1]}\n')
        file.write('UPSTREAM_PROFILE\n')
        for up_val in up:
            file.write(f'{up_val[0]} {up_val[1]}\n')
    return BRIDGE_SUCCESS


def _structure_centerline_comp_id(struct_comp):
    """Get the component id of the centerline of the structure.

    Args:
        struct_comp (StructureComponent): The structure component.

    Returns:
        (int): The component id of the centerline.
    """
    comp_id = -1
    dd = struct_comp.data.data_dict
    specify = dd.get('specify_arc_properties', 0)
    arc_prop = dd.get('arc_properties', '')
    if dd['structure_type'] == 'Culvert':
        specify = dd.get('specify_culvert_arc_properties', 0)
        arc_prop = dd.get('culvert_arc_properties', '')
    if specify and arc_prop:
        df = pd.read_csv(StringIO(arc_prop))
        col_comp_id, col_type = df.columns[0], df.columns[1]
        comp_ids = df[col_comp_id].tolist()
        arc_types = df[col_type].tolist()
        for i, at in enumerate(arc_types):
            if at == 'Bridge' or at == 'Embankment':
                comp_id = comp_ids[i]
                break
    return comp_id
